/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Jul 2014)
**=======================================================================*/

#ifndef _SIMPLE_TILE_CACHE_H_
#define _SIMPLE_TILE_CACHE_H_

#include <Inventor/STL/list>
#include <Inventor/devices/SoCpuBufferObject.h>

// Simple Tile Cache management:
// This cache implementation is based on a list of entry whose size will not grow above m_maxCacheSize
// - each entry contains a pair of tileID<->SoCpuBufferObject
// - it is not threadsafe (ie thread safety has to be managed at user level)
class SimpleTileCache
{
public:
  // type of data in the cache (we store both a ref on the buffer + its minIJK
  typedef std::pair<SoRef<SoCpuBufferObject>, SbVec3i32 > TileCacheData;
  // type of entry in the cache (for a tileID <-> TileCacheData)
  typedef std::pair<SoLDMTileID, TileCacheData> TileCacheEntry;

  // constructor
  SimpleTileCache(size_t maxCacheSize) :m_maxCacheSize(maxCacheSize)
  {}

  // destructor
  ~SimpleTileCache()
  {}

  // return false if not found
  // return true if found and move the entry at the begining of the cache
  bool getEntry(const SoLDMTileID& needTileID, SoCpuBufferObject*& tileBuffer, SbVec3i32& tileMin)
  {
    TileCacheList::iterator it = m_tileCache.begin();
    while (it != m_tileCache.end())
    {
      if ( it->first == needTileID)
      {
        tileBuffer = it->second.first.ptr();
        tileMin = it->second.second;
        std::swap(*it, *(m_tileCache.begin()));
        return true;
      }
      ++it;
    }
    return false;
  }

  // add the given entry at the begining of the cache and remove last entry if maxCacheSize is reach
  void addEntry(const TileCacheEntry& entry)
  {
    m_tileCache.push_front(entry);
    if (m_tileCache.size()>m_maxCacheSize)
      m_tileCache.pop_back();
  }

  // return the current cache size
  size_t getCacheSize() const
  {
    return m_maxCacheSize;
  }

  // needed by SbThreadStorage to release resources on exit
  void unref()
  {
    delete this;
  }
private:
  // cache storage
  typedef std::list<TileCacheEntry> TileCacheList;

  // cache entry list
  TileCacheList m_tileCache;
  size_t m_maxCacheSize;
};

#endif //_SIMPLE_TILE_CACHE_H_
