/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : mmh (Dec 2008)
**=======================================================================*/

// Test:
// MeshViz rendering from a VolumeViz data set
//
// This example shows how to do MeshViz XLM rendering, including
// isosurfaces, using a regular volume mesh and a VolumeViz LDM data 
// set as the scalar data set.  As a result it also shows how to
// implement a custom scalar data source using the MiScalardSetIjk
// interface and data that is stored in a completely different
// format (VolumeViz LDM tiled data).
//
// VolumeViz is an effective, and very efficient, way to render a
// regular or rectilinear scalar data set using slices and/or direct
// volume rendering.  However for some data sets, particularly medical
// and NDT, it is important to be able to use isosurfaces and other
// mesh rendering techniques.  VolumeViz can render an isosurface using
// GPU shader programs and interactively change the iso-value.  But this 
// isosurface capability has limitations.  It is not possible, for
// example to: extract the isosurface as geometry, to color the surface
// with another data set, etc.  MeshViz XLM solves this problem.
//
// With previous versions of MeshViz it was not possible to use a
// VolumeViz volume as a data set without making a complete copy of the
// data and converting to "float" data type.  This was impractical
// because of the time required and/or the memory required.  Now with
// MeshViz XLM we are able to access almost any data source without
// copying the data.
//
// This is not a demo.  To keep it really simple and illustrate the
// principles, there is no user interface.  It would be easy to add one
// using events, DialogViz or your favorite UI toolkit.  By default the 
// example uses the tiny data set "syn_64" and renders an isosurface
// (MoMeshIsosurface).  It has been tested with a few other data sets, 
// for example "Engine", which is a small volume for VolumeViz, but a 
// non-trivial mesh containing 7 million cells.
//
// The main program is actually a simple MeshViz XLM program.  The
// interesting code is in the class VolVizScalarSet, which implements
// the MiScalardSetIjk interface using the LDM Data Access API.
//
// Notes:
//
// - Performance of a debug build can be much slower than the release
//   build. We believe this is due to error checking in the STL.
//
// - Color mapping: You may notice that the coloring of the VolumeViz
//   and MeshViz slices is not exactly the same.  This is because, by
//   default, for an integer valued data set VolumeViz maps the full
//   range of the data type into the color map.  In our MeshViz code
//   we query the actual range of data values in the volume.  So, for
//   example in the SYN_64 data set, VolumeViz maps 0..255 into the
//   color map, while our MeshViz code maps 0..192 into the color map.
//   If necessary we could fix this by using an SoDataRange node in the
//   VolumeViz code or setting the MeshViz color map node's maxVal field
//   to the data type max instead of the actual max.
//
// - Lighting: Like other geometry in Open Inventor, MeshViz rendering
//   objects are only lit on one side. Particularly for slice objects,
//   you will generally want to add an SoShapeHints to enable two sided
//   lighting (so you can see both sides of the slice).
//
// - Lighting: If you don't use two-sided lighting, note inconsistency:
//   The lit side of MoMeshLogicalSlice is the opposite of the other
//   slice objects (InterpolatedSlice, PlaneSlice, etc).
//
// - OIV 7.2: On Windows, you must add "#define ADD_VERSIONING_TO_DLL 1"
//   before the MeshViz header files to link in the correct MeshViz
//   libraries (Ebug 2554).
//
// - OIV 7.2: SoWin::finish (or equivalent) may cause a crash if
//   VolumeViz and MeshViz are both initialized (Ebug 2555).  It is
//   NOT mandatory to call this method (all memory will be reclaimed 
//   when the process exits anyway).
//
// - OIV 7.2: In a debug build, VolumeViz may abort with an assertion
//   failure if you lock a tile of LDM data (using getData) before the
//   first scene graph traversal (Ebug 2556).  You can work around this
//   by not putting the SoVolumeData in the scene graph, delaying the
//   attachment to MeshViz, not locking tiles in VolVizScalarSet, etc.
//
// Future work:
//
// - Many medical, NDT, etc volumes contain a lot of "empty space".
//   These voxels could eliminated from the isosurface computation
//   using the MeshViz XLM "dead cells" mechanism.
//
// - It could be interesting to take into account the VolumeViz ROI
//   (Region of Interest).  Voxels outside the ROI could be excluded
//   using the MeshViz XLM "cell filter" mechanism.
//
// - For large volumes with a "region of interest" it would be more
//   efficient to construct the mesh only for that region.  This
//   could be done by adapting the VolVizScalarSet class to translate
//   from MeshViz cell locations to voxel locations inside the ROI.
//

// Core Inventor header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoTransform.h>

#include <Inventor/nodes/SoShapeHints.h>

// MeshViz header files
#define ADD_VERSIONING_TO_DLL 1 // Work-around for ebug #2554
#include <MeshVizXLM/mapping/MoMeshViz.h> 
#include <mesh/volumes/MbVolumeMeshRegular.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshInterpolatedLogicalSlice.h> 
#include <MeshVizXLM/mapping/nodes/MoMeshIsosurface.h>
#include <MeshVizXLM/mapping/nodes/MoMeshLogicalSlice.h> 
#include <MeshVizXLM/mapping/nodes/MoMeshPlaneSlice.h> 
#include <MeshVizXLM/mapping/nodes/MoPredefinedColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSetIjk.h> 

// VolumeViz header files
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <LDM/nodes/SoTransferFunction.h>

// Custom class that allows a VolumeViz volume to be a data set
#include "VolVizScalarSet.h"

// Utility function to render bounding box
#include "MakeBBox.h"

#include <iostream>
#include <fstream>

#include <Inventor/helpers/SbFileHelper.h>

using namespace std;

// Set the data file to be opened
SbString FileName =  SbFileHelper::expandString("$OIVHOME/examples/data/MeshVizXLM/SYN_64.VOL");
//SbString FileName =  SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/ENGINE.VOL");

const char *FILENAME = FileName.toLatin1();

// Set this to true to enable VolumeViz rendering
// (just to confirm that MeshViz is accessing data correctly)
const bool DO_VOLVIZ_RENDERING = false;

////////////////////////////////////////////////////////////////////////
// main function
int main(int argc, char **argv)
{
  // Confirm data file exists before going any further

  ifstream file( FILENAME );
  if (file.is_open())
    file.close();
  else {
    cerr << "ERROR: Unable to open file '" << FILENAME << "'\n";
    cerr << "Press any key to continue...";
    cin >> argc;
    return -1;
  }

  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Create the root of the scene graph
  SoSeparator *root = new SoSeparator();
  root->ref();

  //====================================================================
  // Begin VolumeViz Section
  //====================================================================

  // Initialize VolumeViz extension
  SoVolumeRendering::init();

  // Data node: Volume data set
  SoRef<SoVolumeData> pVolData = new SoVolumeData();
  pVolData->fileName = FILENAME;

  if (DO_VOLVIZ_RENDERING)
  {
    // Property node: Colormap (aka transfer function)
    //
    // Use a predefined colormap to keep it simple.
    // If you're going to do actual volume rendering (instead of just
    // slices) uncomment the minValue line to remove "noise" voxels.
    SoTransferFunction* pTransFunc = new SoTransferFunction();
    pTransFunc->predefColorMap = SoTransferFunction::GLOW;
    //  pTransFunc->minValue = 39;

    // Rendering node: Orthoslice
    //
    // Draw a slice in the middle of the volume, on the Z axis.
    // This is the same slice we'll be rendering with MeshViz, so if you
    // turn them both on at the same time they will overlap.
    // Alternatively you could uncomment the two lines below and render
    // an X axis slice, perpendicular to the MeshViz slice.
    SoOrthoSlice* pOrthoSlice = new SoOrthoSlice();
    pOrthoSlice->axis = SoOrthoSlice::Z;
    pOrthoSlice->sliceNumber = (pVolData->data.getSize())[2] / 2;
    //  pOrthoSlice->axis = SoOrthoSlice::X;
    //  pOrthoSlice->sliceNumber = (pVolData->getDimension())[0]/2;
    pOrthoSlice->alphaUse = SoSlice::ALPHA_OPAQUE;

    // Property node: Material (global alpha value)
    //
    // For actual volume rendering we'll reduce the alpha (increase the
    // transparency) globally to improve the image.
    //SoMaterial *pVolMat = new SoMaterial();
    //pVolMat->diffuseColor = SbColor(1,1,1);
    //pVolMat->transparency = 0.7f;

    // Rendering node: Volume rendering
    //SoVolumeRender* pVolRender = new SoVolumeRender();
    //pVolRender->numSlices = 200;

    // Assemble VolumeViz scene graph
    // (comment/uncomment lines as desired)
    SoSeparator *pVolVizSep = new SoSeparator();
    pVolVizSep->addChild(pVolData.ptr());
    pVolVizSep->addChild(pTransFunc);
    pVolVizSep->addChild(pOrthoSlice);
    //    pVolVizSep->addChild( pVolMat );
    //    pVolVizSep->addChild( pVolRender );

    root->addChild(pVolVizSep);
  }

  //--------------------------------------------------------------------

  //====================================================================
  // Begin MeshViz Section
  //====================================================================
  MoMeshViz::init();

  // Get dimensions and extent of the associated VolumeViz volume.
  // Dimensions and geometry size of the mesh are defined by this.
  const SbVec3i32 volDim = pVolData->data.getSize();
  const SbBox3f   volExt = pVolData->extent.getValue();
  const SbVec3f   volMin = volExt.getMin();
  const SbVec3f   volMax = volExt.getMax();

  // Construct a regular volume mesh
  // - The geometry is implicit from the mesh bounding box.
  // - The topology is implicit from the mesh type.
  //
  // The obvious approach is to consider a volume data set as a regular
  // volume mesh with per_cell data values.  In other words each voxel
  // is represented as a cell in the mesh and the number of cells on each
  // axis is the same as the number of voxels.  This is a reasonable and 
  // correct interpretation. But one of our main goals is to extract 
  // isosurfaces from the volume data and isosurface extraction only 
  // works with per_node data values.
  //
  // Therefore we will actually represent the volume data as a mesh with
  // its nodes (vertices) at the voxel centers.  In this case the number
  // of cells on each axis is N-1 and the geometry extent of the mesh is
  // 1/2 voxel less than the volume in every direction.

  cout << "Begin mesh creation... ";
  MbVolumeMeshRegular<> *pVolMesh;
  SbVec3i32 meshDim = volDim - SbVec3i32(1,1,1);
  SbVec3f volRange  = volMax - volMin;
  SbVec3f voxelSize = SbVec3f(volRange[0]/volDim[0], volRange[1]/volDim[1], volRange[2]/volDim[2] );
  SbVec3f meshMin   = volMin + voxelSize/2;
  SbVec3f meshMax   = volMax - voxelSize/2;
  pVolMesh = new MbVolumeMeshRegular<>( meshDim[0],meshDim[1],meshDim[2],
    meshMin[0],meshMin[1],meshMin[2], meshMax[0],meshMax[1],meshMax[2] );

  // Data node: Regular volume mesh (topology and geometry)
  MoMesh* pMeshNode = new MoMesh();
  pMeshNode->setMesh( pVolMesh );
  cout << "End mesh creation." << endl;

  // Property node: Color mapping.
  // Set the colormap to the same predefined map we used with VolumeViz.
  // Important: Must set the range of scalar values that will be mapped
  // into the colormap.  By default these values are 0 and 1.
  //
  // We didn't have to do that in the VolumeViz code because the data
  // happened to be unsigned byte and VolumeViz automatically maps the 
  // full range of values for the data type.  If the volume data was
  // float values we would have had to specify a range for VolumeViz,
  // although we would have done that with the SoDataRange node, not
  // the transfer function.
  //
  // Note that we are mapping the range of actual data values into the
  // colormap.  For some integer valued volume data sets this does not
  // produce the same default color mapping as VolumeViz.  This is
  // because, by default, VolumeViz maps the full range of the DATA
  // TYPE into the color map.  For example, for the SYN_64 data set
  // VolumeViz maps 0..255, but our MeshViz code maps 0..192, the actual
  // range of data values.  If necessary we could fix this by using an
  // SoDataRange in our VolumeViz code or by setting maxVal to the max
  // of the data type in our MeshViz code.
  double minVal, maxVal;
  pVolData->getMinMax( minVal, maxVal );
  MoPredefinedColorMapping* pMapping = new MoPredefinedColorMapping();
  pMapping->predefColorMap = MoPredefinedColorMapping::GLOW;
  pMapping->minValue = (float)minVal;
  pMapping->maxValue = (float)maxVal;

  // *****
  // Create scalar data set that encapsulates VolumeViz voxel data.
  //
  // This is a custom class derived from MiScalardSetIjk and MbScalardSet.
  // *****
  VolVizScalarSet *pVolVizScalarSet = new VolVizScalarSet( pVolData.ptr() );

  // Property node: Scalar data set
  MoScalarSetIjk *pScalarSet = new MoScalarSetIjk();
  pScalarSet->setScalarSet( pVolVizScalarSet );

  // Rendering node: Logical slice
  //
  // Position slice in the middle of the volume on the K (volume Z) axis.
  //MoMeshLogicalSlice *pLSlice = new MoMeshLogicalSlice();
  //pLSlice->sliceAxis  = MiMesh::DIMENSION_K;
  //pLSlice->sliceIndex = volDim[2]/2;

  // Rendering node: Interpolated logical slice
  //
  // Position slice in the middle of the volume on the K (volume Z) axis.
  //MoMeshInterpolatedLogicalSlice *pISlice = new MoMeshInterpolatedLogicalSlice();
  //pISlice->sliceAxis  = MiMesh::DIMENSION_K;
  //pISlice->sliceValue = (float)volDim[2]/2;

  // Rendering node: Plane slice
  //
  // Position slice in the middle of the volume on the K (volume Z) axis.
  //SbVec3f normal(0,0,1);
  //float zval = volMin[2] + (volMax[2]-volMin[2])/2;
  //SbVec3f point(0,0,zval);
  //MoMeshPlaneSlice *pPSlice = new MoMeshPlaneSlice();
  //pPSlice->plane = SbPlane( normal, point );
  
  // Rendering node: Isosurface
  //
  // For this simple example we'll just use a value we know is interesting.
  MoMeshIsosurface *pIsosurf = new MoMeshIsosurface();
  pIsosurf->isovalue = 95;
  pIsosurf->isoScalarSetId = 0;

  // Property node: Shape hints
  //
  // Like any Open Inventor geometry, MeshViz rendering objects are only
  // lit on one side by default.  For slice objects in particular it is
  // convenient to be able to see both sides.  Specifying the vertex
  // ordering tells Open Inventor to enable two-sided lighting.
  SoShapeHints *pShapeHints = new SoShapeHints();
  pShapeHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  root->addChild( pShapeHints );

  // Assemble the MeshViz scene graph
  //
  // Note we don't really want to render all these objects at the same
  // time.  We just created them to allow you to try the different
  // rendering objects by commenting and uncommenting them here.
  SoSeparator *pMeshVizSep = new SoSeparator();
  pMeshVizSep->addChild( pMeshNode );
  pMeshVizSep->addChild( pMapping );
  pMeshVizSep->addChild( pScalarSet );
//  pMeshVizSep->addChild( pLSlice );
//  pMeshVizSep->addChild( pISlice );
//  pMeshVizSep->addChild( pPSlice );
  pMeshVizSep->addChild( pIsosurf );

  root->addChild( pMeshVizSep );

  //--------------------------------------------------------------------

  // Add visual representation of bounding box to scene
  root->addChild( makeBBox( pVolData->extent.getValue()) );

  //--------------------------------------------------------------------

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(root);
  myViewer->setTitle("Volume rendering");
  myViewer->setBackgroundColor( SbColor(0,.2f,.2f) );
  myViewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  myViewer->show();

  // Execute
  SoXt::show(myWindow);
  SoXt::mainLoop();

  // Cleanup viewer and scene graph
  delete myViewer;
  root->unref();

  // Cleanup MeshViz data objects
  delete pVolMesh;
  delete pVolVizScalarSet;
  pVolData = NULL;

  // Final cleanup
  MoMeshViz::finish();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


