// VolVizScalarSet implementation
//
// This class implements the MiScalardSetIjk interface and allows a
// VolumeViz LDM data set to serve as a data source for MeshVizXLM.
//
// Original: MHeck, VSG

#include <Inventor/sensors/SoNodeSensor.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/SoLDMTopoOctree.h> 

#include <iostream>
using namespace std;

#ifdef WIN32
#pragma warning( push )
#pragma warning(disable:4250)  // Disable warning about inheritance
#endif // WIN32

#include "VolVizScalarSet.h"


////////////////////////////////////////////////////////////////////////
//
// Constructor
VolVizScalarSet::VolVizScalarSet( const SoVolumeData *pVolData )
: m_pTopoOctree(NULL), m_tileCache(NULL)
{ 
  // The default data set name will be VolVizData
  setName( "VolVizData" );

  // Store VolumeViz data set (and ref to make sure it stays around)
  m_pVolData = const_cast<SoVolumeData*>(pVolData);

  // setup a sensor to catch any changes that could impact cached value/data
  m_volDataSensor = new SoNodeSensor();
  m_volDataSensor->attach(m_pVolData.ptr());
  m_volDataSensor->setData(this);
  m_volDataSensor->setFunction(volDataChangedCB);

  // init members
  volDataChanged();
}

////////////////////////////////////////////////////////////////////////
//
// Destructor
VolVizScalarSet::~VolVizScalarSet()
{
  delete m_volDataSensor;
  delete m_pTopoOctree;
  delete m_tileCache;
}

void
VolVizScalarSet::volDataChangedCB(void *data, SoSensor* /*sensor*/)
{
  VolVizScalarSet* volS = static_cast<VolVizScalarSet*>(data);
  volS->volDataChanged();
}

void
VolVizScalarSet::volDataChanged()
{
  SbThreadAutoLock autoLock(&m_cacheAccessLock);

  // Get volume characteristics
  SbVec3i32 volDim  = m_pVolData->data.getSize();
  m_tileDimension  = m_pVolData->getTileDimension();
  m_dataType = m_pVolData->getDataType();

  // Create and initialize the topoOctree.
  // This object tells us which tile we need for a given voxel.
  delete m_pTopoOctree;
  m_pTopoOctree = new SoLDMTopoOctree();
  m_pTopoOctree->init( volDim, m_tileDimension[0] );

  // setup cache size to be able to store 256x256x256 data
  // we do not have the per thread block size info from MeshViz but it is reasonnable size in most case
  SbVec3i32 cacheDim(128,128,128);
  cacheDim/=m_tileDimension;
  size_t maxCacheSize;
  maxCacheSize = cacheDim[0]*cacheDim[1]*cacheDim[2];
  maxCacheSize = (maxCacheSize>0)?maxCacheSize:1;

  //clean the cache
  delete m_tileCache;
  m_tileCache = new SimpleTileCache(maxCacheSize);

#if defined(_DEBUG)
  cerr << "VolVizScalarSet initialized" << endl;
  cerr << "  dim: " << volDim[0] << " " << volDim[1] << " " << volDim[2] << endl;
#endif
}

////////////////////////////////////////////////////////////////////////
//
// Return data value at specified location.
//
// MeshViz will call this method when it needs the value for a
// particular cell (PER_CELL) or node (PER_NODE).  In both cases we
// interpret i,j,k as a voxel location.
//
// This method will be called frequently and can dramatically affect
// the performance of mesh extraction and rendering.
//
// getData returns a pointer to the actual data for the tile that
// contains the specified voxel location (no copying of data) and
// this tile is now locked.  On subsequent requests, first check if
// we can satisfy the request using the currently locked tile.  If
// necessary, release that tile and load the one we need.
//
// Always return the voxel value as a double. Therefore we generally
// to convert the VolumeViz data value.

double
VolVizScalarSet::_get(size_t i, size_t j, size_t k)
{
  if (m_pVolData.ptr()==NULL)
    return 0;
  
  double value = 0;

  // Requested voxel position
  SbVec3i32 voxelPos((int32_t)i, (int32_t)j, (int32_t)k);

  // Get the ID of the tile we need for the requested voxel.
  SoLDMTileID needTileID = m_pTopoOctree->getTileID( voxelPos, 0 );
  if ( needTileID==-1)
    return 0;


  SoCpuBufferObject* curTileBufferObject = NULL;
  SbVec3i32 curTileMin;

  // 0. get or allocate tls cache for current thread
  SimpleTileCache* tlsTileCache = static_cast<SimpleTileCache*>(m_tileCacheTLS.getValue());
  if ( tlsTileCache==NULL )
  {
    // for now we use the same cache size for thread than in global cache
    // it means that we might keep numThread*cacheSize tile in memory at a time
    tlsTileCache = new SimpleTileCache(m_tileCache->getCacheSize());
    m_tileCacheTLS = tlsTileCache;
  }
  
  // 1. check in our TLS cache if the needed entry exist
  tlsTileCache->getEntry(needTileID, curTileBufferObject, curTileMin);

  // 2. check in global cache if the needed entry exist
  if (curTileBufferObject==NULL)
  {
    SbThreadAutoLock autoLock(&m_cacheAccessLock);
    m_tileCache->getEntry(needTileID,curTileBufferObject, curTileMin);
    if ( curTileBufferObject!=NULL )
    {
      // if the entry is in the global cache we just have to push it for latter reference in the thread cache
      SimpleTileCache::TileCacheEntry entry;
      entry.first         = needTileID;
      entry.second.first  = curTileBufferObject;
      entry.second.second = curTileMin;
      tlsTileCache->addEntry(entry);
    }
    else
    {
      // we have to get the data from LDM and store it in our cache
      const SoLDMDataAccess::DataInfo& dataInfo = m_pVolData->getLdmDataAccess().getData( 0, voxelPos );
      if (dataInfo.errorFlag != SoLDMDataAccess::CORRECT)
      {
        cerr << "Error in getData: " << dataInfo.errorFlag << ", voxel: " << i << " " << j << " " << k << endl;
        return 0;
      }
      SoRef<SoCpuBufferObject> LDMTileBufferObject = new SoCpuBufferObject(dataInfo.tileData, (size_t)dataInfo.bufferSize);

      // Copy the tile in our cache
      curTileBufferObject = new SoCpuBufferObject();
      curTileBufferObject->setSize((size_t)dataInfo.bufferSize);
      curTileBufferObject->memcpy(LDMTileBufferObject.ptr());

      // And Release the tile in LDM cache
      m_pVolData->getLdmDataAccess().releaseData( dataInfo.tileID);

      // init curTileMin
      curTileMin = m_pTopoOctree->getTilePos(needTileID).getMin();

      // create a cache entry and push it to the global cache
      SimpleTileCache::TileCacheEntry entry;
      entry.first         = needTileID;
      entry.second.first  = curTileBufferObject;
      entry.second.second = curTileMin;
      // push the entry in both global and thread cache
      m_tileCache->addEntry(entry);
      tlsTileCache->addEntry(entry);
    }
  }


  // Compute offset to voxel location in this tile
  // voxelPos is abs IJK position in volume.
  // Need position relative to tile.
  // offset in voxels
  size_t offset = (k - curTileMin[2])*(m_tileDimension[0] * m_tileDimension[1])
                + (j - curTileMin[1])* m_tileDimension[0]
                + (i - curTileMin[0]);
  unsigned int bytesPerVoxel = SbDataType(m_dataType).getSize();

  unsigned char *pValue = (unsigned char*)(curTileBufferObject->map(SoBufferObject::READ_ONLY));
  pValue += offset*bytesPerVoxel;

  // Convert the data value to "double"
  switch (m_dataType) {
    case SbDataType::UNSIGNED_BYTE:
      value = (double)(*pValue);
      break;
    case SbDataType::UNSIGNED_SHORT:
      value = (double)(*(unsigned short*)pValue);
      break;
    case SbDataType::UNSIGNED_INT32:
      value = (double)(*(unsigned int*)pValue);
      break;
    case SbDataType::SIGNED_BYTE:
      value = (double)(*(char *)pValue);
      break;
    case SbDataType::SIGNED_SHORT:
      value = (double)(*(short *)pValue);
      break;
    case SbDataType::SIGNED_INT32:
      value = (double)(*(int *)pValue);
      break;
    case SbDataType::FLOAT:
      value = (double)(*(float *)pValue);
      break;
    default:
      break;
  }
  curTileBufferObject->unmap();

  return value;
}


////////////////////////////////////////////////////////////////////////
//
// Return minimum value in data set.

double 
VolVizScalarSet::getMin() const
{
  if (m_pVolData.ptr())
  {
    double minVal, maxVal;
    m_pVolData->getMinMax( minVal, maxVal );
    return minVal;
  }
  else
    return 0;
}
  
////////////////////////////////////////////////////////////////////////
//
// Return maximum value in data set.

double 
VolVizScalarSet::getMax() const
{
  if (m_pVolData.ptr())
  {
    double minVal, maxVal;
    m_pVolData->getMinMax( minVal, maxVal );
    return maxVal;
  }
  else
    return 1;
}

#ifdef WIN32
#pragma warning( pop ) 
#endif // WIN32
