#ifndef _VOLVIZSCALARSET__H
#define _VOLVIZSCALARSET__H

//-----------------------------------------------------------------------------
/**
* @DTEXT VolumeViz scalar data set
* 
* @ingroup MeshVizXLM_Implement_Data
* 
* @DESCRIPTION
* This class implements the MiScalardSetIjk interface and allows a
* VolumeViz LDM data set to serve as a data source for MeshVizXLM.
*
* MeshVizXLM provides predefined implementation classes, e.g. MbVec3SetIjk,
* for vector data but not (currently) for scalar data.
*
* This class inherits from the interface class MiScalardSetIjk and, for
* convenience, from the implementation class MbScalardSet (both of
* which are convenience specializations of the template classes).
*
* MiScalardSetIjk provides the appropriate "get" method for structured
* volume meshes.  MbScalardSet provides implementations of some of
* the miscellaneous interface methods (getName, getTimeStamp, etc).
*
* To complete the implementation of the interface we must provide:
*    virtual double get(size_t i, size_t j, size_t k) const;
*    virtual double getMin() const;
*    virtual double getMax() const;
*
* Create and initialize the object by passing an SoVolumeData node to
* the constructor.  The resulting object can be used as any other
* MeshViz scalar data set.  The constructor initializes the data set
* name to "VolVizData", the data binding to PER_NODE and the timestamp
* to 1.
*
* The obvious approach is to represent a volume data set as a regular
* volume mesh with per_cell data values.  In other words each voxel
* is represented as a cell in the mesh and the number of cells on each
* axis is the same as the number of voxels.  To take this approach,
* explicitly set the data binding to PER_CELL using setBinding().
*
* Generally though the goal is to extract isosurfaces from the volume 
* data and isosurface extraction only works with per_node data values.
* In this case we should represent the volume data as a mesh with
* its nodes (vertices) at the voxel centers.  In this case the number
* of cells on each axis is N-1 and the geometry extent of the mesh is
* 1/2 voxel less than the volume in every direction.
* 
*/

//
// TODO:
//  - Implement more constructors (specific name, etc) like MbDataSet.
//
// TO CONSIDER:
//  - Emulate effect of SoTransferFunction min/maxValue fields?
//  - Limit rendering to current VolumeViz ROI?
//
// Original: MHeck, VSG

#define ADD_VERSIONING_TO_DLL 1 // Hack for ebug #2554
#include <data/MbDataSet.h>
#include <MeshVizXLM/mesh/data/MiDataSetIjk.h> 
#include <data/MbScalarSetI.h>

#include <SimpleTileCache.h>

class  SoNodeSensor;
class  SoVolumeData;
class  SoLDMTopoOctree;

class VolVizScalarSet : virtual public MiScalardSetIjk, public MbScalarSetI<double>
{
public:
  /**
  * Constructor. @BR
  * The parameter is a pointer to the volume data node that will
  * be the data source.  This node will be ref'd.
  *
  * Initializes name to "VolVizData", data binding to PER_NODE and
  * timestamp to 1.
  */
  VolVizScalarSet( const SoVolumeData *volume );

  /**
   * Destructor.
   */
  ~VolVizScalarSet();

  // ----- Implementation methods for interface -----
  /**
  * Returns the ijk-th value of this set.
  */
  virtual double get(size_t i, size_t j, size_t k) const;

  /**
  * Returns the minimum value of this set.
  */
  virtual double getMin() const;
  
  /**
  * Returns the maximum value of this set.
  */
  virtual double getMax() const;

private:
  // ----- Utility functions for this implementation -----

  // Internal non-const query to allow modifying object state.
  double _get(size_t i, size_t j, size_t k);
  SoRef<SoVolumeData>       m_pVolData;
  SbVec3i32                 m_tileDimension;
  SoDataSet::DataType       m_dataType;
  SoLDMTopoOctree           *m_pTopoOctree;

  // cache common to all threads, it is protected by mutex to avoid issue on concurrency
  SbThreadMutex m_cacheAccessLock;
  SimpleTileCache* m_tileCache;
  // cache specific to each threads (feed by reference to global cache entry)
  SbThreadStorage<SimpleTileCache*> m_tileCacheTLS;

  // A nodesensor attached to the volumeData is in charge of cleaning the cache 
  // if any modification is detected on the VolumeData.
  SoNodeSensor* m_volDataSensor;                    // sensor to track input data changes
  static void volDataChangedCB(void *data, SoSensor *sensor); // cb called when data changed
  void volDataChanged();
};

////////////////////////////////////////////////////////////////////////
inline double
VolVizScalarSet::get(size_t i, size_t j, size_t k) const
{ // Call non-const version
  // (Because we need to modify m_dataInfo to track current data tile.)
  return (const_cast<VolVizScalarSet*>(this))->_get( i, j, k );
}

#endif // VOLVIZSCALARSET


