/**
* @page TutorialCombineColorMapping combining scalars into color tutorial
*
* @DTEXT Demonstrates the combinecolormapping in an interactive program.
*
* @DESCRIPTION This program shows how to combine scalar values into a color on a mesh representation.
*
* A hexahedron mesh is loaded at start-up.
* The full mesh skin is added to the scene.
*
* Press 1 to map the first scalarset.
* Press 2 to map the second scalarset.
* Press 3 to combine both scalarsets.
*
*/
#include "TutorialCombineColorMapping.h"

#include <Inventor/nodes/SoEventCallback.h>

#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoScalarSet.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoLinearColorMapping.h>
#include <MeshVizXLM/mapping/nodes/MoCombineColorMapping.h>

#include <Inventor/nodes/SoSwitch.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialCombineColorMapping::TutorialCombineColorMapping() : 
TutorialBase("TutorialCombineColorMapping"), m_currentScalar(0)
{
}

//---------------------------------------------------------------------
TutorialCombineColorMapping::~TutorialCombineColorMapping()
{
}

//---------------------------------------------------------------------
const MbVolumeMeshHexahedron<>*
TutorialCombineColorMapping::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;
  RootSphereFctor rsphere;
  m_meshBuilder.addScalarFctor(&rsphere, "RootSphere");
  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  return &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize, MbVec3d(0), MbVec3d(m_defaultMeshSize));
}

//---------------------------------------------------------------------
SoNode*
TutorialCombineColorMapping::buildSceneGraph()
{
  cout << "Press 1 to display sphere scalarset." << endl;
  cout << "Press 2 to display gradient scalarset." << endl;
  cout << "Press 3 to combine both scalarset." << endl;
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MbVolumeMeshHexahedron<>* mesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(mesh);

  m_colorMapSwitch = new SoSwitch;
  m_colorMapSwitch->whichChild = 0;

  const MbScalarSetI<double>* scalar1 = mesh->getScalarSet("RootSphere");
  const MbScalarSetI<double>* scalar2 = mesh->getScalarSet("$MyScalarSet");
  MoLinearColorMapping* linearColorMapping = new MoLinearColorMapping;
  float values[2];
  values[0] = std::min((float)scalar1->getMin(), (float)scalar2->getMin());
  values[1] = std::max((float)scalar1->getMax(), (float)scalar2->getMax());
  linearColorMapping->values.setValues(0, 2, values);
  SbVec4f colors[2];
  colors[0].setValue(1, 0, 0, 1);
  colors[1].setValue(0, 0, 1, 1);
  linearColorMapping->colors.setValues(0, 2, colors);

  std::vector< std::pair<double, SbColorRGBA> > colormap;
  colormap.reserve(2);
  for (int i = 0; i < 2; i++)
    colormap.push_back(std::pair<double, SbColorRGBA>(values[i], colors[i]));
  m_AverageToColorMapping.setColormap(colormap);
  MoCombineColorMapping* combineColorMapping = new MoCombineColorMapping;
  combineColorMapping->setColorMapping(&m_AverageToColorMapping);

  m_scalarSwitch = new SoSwitch;
  m_scalarSwitch->whichChild = SO_SWITCH_ALL;
  // Scalar Value
  MoScalarSet *moScalarSet1 = new MoScalarSet;
  moScalarSet1->setScalarSet(scalar1);
  // Sphere Value
  MoScalarSet *moScalarSet2 = new MoScalarSet;
  moScalarSet2->setScalarSet(scalar2);

  MoMeshSkin *skin = new MoMeshSkin;
  skin->colorScalarSetId = 0;

  SoSeparator* root = new SoSeparator;
  root->addChild(soShapeHint);
  root->addChild(m_colorMapSwitch);
  {
    m_colorMapSwitch->addChild(linearColorMapping);
    m_colorMapSwitch->addChild(combineColorMapping);
  }
  root->addChild(m_scalarSwitch);
  {
    m_scalarSwitch->addChild(moScalarSet1);
    m_scalarSwitch->addChild(moScalarSet2);
  }
  root->addChild(moMesh);
  root->addChild(skin);

  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialCombineColorMapping::keyPressed(SoEventCallback *eventCB)
{
  bool handled = true;
  const SoEvent *ev = eventCB->getEvent();

  if (SO_KEY_PRESS_EVENT(ev, NUMBER_1))
  {
    m_colorMapSwitch->whichChild = 0;
    m_scalarSwitch->whichChild = 0;
    std::cout << "Mapping sphere scalar set." << std::endl;
  }
  else if (SO_KEY_PRESS_EVENT(ev, NUMBER_2))
  {
    m_colorMapSwitch->whichChild = 0;
    m_scalarSwitch->whichChild = 1;
    std::cout << "Mapping gradient scalar set." << std::endl;
  }
  else if (SO_KEY_PRESS_EVENT(ev, NUMBER_3))
  {
    m_colorMapSwitch->whichChild = 1;
    m_scalarSwitch->whichChild = SO_SWITCH_ALL;
    std::cout << "Mapping combined scalar sets." << std::endl;
  }
  else
    handled = false;

  if (handled)
    eventCB->setHandled();
}


