 /**
 * @page TutorialPicking1 Picking tutorial 1
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation.
 *
 * A hexahedron mesh is loaded at start-up.
 * The full mesh skin is added to the scene. 
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * The cell id of the selected cell is displayed in the console.
 */
#include "TutorialPicking1.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialPicking1::TutorialPicking1() : 
  TutorialBase("TutorialPicking1")
{
}

//---------------------------------------------------------------------
TutorialPicking1::~TutorialPicking1()
{
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking1::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking1::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moDrawStyle = new MoDrawStyle;
  moDrawStyle->displayFaces = true;
  moDrawStyle->displayEdges = true;

  MoMaterial* moSkinMaterial = new MoMaterial;
  moSkinMaterial->faceColoring = MoMaterial::COLOR;
  moSkinMaterial->faceColor = SbColor(1,0,0);
  moSkinMaterial->lineColoring = MoMaterial::COLOR;
  moSkinMaterial->lineColor = SbColor(0,0,1);

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;

  root->addChild(soShapeHint);
  root->addChild(moMesh);
  root->addChild(moDrawStyle);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moSkinMaterial);
    sepSkin->addChild(moMeshSkin);
  }
  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialPicking1::mousePressed(SoEventCallback *eventCB)
{
  // listen only mouse pressed event for button1
  if (!SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
    return;

  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t cellId = fdetail->getCellIndex();

      //std::string propertyName = mdetail->getColorScalarSet()->getName();
      
      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.

      cout << "cell picked: " << cellId << endl;
    }
  }

  eventCB->setHandled();
}


