 /**
 * @page TutorialPicking4 Picking tutorial 4
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation
 * and how to highlight and manage the list of picked cells.
 * Similar to tutorial 2 but removes the selected cells from the skin
 * using a cell filter.
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * Picked cells are drawn "shrunken" (smaller size) in cyan.
 */
#include "TutorialPicking4.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMeshCellShape.h>
#include <MeshVizXLM/mapping/nodes/MoCellFilter.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
class MyCellFilterI: public MiCellFilterI
{
private:
  MoMeshCellShape *m_cellShape;
  size_t m_timeStamp;
public:
  MyCellFilterI(MoMeshCellShape* cellShape) : m_cellShape(cellShape) {}

  virtual bool acceptCell(size_t cellIndex) const
  {
    // We use the cellIndices field as criterion of the cell filter.
    // However this field is not sorted, so using the SoMFint::find is 
    // not the most optimal implementation of accepCell. 
    // For instance, using an additional std::set to store the ids of
    // the selected cell should be more efficient.
    return (m_cellShape->cellIndices.find((int)cellIndex) == -1);
  }

  virtual size_t getTimeStamp() const 
  {
    return m_timeStamp;
  }

  void touch()
  {
    m_timeStamp = MxTimeStamp::getTimeStamp();
  }
};

//---------------------------------------------------------------------
TutorialPicking4::TutorialPicking4() : 
  TutorialBase("TutorialPicking4")
{
}

//---------------------------------------------------------------------
TutorialPicking4::~TutorialPicking4()
{
  delete m_cellFilterI;
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking4::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking4::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moDrawStyle = new MoDrawStyle;
  moDrawStyle->displayFaces = true;
  moDrawStyle->displayEdges = true;

  MoMaterial* moSkinMaterial = new MoMaterial;
  moSkinMaterial->faceColoring = MoMaterial::COLOR;
  moSkinMaterial->faceColor = SbColor(1,0,0);
  moSkinMaterial->lineColoring = MoMaterial::COLOR;
  moSkinMaterial->lineColor = SbColor(0,0,1);

  MoMaterial* moCellShapeMaterial = new MoMaterial;
  moCellShapeMaterial->faceColoring = MoMaterial::COLOR;
  moCellShapeMaterial->faceColor = SbColor(0,1,1);
  moCellShapeMaterial->lineColoring = MoMaterial::COLOR;
  moCellShapeMaterial->lineColor = SbColor(0,0,1);

  m_cellshape = new MoMeshCellShape;
  m_cellshape->inclusiveList = true;
  m_cellshape->factor = 0.8f;
  m_cellshape->showNodeName = FALSE;

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  m_cellFilterI = new MyCellFilterI(m_cellshape);
  MoCellFilter* moCellFilter = new MoCellFilter;
  moCellFilter->setCellFilter(m_cellFilterI);

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;
  SoSeparator* sepCellShape = new SoSeparator;

  root->addChild(soShapeHint);
  root->addChild(moMesh);
  root->addChild(moDrawStyle);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moSkinMaterial);
    sepSkin->addChild(moCellFilter);
    sepSkin->addChild(moMeshSkin);
  }

  root->addChild(sepCellShape);
  {
    sepCellShape->addChild(moCellShapeMaterial);
    sepCellShape->addChild(m_cellshape);
  }

  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialPicking4::mousePressed(SoEventCallback *eventCB)
{
  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t cellId = fdetail->getCellIndex();
      //double value = fdetail->getValue(pickedPoint->getPoint());

      // Get detail about the picked mesh 
      //const MoMeshDetail* mdetail = fdetail->getMeshDetail();
      //const MiMesh* pickedMesh = mdetail->getMesh();
      //MeshType meshType = mdetail->getMeshType();
      //std::string propertyName = mdetail->getColorScalarSet()->getName();

      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.

      // add the id of picked cell in the selection only if it is not already selected.
      bool addIfNotAlreadySelected = true;
      int foundId= m_cellshape->cellIndices.find((int)cellId,addIfNotAlreadySelected);
      if (foundId == -1 )
      {
        cout << "new selected cell: " << cellId << endl;
        m_cellFilterI->touch();
      }
    }
  }

  eventCB->setHandled();
}
