 /**
 * @page TutorialPicking5 Picking tutorial 5
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation
 * and how to highlight and manage the list of picked cells.
 * Similar to tutorial 4 but it uses a class SelectedCells inherited 
 * from MoMeshCellShape that also implements the MiCellFilterI interface.
 * The SelectedCells is an optimal cell filtering implementation 
 * compared to tutorial 4 thanks to an array containing one 1 bit flag per cell.
 * The SelectedCells instance is given to the MoCellFilter instance. 
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * The selected cell is drawn "shrunken" (smaller size) in cyan.
 */
#include "TutorialPicking5.h"
#include "SelectedCells.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMeshCellShape.h>
#include <MeshVizXLM/mapping/nodes/MoCellFilter.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialPicking5::TutorialPicking5() : 
  TutorialBase("TutorialPicking5")
{
}

//---------------------------------------------------------------------
TutorialPicking5::~TutorialPicking5()
{
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking5::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking5::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moDrawStyle = new MoDrawStyle;
  moDrawStyle->displayFaces = true;
  moDrawStyle->displayEdges = true;

  MoMaterial* moSkinMaterial = new MoMaterial;
  moSkinMaterial->faceColoring = MoMaterial::COLOR;
  moSkinMaterial->faceColor = SbColor(1,0,0);
  moSkinMaterial->lineColoring = MoMaterial::COLOR;
  moSkinMaterial->lineColor = SbColor(0,0,1);

  MoMaterial* moCellShapeMaterial = new MoMaterial;
  moCellShapeMaterial->faceColoring = MoMaterial::COLOR;
  moCellShapeMaterial->faceColor = SbColor(0,1,1);
  moCellShapeMaterial->lineColoring = MoMaterial::COLOR;
  moCellShapeMaterial->lineColor = SbColor(0,0,1);

  m_cellshape = new SelectedCells(NI*NJ*NK);
  m_cellshape->inclusiveList = true;
  m_cellshape->factor = 0.8f;
  m_cellshape->showNodeName = FALSE;

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  MoCellFilter* moCellFilter = new MoCellFilter;
  moCellFilter->setCellFilter(m_cellshape);

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;
  SoSeparator* sepCellShape = new SoSeparator;

  root->addChild(soShapeHint);
  root->addChild(moMesh);
  root->addChild(moDrawStyle);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moSkinMaterial);
    sepSkin->addChild(moCellFilter);
    sepSkin->addChild(moMeshSkin);
  }

  root->addChild(sepCellShape);
  {
    sepCellShape->addChild(moCellShapeMaterial);
    sepCellShape->addChild(m_cellshape);
  }

  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialPicking5::mousePressed(SoEventCallback *eventCB)
{
  if (!SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
    return;

  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t cellId = fdetail->getCellIndex();
      //double value = fdetail->getValue(pickedPoint->getPoint());

      // Get detail about the picked mesh 
      //const MoMeshDetail* mdetail = fdetail->getMeshDetail();
      //const MiMesh* pickedMesh = mdetail->getMesh();
      //MeshType meshType = mdetail->getMeshType();
      //std::string propertyName = mdetail->getColorScalarSet()->getName();

      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.

      // add the picked cell to the indices of the cell shape.
      // this cell shape is also used as cell filter for the skin.
      m_cellshape->addPickedCell(cellId);
    }
  }

  eventCB->setHandled();
}
