 /**
 * @page TutorialPicking7 Picking tutorial 7
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation
 * and how to highlight and manage the list of picked faces on the skin.
 *
 * This tutorial is similar to tutorial 6 but the selected faces are also 
 * removed from the skin displayed. That generates some holes in the skin where
 * the faces are picked.
 * It uses 2 MoCellShape instance: one instance for the skin representation 
 * and one for the selected faces drawn in blue.
 * It uses also the SoField::connectFrom in order to share the cellIndices
 * between both instances of MoCellShape.
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * Picked cells are drawn in cyan. 
 * Number of picked faces is displayed in the console window.
 */
#include "TutorialPicking7.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMeshCellShape.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialPicking7::TutorialPicking7() : 
  TutorialBase("TutorialPicking7")
{
}

//---------------------------------------------------------------------
TutorialPicking7::~TutorialPicking7()
{
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking7::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking7::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering = SoShapeHints::CLOCKWISE;

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moSkinDrawstyle = new MoDrawStyle;
  // display only edge of skin: let edge be always visible 
  // for cell selected.
  // face of cell are only drawn by the following m_cellShapeForSkin
  moSkinDrawstyle->displayFaces = false;
  moSkinDrawstyle->displayEdges = true;

  MoDrawStyle* moCellShapeDrawstyle = new MoDrawStyle;
  moCellShapeDrawstyle->displayFaces = true;
  moCellShapeDrawstyle->displayEdges = true;

  MoMaterial* moCellShapeMaterial = new MoMaterial;
  moCellShapeMaterial->faceColoring = MoMaterial::COLOR;
  moCellShapeMaterial->faceColor = SbColor(1,0,0);
  moCellShapeMaterial->lineColoring = MoMaterial::COLOR;
  moCellShapeMaterial->lineColor = SbColor(1,1,1);

  MoMaterial* moCellShapeMaterial2 = new MoMaterial;
  moCellShapeMaterial2->faceColoring = MoMaterial::COLOR;
  moCellShapeMaterial2->faceColor = SbColor(0,1,1);
  moCellShapeMaterial2->lineColoring = MoMaterial::COLOR;
  moCellShapeMaterial2->lineColor = SbColor(1,1,0);

  SoPickStyle* moCellShapePickStyle = new SoPickStyle;
  moCellShapePickStyle->style = SoPickStyle::UNPICKABLE;

  m_cellShapeForSkin = new SelectedFaces;
  m_cellShapeForSkin->inclusiveList = false;
  m_cellShapeForSkin->factor = 1.0f;

  MoMeshCellShape* moCellShapeForSelectedFaces = new MoMeshCellShape;
  moCellShapeForSelectedFaces->inclusiveList = true;
  moCellShapeForSelectedFaces->factor = 0.8f;
  moCellShapeForSelectedFaces->cellIndices.connectFrom(&m_cellShapeForSkin->cellIndices);

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  MoMesh* moMeshFromSkin = new MoMesh;
  moMeshFromSkin->connectFrom(moMeshSkin);

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;
  SoSeparator* sepCellShape = new SoSeparator;

  root->addChild(soShapeHint);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moMesh);
    sepSkin->addChild(moSkinDrawstyle);
    sepSkin->addChild(moMeshSkin);
  }

  root->addChild(sepCellShape);
  {
    sepCellShape->addChild(moCellShapePickStyle);
    sepCellShape->addChild(moMeshFromSkin);
    sepCellShape->addChild(moCellShapeDrawstyle);
    sepCellShape->addChild(moCellShapeMaterial);
    sepCellShape->addChild(m_cellShapeForSkin);
    sepCellShape->addChild(moCellShapeMaterial2);
    sepCellShape->addChild(moCellShapeForSelectedFaces);
  }

  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialPicking7::mousePressed(SoEventCallback *eventCB)
{
  if (!SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
    return;

  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t faceid = fdetail->getFaceIndex();
      //double value = fdetail->getValue(pickedPoint->getPoint());

      // Get detail about the picked mesh 
      //const MoMeshDetail* mdetail = fdetail->getMeshDetail();
      //const MiMesh* pickedMesh = mdetail->getMesh();
      //MeshType meshType = mdetail->getMeshType();
      //std::string propertyName = mdetail->getColorScalarSet()->getName();

      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.
      m_cellShapeForSkin->addPickedFace(faceid);
    }
  }

  eventCB->setHandled();
}
