 /**
 * @page TutorialPicking8 Picking tutorial 8
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation
 * and how to highlight and manage the list of picked faces on the skin.
 *
 * This tutorial is similar to tutorial 6 but the node ids of the selected faces
 * are displayed. It also prints in the console the relative number of the 
 * picked face to the selected cell. In order to get this number, it compares 
 * the center of the selected face to each face 
 * center of the selected volume cell.
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * Picked faces are drawn in cyan.
 * Node ids of the picked faces are displayed in white.
 */
#include "TutorialPicking8.h"

#include "SelectedFaces.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDepthOffset.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialPicking8::TutorialPicking8() : 
  TutorialBase("TutorialPicking8")
{
}

//---------------------------------------------------------------------
TutorialPicking8::~TutorialPicking8()
{
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking8::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking8::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moDrawStyle = new MoDrawStyle;
  moDrawStyle->displayFaces = true;
  moDrawStyle->displayEdges = true;

  MoMaterial* moSkinMaterial = new MoMaterial;
  moSkinMaterial->faceColoring = MoMaterial::COLOR;
  moSkinMaterial->faceColor = SbColor(1,0,0);
  moSkinMaterial->lineColoring = MoMaterial::COLOR;
  moSkinMaterial->lineColor = SbColor(0,0,1);

  MoMaterial* moCellShapeMaterial = new MoMaterial;
  moCellShapeMaterial->faceColoring = MoMaterial::COLOR;
  moCellShapeMaterial->faceColor = SbColor(0,1,1);
  moCellShapeMaterial->lineColoring = MoMaterial::COLOR;
  moCellShapeMaterial->lineColor = SbColor(0,0,1);

  SoPickStyle* moCellShapePickStyle = new SoPickStyle;
  moCellShapePickStyle->style = SoPickStyle::UNPICKABLE;

  m_cellShapeSelection = new SelectedFaces;
  m_cellShapeSelection->inclusiveList = true;
  m_cellShapeSelection->factor = 0.8f;
  m_cellShapeSelection->showCellName = true;
  m_cellShapeSelection->showNodeName = true;

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  MoMesh* moMeshFromSkin = new MoMesh;
  moMeshFromSkin->connectFrom(moMeshSkin);

  SoDepthOffset* soDepthOffset = new SoDepthOffset;
  soDepthOffset->offset = 0.01f;
  soDepthOffset->on = true;

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;
  SoSeparator* sepCellShape = new SoSeparator;

  root->addChild(soShapeHint);
  root->addChild(moDrawStyle);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moMesh);
    sepSkin->addChild(moSkinMaterial);
    sepSkin->addChild(moMeshSkin);
  }

  root->addChild(sepCellShape);
  {
    sepCellShape->addChild(soDepthOffset);
    sepCellShape->addChild(moCellShapePickStyle);
    sepCellShape->addChild(moMeshFromSkin);
    sepCellShape->addChild(moCellShapeMaterial);
    sepCellShape->addChild(m_cellShapeSelection);
  }

  return root;
}

//---------------------------------------------------------------------
// Method called each time the mouse button is clicked in the viewer window
void 
TutorialPicking8::mousePressed(SoEventCallback *eventCB)
{
  if (!SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
    return;

  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t cellid = fdetail->getCellIndex();
      size_t faceid = fdetail->getFaceIndex();
      //double value = fdetail->getValue(pickedPoint->getPoint());

      // Get detail about the picked mesh 
      const MoMeshDetail* mdetail = fdetail->getMeshDetail();
      const MiMesh* pickedMesh = mdetail->getMesh();
      //MeshType meshType = mdetail->getMeshType();
      //std::string propertyName = mdetail->getColorScalarSet()->getName();

      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.

      const MeXSurfaceMeshUnstructured* mexmesh = fdetail->getMeshRepresentationDetail()->getExtractedMesh();
      m_cellShapeSelection->addPickedFace(faceid,mexmesh, cellid,pickedMesh);

    }
  }

  eventCB->setHandled();
}
