 /**
 * TutorialPicking9 Picking tutorial 9
 *
 * @DTEXT Demonstrates simple picking in an interactive program.
 *
 * @DESCRIPTION This program shows how to pick cells on a mesh representation
 * and how to highlight and manage the list of picked faces on the skin.
 *
 * This tutorial is similar to tutorial 8 but the cell id is drawn at the 
 * center of each selected faces. The cell id displayed is the id of the
 * cell containing the selected faces.
 * A new class MeshOfSelectedFaces is used to manage the list of selected 
 * faces. This class implement the MiSurfaceMeshUnstructured.
 *
 * Note: the selected faces are drawn by using a MoMeshCellShape in order
 * to shrink them and to display the id of node and selected cell.
 * An simple MoMeshSurface could also be used to display the selected
 * face as is.
 *
 * Click on the viewer's arrow icon to switch from viewing mode to selection mode.
 *
 * Move the mouse cursor over the mesh skin and click the left mouse button.
 * Picked faces are drawn in cyan.
 * Cell id and node ids of the picked faces are displayed in white.
 */
#include "TutorialPicking9.h"

#include <Inventor/SoPickedPoint.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDepthOffset.h>

#include <MeshVizXLM/mapping/details/MoFaceDetailI.h>
#include <MeshVizXLM/mapping/details/MoMeshDetail.h>
#include <MeshVizXLM/mapping/nodes/MoMesh.h>
#include <MeshVizXLM/mapping/nodes/MoMeshSkin.h>
#include <MeshVizXLM/mapping/nodes/MoDrawStyle.h>
#include <MeshVizXLM/mapping/nodes/MoMaterial.h>
#include <MeshVizXLM/mapping/nodes/MoMeshCellShape.h>

using namespace::std;

#define NI 10
#define NJ 10
#define NK 10

//---------------------------------------------------------------------
TutorialPicking9::TutorialPicking9() : 
  TutorialBase("TutorialPicking9")
{
}

//---------------------------------------------------------------------
TutorialPicking9::~TutorialPicking9()
{
}

//---------------------------------------------------------------------
const MiMesh* 
TutorialPicking9::buildUnstMesh()
{
  cout << "Displaying Hexahedron mesh" << endl;

  MbVec3<size_t> m_defaultMeshSize(NI,NJ,NK);
  const MiVolumeMeshUnstructured* meshh = &m_meshBuilder.getMeshHexahedron(m_defaultMeshSize,MbVec3d(0),MbVec3d(m_defaultMeshSize));

  m_meshOfSelectedFaces = new MeshOfSelectedFaces;
  m_meshOfSelectedFaces->setVolumeMesh(meshh);

  return meshh;

}

//---------------------------------------------------------------------
SoNode*
TutorialPicking9::buildSceneGraph()
{
  SoShapeHints * soShapeHint = new SoShapeHints;
  soShapeHint->vertexOrdering.setValue(SoShapeHints::CLOCKWISE);

  const MiMesh* myMesh = buildUnstMesh();
  MoMesh* moMesh = new MoMesh();
  moMesh->setMesh(myMesh);

  MoDrawStyle* moDrawStyle = new MoDrawStyle;
  moDrawStyle->displayFaces = true;
  moDrawStyle->displayEdges = true;

  MoMaterial* moMaterialSkin = new MoMaterial;
  moMaterialSkin->faceColoring = MoMaterial::COLOR;
  moMaterialSkin->faceColor = SbColor(1,0,0);
  moMaterialSkin->lineColoring = MoMaterial::COLOR;
  moMaterialSkin->lineColor = SbColor(0,0,1);

  MoMaterial* moMaterialSelection = new MoMaterial;
  moMaterialSelection->faceColoring = MoMaterial::COLOR;
  moMaterialSelection->faceColor = SbColor(0,1,1);
  moMaterialSelection->lineColoring = MoMaterial::COLOR;
  moMaterialSelection->lineColor = SbColor(1,1,0);

  SoPickStyle* moPickStyleSelection = new SoPickStyle;
  moPickStyleSelection->style = SoPickStyle::UNPICKABLE;

  MoMeshCellShape* moCellShapeSelection = new MoMeshCellShape;
  moCellShapeSelection->inclusiveList = false;
  moCellShapeSelection->factor = 0.8f;
  moCellShapeSelection->showCellName = true;
  moCellShapeSelection->showNodeName = true;

  MoMeshSkin* moMeshSkin = new MoMeshSkin;

  SoDepthOffset* soDepthOffset = new SoDepthOffset;
  soDepthOffset->offset = 0.01f;
  soDepthOffset->on = true;

  SoSeparator* root = new SoSeparator;

  SoSeparator* sepSkin = new SoSeparator;
  SoSeparator* sepSelection = new SoSeparator;

  root->addChild(soShapeHint);
  root->addChild(moDrawStyle);

  MoMesh* moMeshSelection = new MoMesh;
  moMeshSelection->setMesh(m_meshOfSelectedFaces);

  root->addChild(sepSkin);
  {
    sepSkin->addChild(moMesh);
    sepSkin->addChild(moMaterialSkin);
    sepSkin->addChild(moMeshSkin);
  }

  root->addChild(sepSelection);
  {
    sepSelection->addChild(moMeshSelection);
    sepSelection->addChild(soDepthOffset);
    sepSelection->addChild(moPickStyleSelection);
    sepSelection->addChild(moMaterialSelection);
    sepSelection->addChild(moCellShapeSelection);
  }

  return root;
}

//---------------------------------------------------------------------
// Method called each time mouse button is clicked in the viewer window
void 
TutorialPicking9::mousePressed(SoEventCallback *eventCB)
{
  if (!SoMouseButtonEvent::isButtonPressEvent(eventCB->getEvent(),SoMouseButtonEvent::BUTTON1))
    return;

  // Get the picked point (if any)
  // Picking is automatically performed using the event location
  const SoPickedPoint *pickedPoint = eventCB->getPickedPoint();
  if (pickedPoint) 
  {
    // Something has been picked : It could be either an Open Inventor shape 
    // like SoCone or a mesh representation. Get detail and check type.
    const SoDetail* detail = pickedPoint->getDetail();
    if (detail->isOfType(MoFaceDetailI::getClassTypeId()))
    {
      // A face of a mesh representation of a unstructured mesh 
      // has been picked. Get detail about the picked face.
      MoFaceDetailI* fdetail = (MoFaceDetailI*) detail;

      // Get the picked cell and the value at the picked point.
      size_t cellid = fdetail->getCellIndex();
      size_t faceid = fdetail->getFaceIndex();
      //double value = fdetail->getValue(pickedPoint->getPoint());

      // Get detail about the picked mesh 
      const MoMeshDetail* mdetail = fdetail->getMeshDetail();
      const MiMesh* pickedMesh = mdetail->getMesh();
      //MeshType meshType = mdetail->getMeshType();
      //std::string propertyName = mdetail->getColorScalarSet()->getName();

      // Notes: 
      // value, pickedMesh, meshType, propertyName are useful most of the time
      // when doing a picking even if they are not used in this tutorial.
      //
      // The pickedMesh can be cast to the application
      // real mesh class in order to get any application data.

      const MeXSurfaceMeshUnstructured* mexmesh = fdetail->getMeshRepresentationDetail()->getExtractedMesh();
      m_meshOfSelectedFaces->addPickedFace(faceid,mexmesh, cellid,pickedMesh);
    }
  }

  eventCB->setHandled();
}
