 /**
 * TutorialSliceArea computes the area of an extracted plane slice.
 *
 * @DTEXT Shows how to compute the area of an extracted plane slice.
 *
 * @DESCRIPTION This program shows how to compute the area of an extracted plane slice.
 *
 * A regular mesh is built at start-up.
 * A plane slice is built by calling extractSlice on a MiPlaneSliceExtractIjk extractor.
 * The area of this slice is computed by the method computeAreaOfSurface.
 * This method sums the areas of all the cells in the plane slice surface mesh.
 * 
 * A second mesh composed by tetrahedrons is built. It contains some dead cells.
 * A plane slice is built by calling extractSlice on a MiPlaneSliceExtractUnstructured extractor.
 * The area of this second slice is computed by the method computeAreaOfSurface.
 * The second area is not equal to the first one because of the existing dead cells.
 *
 * Exits after printing the second area.
 */
#include <MeshVizXLM/MiMeshViz.h>
#include <MeshVizXLM/extractors/MiPlaneSliceExtractUnstructured.h>
#include <MeshVizXLM/extractors/MiPlaneSliceExtractIjk.h>
#include <Inventor/SoDB.h>

#include <MbSampleMeshBuilder.h>

using namespace std;

MiPlaneSliceExtractIjk* s_planeSliceExtractR;
MiPlaneSliceExtractUnstructured* s_planeSliceExtractU;
MbSampleMeshBuilder<MbVec3d,double> m_meshBuilder;

class MyDeadCellFctIjk : public MbDeadCellFctorIjk
{
public:
  /** custom function to compute a dead cell for the given index 
    exclude a single line of cells
  */
  virtual bool operator()(size_t /*i*/, size_t j,size_t k) const
  {
    return (k==0 && j==5) ? true: false;
  }
};
MyDeadCellFctIjk deadCellsFctorIjk;

class MyDeadCellFctI : public MbDeadCellFctorI
{
public:
  /** custom function to compute a dead cell for the given index 
    exclude the second cell only 
  */
  virtual bool operator()(size_t i) const 
  {
    return (i==1) ? true: false;
  }
};
MyDeadCellFctI deadCellsFctorI;


MbVolumeMeshRegular<double,double,MbVec3d>& getMeshRegular(size_t numCellsPerDim, double length, bool deadCells)
{
  MbVec3<size_t> dim(numCellsPerDim);
  MbVec3d max(length);
  size_t numCells = dim[0]*dim[1]*dim[2];

  if (deadCells)
    m_meshBuilder.addDeadCellFctor(&deadCellsFctorIjk);

  MbVolumeMeshRegular<double,double,MbVec3d>& mesh = m_meshBuilder.getVolumeMeshRegular(dim,MbVec3d(0),max);

  std::cout << "Regular volume mesh built, num cells=" << numCells 
    << ", num nodes=" << (dim[0]+1)*(dim[1]+1)*(dim[2]+1);
  if (deadCells)
    std::cout << " with dead cells" << std::endl;
  else
    std::cout << std::endl;

  return mesh;
}


MbVolumeMeshTetrahedron<MbVec3d,double,MbVec3d>& getMeshTetrahedron(size_t numCellsPerDim, double length, bool deadCells)
{
  MbVec3<size_t> dim(numCellsPerDim);
  MbVec3d max(length);
  size_t numCells = dim[0]*dim[1]*dim[2];

  if (deadCells)
    m_meshBuilder.addDeadCellFctor(&deadCellsFctorI);

  MbVolumeMeshTetrahedron<MbVec3d,double,MbVec3d>& mesh = m_meshBuilder.getMeshTetrahedron(dim,MbVec3d(0),max);

  std::cout << "Regular volume mesh built, num cells=" << numCells 
    << ", num nodes=" << (dim[0]+1)*(dim[1]+1)*(dim[2]+1);
  if (deadCells)
    std::cout << " with dead cells" << std::endl;
  else
    std::cout << std::endl;

  return mesh;
}


//-----------------------------------------------------------------------------
double computeAreaOfSurface(const MeXSurfaceMeshUnstructured& surface)
{
  const MeXSurfaceTopologyExplicitI& topology = surface.getTopology();
  const MiGeometryI& geometry = surface.getGeometry();
  double area = 0;
  for (size_t i=0; i<topology.getNumCells(); i++)
    area += topology.getCell(i)->getArea(&geometry);
  return area;
}


//-----------------------------------------------------------------------------
int
main(int, char **)
{
  SoDB::init();
  MiMeshViz::init();

  // Compute the area of a plane slice extracted from a regular mesh
  MbVolumeMeshRegular<double,double,MbVec3d>& volumeMeshR = getMeshRegular(50,100.,false);
  s_planeSliceExtractR = MiPlaneSliceExtractIjk::getNewInstance(volumeMeshR);
  
  const MeXSurfaceMeshUnstructured& sliceR = s_planeSliceExtractR->extractSlice(MbVec3d(0,0,1),0.1);
  double areaR = computeAreaOfSurface(sliceR);
  cout << "Area R is :" << areaR << endl;
  
  // Compute the area of a plane slice extracted from a regular mesh subdivided in tetrahedrons.
  // The true boolean value is set to define some dead cells.
  MbVolumeMeshTetrahedron<MbVec3d,double,MbVec3d>& volumeMeshU = getMeshTetrahedron(50,100.,true);
  s_planeSliceExtractU = MiPlaneSliceExtractUnstructured::getNewInstance(volumeMeshU);

  const MeXSurfaceMeshUnstructured& sliceU = s_planeSliceExtractU->extractSlice(MbVec3d(0,0,1),0.1);
  double areaU = computeAreaOfSurface(sliceU);
  cout << "Area U is :" << areaU << endl;

  MiMeshViz::finish();
  SoDB::finish();

  return 0;
}
