#include <cstdio>
#include <Inventor/STL/string>
#include "PolyhedralMeshReader.h"
#include <mesh/volumes/MbVolumeMeshPolyhedron.h>

#include <Inventor/helpers/SbFileHelper.h>

#ifdef _WIN32
#  pragma warning( push )
// disable deprecation warning for fopen, fscanf ...
#  pragma warning(disable:4996)
#endif

using namespace std;

PolyhedralMeshReader::PolyhedralMeshReader()
: m_mesh(NULL), m_vec3Set(NULL), m_numNodePerFaceList(NULL), m_faceNodeList(NULL), m_elementFaces(NULL)
{
}

PolyhedralMeshReader::~PolyhedralMeshReader()
{
  if (m_elementFaces)
    delete m_elementFaces;
  if (m_numNodePerFaceList)
    delete m_numNodePerFaceList;
  if (m_faceNodeList)
    delete m_faceNodeList;
}

void PolyhedralMeshReader::clear()
{
  if (m_elementFaces)
    delete m_elementFaces;
  if (m_numNodePerFaceList)
    delete m_numNodePerFaceList;
  if (m_faceNodeList)
    delete m_faceNodeList;
  for ( size_t i = 0; i < m_scalarSets.size(); ++i)
    delete m_scalarSets[i];
  m_scalarSets.clear();

  delete m_vec3Set;

  delete m_mesh;
}

char* PolyhedralMeshReader::getLine() 
{
	int goOn=1;
	char* p = fgets(m_Line,1024,fp);
	while (goOn && p) {
		m_LineNumber++;
		goOn=0;
		if (CompareKeyword("--")) {
			p = fgets(m_Line,1024,fp);
			goOn=1;
		}
		if (p[0]==10) {
			p = fgets(m_Line,1024,fp);
			goOn=1;
		}
	}
	return(p);
}

size_t PolyhedralMeshReader::CompareKeyword(const char* keyword)
{
	size_t length=strlen(keyword);
	size_t res = strncmp(m_Line,keyword,length);
	return(res==0);
}

template < typename _T>
int PolyhedralMeshReader::read(size_t ndata,_T* databuffer, int& res_err, size_t& ndata_read)
{
	char seps[]= " \t\n";
	char* token;
	_T d1;
	ndata_read=0;
	res_err=0;

	char* p = getLine();
	m_LineNumber++;
	if (!p) {
		res_err = -1;
		return 0;
	}

	int goOn=1;
	while (goOn) {
		token = strtok(m_Line,seps);

		if (!token) goOn=0;
		while ( (token != NULL) && goOn ) {

      istringstream stoken(token,istringstream::in);
      stoken >> d1;

			if (!stoken.fail()) {
				databuffer[ndata_read]= d1;
				ndata_read++;
			} else {
				//check if reach end of record or just wrong coords
        if (token[0]=='#') {
          printf("warning: missing data \n");
					goOn=0;
				} else { //dummy 
					databuffer[ndata_read]= (_T) 1;
					ndata_read++; 
				}	
      }
			if (ndata_read>=ndata) 
        goOn=0;
			token = strtok(NULL,seps);
		}
		if (goOn) {
			p = getLine();
			m_LineNumber++;
		}
	}
	res_err = 1;
	return 1;
}

int PolyhedralMeshReader::ReadHeader()
{
  m_numVariables = 0;
  do
  {
    ++m_numVariables;
    getLine();
    m_LineNumber++;
  }while ( m_Line[0] == '"' );

  return 1;
}

int PolyhedralMeshReader::ReadZone()
{
 char seps[]= " ,\t\n";
 char* token = strtok(m_Line,seps);
 char word[256];
 

 do
 {
   istringstream stoken(token,istringstream::in);
   stoken.getline(word,256,'=');
   if(strcmp(word,"Nodes")==0)
   {
     stoken >> m_numNodes;
   }
   if(strcmp(word,"Faces")==0)
   {
     stoken >> m_numFaces;
   }
   if(strcmp(word,"Elements")==0)
   {
     stoken >> m_numElements;
     m_elementFaces = new vector<size_t>[m_numElements];
   }
   if(strcmp(word,"TotalNumFaceNodes")==0)
   {
     stoken >> m_totalNumFaceNodes;
   }
   
   //next token
   token = strtok(NULL,seps);
   if ( token == NULL)
   {
     getLine();
     m_LineNumber++;
     token = strtok(m_Line,seps);
   }

 } while ( token!=NULL && strcmp(word,"DT")!=0  );

 //read data blocks
  do
  {
   if (CompareKeyword(" DT")) {
     ReadData();
   }
   if (CompareKeyword("# node count")) {
     ReadNumNodeFace();
   }
   if (CompareKeyword("# face nodes")) {
     ReadFaceNodes();
   }
   if (CompareKeyword("# left elements") || CompareKeyword("# right elements")) {
     ReadElementFaces();
   }
   getLine();
   m_LineNumber++;
  } while (!feof(fp) && !CompareKeyword("ZONE"));
  
  return 1;
}

int PolyhedralMeshReader::ReadData()
{
	size_t ndoubles = m_numVariables * m_numNodes;

	size_t ndoubles_read;
  double *coords = new double[ndoubles];
	int res = read(ndoubles,coords,m_data_res,ndoubles_read);

  m_coord.reserve(m_numNodes);
  for ( size_t n=0; n < m_numNodes; ++n)
    m_coord.push_back(MbVec3d(coords[n],coords[n+m_numNodes],coords[n+2*m_numNodes]));

  size_t num = m_numVariables-3;
  m_scalars.reserve(m_numNodes*(num));
  for(size_t d=0; d < num; ++d)
    for ( size_t n=0; n < m_numNodes; ++n)
      m_scalars.push_back(coords[n+num*m_numNodes]);

  delete[] coords;
	return res;
}

int PolyhedralMeshReader::ReadNumNodeFace()
{
	size_t ndata_read;
  m_numNodePerFaceList = new size_t[m_numFaces+1];

	int res = read(m_numFaces,m_numNodePerFaceList+1,m_NumFaces_res,ndata_read);
  
  m_numNodePerFaceList[0] = 0;
  for ( size_t n=1; n <= m_numFaces; ++n)
    m_numNodePerFaceList[n] +=  m_numNodePerFaceList[n-1];
	
  return res;
}

int PolyhedralMeshReader::ReadFaceNodes()
{
	size_t ndata_read;
  m_faceNodeList = new size_t[m_totalNumFaceNodes];

	int res = read(m_totalNumFaceNodes,m_faceNodeList,m_NumFaces_res,ndata_read);
  
  for ( size_t n=0; n < m_totalNumFaceNodes; ++n)
    m_faceNodeList[n]--;
	
  return res;
}

int PolyhedralMeshReader::ReadElementFaces()
{
  int* buffer = new int[m_numFaces];
	size_t ndata_read;

	int res = read(m_numFaces,buffer,m_ElementFaces_res,ndata_read);
	
  for ( size_t n=0; n < m_numFaces; ++n)
    if ( buffer[n] > 0)
      m_elementFaces[(size_t)buffer[n]-1].push_back(n);

  delete[] buffer;
  return res;
}

MbVolumeMeshUnstructured<MbVec3d,double,MbVec3d>& 
PolyhedralMeshReader::readMesh(const std::string& filename) 
{
  fp = SbFileHelper::open( filename.c_str() , "r" );
  if (fp != NULL)
  {
  m_LineNumber = 0;

  m_Spec_res = 0;
  m_data_res = 0;
  m_NumFaces_res = 0;
  m_ElementFaces_res = 0;
  m_Actnum_res = 0;

  clear();

  int zRead = 0;
  while (!feof(fp) && zRead <= 0) {
    getLine();
    m_LineNumber++;
    if (CompareKeyword("VARIABLES =")) {
      ReadHeader();
    }
    if (CompareKeyword("ZONE")) {
      zRead = ReadZone();
    }
  }
  }
  m_mesh = new MbVolumeMeshPolyhedron<>(m_coord.begin(),m_coord.end(),m_faceNodeList,m_faceNodeList+m_totalNumFaceNodes,
                                        m_numNodePerFaceList,m_numNodePerFaceList+m_numFaces+1,m_elementFaces,m_numElements);

  // Create data array: the value mapped is a dummy function of I and K
  size_t num = m_numVariables-3;
  m_scalarSets.reserve(num);
  for ( vector<double>::iterator it = m_scalars.begin(); it < m_scalars.end(); it+=m_numNodes)
  {
    m_scalarSets.push_back(new MbScalarSetI<double>(it,it+m_numNodes,"MyScalarSet",MiDataSet::PER_NODE));
    m_mesh->addScalarSet(m_scalarSets.back());
  }

  m_vec3Set = new MbVec3SetI<MbVec3d>(m_coord.begin(),m_coord.end(),"MyGeometry",MiDataSet::PER_NODE);
  m_mesh->addVec3Set(m_vec3Set);

  fclose(fp);

  delete m_numNodePerFaceList;
  m_numNodePerFaceList = NULL;
  delete m_faceNodeList;
  m_faceNodeList = NULL;
  delete [] m_elementFaces;
  m_elementFaces = NULL;
  m_coord.clear();
  m_scalars.clear();


  return *m_mesh;
}	

#ifdef _WIN32
#  pragma warning( pop )
#endif
