/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MBTETRAHEDRONCELL_H
#define _MBTETRAHEDRONCELL_H

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4250)
#endif

#include <MeshVizXLM/extractors/MxTetrahedronCellExtract.h>
#include <MeshVizXLM/mesh/cell/MiVolumeCell.h>
#include <MbMeshVizImpl.h>

class PiMarchingCase;

/**
* @DTEXT Build a tetrahedron cell.
* 
* @ingroup MeshVizXLM_Implement_Cell
* 
* @DESCRIPTION
*    A tetrahedron cell has 4 faces which can be in anywhere in
*    the 3D space. Each face is a triangle.
* 
*    Facets and nodes are numbered as following :
* \verbatim
                        n2

                       /|\
                      / | \
                     /  |  \
                    /   |   \                  the face n0,n1,n2
                   /    |    \                 is oriented towards
                  /     |     \                the interior of the cell
              n0 x------|------x n1
                  \     |     /
                   \    |    /
                    \   |   /
                     \  |  /
                      \ | /
                       \|/
                        x

                        n3
  \endverbatim
*/
class MbTetrahedronCell : virtual public MiVolumeCell
{
public:

  /** 
  * Construct a tetrahedron.
  */
  MbTetrahedronCell(size_t nodeId0, size_t nodeId1, size_t nodeId2, size_t nodeId3);

  /**
  * Gets the volume of the cell. 
  */
  double getVolume(const MiGeometryI* meshGeometry) const
  {
    return MxTetrahedronCellExtract::getVolume(*meshGeometry,this);
  }

  /**
  * Gets the number of facets.
  */
  virtual size_t getNumFacets() const { return 4; }

  /**
  * Gets the number of edges.
  */
  virtual size_t getNumEdges() const { return 6; }

  /**
  * Gets the number of nodes.
  */
  virtual size_t getNumNodes() const { return 4; }

  virtual size_t getNodeIndex(size_t nod) const { return m_nodeIds[nod]; }

  virtual size_t appendNodesIndexOfFacet (size_t facet, std::vector<size_t>& facetNodes) const;
  
  virtual void getIsosurfTopology(unsigned char caseId, std::vector< std::pair<size_t,size_t> >& edgeList ) const
  {
    MxTetrahedronCellExtract::getIsosurfTopology(caseId, edgeList);
  }

  virtual unsigned char getMarchingCaseId(std::vector<bool> &nodesSign, size_t beginNodeId=0) const;

  //Reimplemented from MiCell 
  virtual MbVec3d getIsoParametricCoord(size_t indexNode) const
  {
    return MxTetrahedronCellExtract::getIsoParametricCoord(indexNode);
  }

  virtual double getRelativeSize(const MiGeometryI* meshGeometry) const 
  {
    return MxTetrahedronCellExtract::getLongestEdgeLength(*meshGeometry,this);
  }

  //Reimplemented from MiCell 
  virtual bool isPointInsideCell(const MiGeometryI& meshGeometry, const MbVec3d &point, std::vector<double>& weights) const 
  {
    return MxTetrahedronCellExtract::isPointInsideCell(meshGeometry,this,point,weights);
  }

protected:
  /**
  * Defines a specialized output on stream for MbTetrahedronCell
  */
  virtual std::ostream& toStream(std::ostream& s) const;

  size_t m_nodeIds[4];
  static size_t s_facetNode[4][3];
};

//-----------------------------------------------------------------------------
inline
MbTetrahedronCell::MbTetrahedronCell(size_t nodeId0, size_t nodeId1, size_t nodeId2, size_t nodeId3)
{
  m_nodeIds[0] = nodeId0;
  m_nodeIds[1] = nodeId1;
  m_nodeIds[2] = nodeId2;
  m_nodeIds[3] = nodeId3;
}

//-----------------------------------------------------------------------------
inline size_t 
MbTetrahedronCell::appendNodesIndexOfFacet (size_t facet, std::vector<size_t>& facetNodes) const
{
  facetNodes.push_back(m_nodeIds[s_facetNode[facet][0]]);
  facetNodes.push_back(m_nodeIds[s_facetNode[facet][1]]);
  facetNodes.push_back(m_nodeIds[s_facetNode[facet][2]]);
  return 3;
}

//-----------------------------------------------------------------------------
inline std::ostream& 
MbTetrahedronCell::toStream(std::ostream& s) const
{
  s << "MbTetrahedronCell (" << 
    m_nodeIds[0] << "," << 
    m_nodeIds[1] << "," << 
    m_nodeIds[2] << "," << 
    m_nodeIds[3] << 
    ")";
  return s;
}

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif


