/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _MBMESHIJK_H
#define  _MBMESHIJK_H

#ifdef _WIN32
#  pragma warning( push )
#  pragma warning(disable:4250)
#endif

#include <MeshVizXLM/mesh/data/MiDataSetIjk.h>
#include <MeshVizXLM/mesh/MiMeshIjk.h>

#include <MbMeshVizImpl.h>
#include <topology/MbTopologyIjk.h>
#include <data/MbVec3SetIjk.h>
#include <data/MbScalarSetIjk.h>
#include <data/MbStringSetIjk.h>
#include <MbDataSetComparator.h>

#include <Inventor/STL/set>
#include <Inventor/STL/algorithm>
#include <Inventor/STL/functional>
#include <Inventor/STL/iostream>

/**
 * @DTEXT Base class for all basic mesh implementation.
 * 
 * @ingroup MeshVizXLM_Implement_Mesh
 * 
 * @DESCRIPTION
 * 
 */
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout = MiMeshIjk::LAYOUT_KJI>
class MbMeshIjk : virtual public MiMeshIjk
{
public:
  virtual ~MbMeshIjk(){};

  /**
  * Add the scalar data set to this mesh if a scalar data set with the same name
  * is not already stored in this mesh.
  * The dataset are stored in a alphabetical order according to their name.
  * @return false if the dataset is not added.
  */ 
  bool addScalarSetIjk(const MbScalarSetIjk<_ScalarT,_Layout>* scalarSet);

  /**
  * Add the vector data set to this mesh if a vector data set with the same name
  * is not already stored in this mesh.
  * The dataset are stored in a alphabetical order according to their name.
  * @return false if the dataset is not added.
  */ 
  bool addVec3SetIjk(const MbVec3SetIjk<_Vec3T,_Layout>* vectorSet);

  /**
  * Add the string data set to this mesh if a string data set with the same name
  * is not already stored in this mesh.
  * The dataset are stored in a alphabetical order according to their name.
  * @return false if the dataset is not added.
  */
  bool addStringSetIjk(const MbStringSetIjk<_Layout>* stringSet);

  /**
  * Get a vector dataset according to its alphabetical ordering number.
  * For instance getVec3SetIjk(0) returns the first dataset in the alphabetical order.
  * @return NULL if id >= getNumVec3SetsIjk().
  */
  virtual const MbVec3SetIjk<_Vec3T,_Layout>* getVec3SetIjk(size_t id) const;

  /**
  * Get a vector dataset according to its name.
  * @return NULL if not found.
  */
  virtual const MbVec3SetIjk<_Vec3T,_Layout>* getVec3SetIjk(std::string name) const;

  /**
  * Get a scalar dataset according to its alphabetical ordering number.
  * For instance getScalarSetIjk(0) returns the first dataset in the alphabetical order.
  * @return NULL if id >= getNumScalarSetsIjk().
  */
  virtual const MbScalarSetIjk<_ScalarT,_Layout>* getScalarSetIjk(size_t id) const;

  /**
  * Get a scalar dataset according to its name.
  * @return NULL if not found.
  */
  virtual const MbScalarSetIjk<_ScalarT,_Layout>* getScalarSetIjk(std::string name) const;

  /**
  * Get a string dataset according to its alphabetical ordering number.
  * For instance getStringSetIjk(0) returns the first dataset in the alphabetical order.
  * @return NULL if id >= getNumStringSetsIjk().
  */
  virtual const MbStringSetIjk<_Layout>* getStringSetIjk(size_t id) const;

  /**
  * Get a string dataset according to its name.
  * @return NULL if not found.
  */
  virtual const MbStringSetIjk<_Layout>* getStringSetIjk(std::string name) const;

  //@{
  /** Get the size of data sets */
  virtual size_t getNumVec3SetsIjk() const;
  virtual size_t getNumScalarSetsIjk() const;
  virtual size_t getNumStringSetsIjk() const;
  //@}

protected:
  virtual std::ostream& toStream(std::ostream& s) const;

  typedef std::set< const MbVec3SetIjk<_Vec3T,_Layout>*,  MbDataSetComparator<const MbVec3SetIjk<_Vec3T,_Layout>* > > SetOfVec3Set;
  typedef std::set< const MbScalarSetIjk<_ScalarT,_Layout>*, MbDataSetComparator<const MbScalarSetIjk<_ScalarT,_Layout>* > > SetOfScalarSet;
  typedef std::set< const MbStringSetIjk<_Layout>*, MbDataSetComparator<const MbStringSetIjk<_Layout>* > > SetOfStringSet;

  // MbMeshIjk is abstract
  MbMeshIjk();

  SetOfScalarSet m_scalarSets;
  SetOfVec3Set m_vec3Sets;
  SetOfStringSet m_stringSets;
};

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
bool 
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::addScalarSetIjk(const MbScalarSetIjk<_ScalarT,_Layout>* scalarSet)
{
  std::pair<typename SetOfScalarSet::iterator,bool> ret = m_scalarSets.insert(scalarSet);
  return ret.second;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
bool 
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::addVec3SetIjk(const MbVec3SetIjk<_Vec3T,_Layout>* vectorSet)
{
  std::pair<typename SetOfVec3Set::iterator,bool> ret = m_vec3Sets.insert(vectorSet);
  return ret.second;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
bool
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::addStringSetIjk(const MbStringSetIjk<_Layout>* stringSet)
{
  std::pair<typename SetOfStringSet::iterator,bool> ret = m_stringSets.insert(stringSet);
  return ret.second;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::MbMeshIjk()
{
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbVec3SetIjk<_Vec3T,_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getVec3SetIjk(size_t id) const
{
  typename SetOfVec3Set::const_iterator it=m_vec3Sets.begin();
  for (size_t i=0; i<id && it!=m_vec3Sets.end(); ++i,++it);
  if ( it!=m_vec3Sets.end() ) 
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbVec3SetIjk<_Vec3T,_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getVec3SetIjk(std::string name) const
{
  MbVec3SetIjk<_Vec3T,_Layout> tmp;
  tmp.setName(name);
  typename SetOfVec3Set::const_iterator it;
  it = m_vec3Sets.find(&tmp);
  if ( it!=m_vec3Sets.end() ) 
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbScalarSetIjk<_ScalarT,_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getScalarSetIjk(size_t id) const
{
  typename SetOfScalarSet::const_iterator it=m_scalarSets.begin();
  for (size_t i=0; i<id && it!=m_scalarSets.end(); ++i,++it);
  if ( it!=m_scalarSets.end() ) 
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbScalarSetIjk<_ScalarT,_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getScalarSetIjk(std::string name) const
{
  MbScalarSetIjk<_ScalarT,_Layout> tmp;
  tmp.setName(name);
  typename SetOfScalarSet::const_iterator it;
  it =  m_scalarSets.find(&tmp);
  if ( it!=m_scalarSets.end() )
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbStringSetIjk<_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getStringSetIjk(size_t id) const
{
  typename SetOfStringSet::const_iterator it=m_stringSets.begin();
  for (size_t i=0; i<id && it!=m_stringSets.end(); ++i,++it);
  if ( it!=m_stringSets.end() )
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
const MbStringSetIjk<_Layout>*
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getStringSetIjk(std::string name) const
{
  MbStringSetIjk<_Layout> tmp(0, 0, 0);
  tmp.setName(name);
  typename SetOfStringSet::const_iterator it;
  it =  m_stringSets.find(&tmp);
  if ( it!=m_stringSets.end() )
    return *it;
  else
    return NULL;
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
size_t 
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getNumVec3SetsIjk() const
{
  return m_vec3Sets.size();
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
size_t 
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getNumScalarSetsIjk() const
{
  return m_scalarSets.size();
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
size_t
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::getNumStringSetsIjk() const
{
  return m_stringSets.size();
}

//-----------------------------------------------------------------------------
template <typename _ScalarT, typename _Vec3T, MiMeshIjk::StorageLayout _Layout>
inline std::ostream& 
MbMeshIjk<_ScalarT,_Vec3T,_Layout>::toStream(std::ostream& s) const
{
  size_t i;
  // output the scalar set
  s << "# num scalar sets" << std::endl;
  s << getNumScalarSetsIjk() << std::endl;
  for ( i=0; i<getNumScalarSetsIjk(); ++i )
  {
    const MiScalardSetIjk* dataSet = getScalarSetIjk(i);
    s << "# scalar set : " << i << std::endl;
    s << *(dataSet);
  }

  s << "# num vector sets" << std::endl;
  s << getNumVec3SetsIjk() << std::endl;
  for ( i=0; i<getNumVec3SetsIjk(); ++i )
  {
    const MiVec3dSetIjk* dataSet = getVec3SetIjk(i);
    s << "# vector set : " << i << std::endl;
    s << *(dataSet);
  }

  return s;
}

#ifdef _WIN32
#  pragma warning( pop )
#endif

#endif 







