/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MbHexahedronTopologyExplicitIjk_H
#define _MbHexahedronTopologyExplicitIjk_H

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4250)
#pragma warning(disable:4251)
#endif

#include <MeshVizXLM/mesh/topology/MiHexahedronTopologyExplicitIjk.h>
#include <topology/MbTopologyIjk.h>
#include <MbError.h>

#include <Inventor/SbTime.h>
#include <Inventor/STL/limits>

//-----------------------------------------------------------------------------
/**
* @DTEXT List of volume cells' indices.
* 
* @ingroup MeshVizXLM_Implement_Topology
* 
* @DESCRIPTION
*    A topology for a unstructured explicit hexahedron mesh Ijk. 
*
*    Hexahedrons are defines as follows:
* \verbatim
                         J
                         |
                         |

                        n3----------n2   facet 0 = 0123
                        /|          /|   facet 1 = 4765
                      /  |        /  |   facet 2 = 0374
                    /    |      /    |   facet 3 = 1562
                  n7---------n6      |   facet 4 = 0451
                   |     |    |      |   facet 5 = 3267
                   |    n0----|-----n1    --- I
                   |    /     |     /
                   |  /       |   /
                   |/         | /
                  n4---------n5

                /
              /
            K
  \endverbatim
*/
template <MiMeshIjk::StorageLayout layout = MiMeshIjk::LAYOUT_KJI>
class MbHexahedronTopologyExplicitIjk : public MiHexahedronTopologyExplicitIjk, public MbTopologyIjk<layout>
{
public:
  /**
  * Constructor of an empty topology defined by its size in i,j,k
  */
  MbHexahedronTopologyExplicitIjk(size_t numCellI, size_t numCellJ, size_t numCellK);
  MbHexahedronTopologyExplicitIjk(const MbHexahedronTopologyExplicitIjk& topology);

#ifdef MBMESH_DONT_USE_STL
  virtual ~MbHexahedronTopologyExplicitIjk();
#endif


  /**
  * Returns 8 indices of a cell of this topology. The cell 
  * is identified by three id i,j,k.
  */
  virtual void getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                size_t& n4, size_t& n5, size_t& n6, size_t& n7) const;
  const size_t* getCellNodeIndicesBuffer(size_t i, size_t j, size_t k) const;

  virtual MiMeshIjk::StorageLayout getStorageLayout() const { return layout; }

  size_t getBeginNodeId() const;

  /**
  * Returns the max node id +1 used by this topology. This topology used only 
  * node index in the interval [beginNodeId,EndNodeId[. Thus the maximum node
  * index used by the topology is getEndNodeId() -1 and the number of 
  * nodes used by this topology is getEndNodeId()-getBeginNodeId().
  */
  size_t getEndNodeId() const;

    /**
  * Returns the number of cells on the first logical axis.
  */
  size_t getNumCellsI() const { return this->m_numI; }

  /**
  * Returns the number of cells on the second logical axis.
  */
  size_t getNumCellsJ() const { return this->m_numJ; }

  /**
  * Returns the number of cells on the third logical axis.
  */
  size_t getNumCellsK() const { return this->m_numK; }

  /** 
  * Returns true if the cell of index (i,j,k) should be ignored.
  * This value is ignored if the hasDeadCell() method returns false.
  * The default implementation returns always false.
  * @param (i,j,k) the index of the cell to check
  */ 
  bool isDead(size_t i, size_t j, size_t k) const 
  {
    return MbTopologyIjk<layout>::isDead(i,j,k);
  }

  /**
  * Set the nodes indices of cell (i,j,k)
  */
  virtual void setCellNodeIndices(size_t i, size_t j, size_t k, 
                              const size_t* beginId, const size_t* endId);

  /**
  * Returns the number of cells
  */
  size_t getNumCells() const;

protected:
  /**
  * Defines a specialized output on stream 
  */
  virtual std::ostream& toStream(std::ostream& s) const;

#ifndef MBMESH_DONT_USE_STL
  std::vector<size_t> m_indices;
#else
  size_t* m_indices;
#endif

  size_t m_beginNodeId;
  size_t m_endNodeId;

#ifdef MBMESH_DATA_ACCESS_MICROSLEEP
public:
   static size_t s_microsleep;
#endif
};

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline
MbHexahedronTopologyExplicitIjk<layout>::MbHexahedronTopologyExplicitIjk(size_t numCellI, size_t numCellJ, size_t numCellK)
: MiHexahedronTopologyExplicitIjk()
, MbTopologyIjk<layout>(numCellI,numCellJ,numCellK)
, m_beginNodeId(0)
, m_endNodeId(0)
{
  m_beginNodeId = std::numeric_limits<size_t>::max();
#ifndef MBMESH_DONT_USE_STL
  m_indices.resize(getNumCells()*8);
#else
  m_indices = new size_t[getNumCells()*8];
#endif
}

//-----------------------------------------------------------------------------
#ifdef MBMESH_DONT_USE_STL
template <MiMeshIjk::StorageLayout layout>
inline
MbHexahedronTopologyExplicitIjk<layout>::~MbHexahedronTopologyExplicitIjk()
{
  delete[] m_indices;
}
#endif

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
#ifndef MBMESH_DONT_USE_STL
inline
MbHexahedronTopologyExplicitIjk<layout>::MbHexahedronTopologyExplicitIjk(const MbHexahedronTopologyExplicitIjk& topology)
: MiTopology(), MiHexahedronTopologyExplicitIjk()
, MbTopologyIjk<layout>(topology)
, m_indices(topology.m_indices)
, m_beginNodeId(topology.m_beginNodeId)
, m_endNodeId(topology.m_endNodeId)
{
}
#else
inline
MbHexahedronTopologyExplicitIjk<layout>::MbHexahedronTopologyExplicitIjk(const MbHexahedronTopologyExplicitIjk& topology)
: MiTopology(), MiHexahedronTopologyExplicitIjk()
, MbTopologyIjk<>(topology)
, m_beginNodeId(topology.m_beginNodeId)
, m_endNodeId(topology.m_endNodeId)
{
  m_indices = new size_t[m_endNodeId];
  for (size_t i=m_beginNodeId; i!=m_endNodeId; ++i)
    m_indices[i] = topology.m_indices[i];
}
#endif



//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline size_t
MbHexahedronTopologyExplicitIjk<layout>::getNumCells() const 
{
  return this->m_numK*this->m_numI*this->m_numJ;
}

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline 
void 
MbHexahedronTopologyExplicitIjk<layout>::getCellNodeIndices(size_t i, size_t j, size_t k, size_t& n0, size_t& n1, size_t& n2, size_t& n3,
                                                                                  size_t& n4, size_t& n5, size_t& n6, size_t& n7) const
{
  const size_t *it = this->getCellNodeIndicesBuffer(i,j,k);
  n0 = *it++;
  n1 = *it++;
  n2 = *it++;
  n3 = *it++;
  n4 = *it++;
  n5 = *it++;
  n6 = *it++;
  n7 = *it;
}

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline const size_t*
MbHexahedronTopologyExplicitIjk<layout>::getCellNodeIndicesBuffer(size_t i, size_t j, size_t k) const
{
#ifndef MBMESH_DONT_USE_ASSERT
  if (hasDeadCells() && isDead(i,j,k)) 
    throw MbError("accessing dead cells");
#endif

#ifndef MBMESH_DONT_USE_STL
  const size_t *it = &m_indices.front();
#else
  const size_t *it = m_indices;
#endif
  it += this->getI(i,j,k) * 8;

#ifdef MBMESH_DATA_ACCESS_MICROSLEEP
  if (s_microsleep>0)
    SbTime::usleep(s_microsleep);
#endif

  return it;
}


//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline size_t
MbHexahedronTopologyExplicitIjk<layout>::getBeginNodeId() const
{
  if (m_endNodeId == 0) 
    return 0;
  else
    return m_beginNodeId;
}

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline size_t
MbHexahedronTopologyExplicitIjk<layout>::getEndNodeId() const
{
  return m_endNodeId;
}

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline void
MbHexahedronTopologyExplicitIjk<layout>::setCellNodeIndices(size_t i, size_t j, size_t k, const size_t* beginId, const size_t* endId)
{
#ifndef MBMESH_DONT_USE_ASSERT
  assert( std::distance(beginId,endId) == 8 );
#endif

#ifndef MBMESH_DONT_USE_STL
  std::vector<size_t>::iterator index = m_indices.begin() + (this->getI(i,j,k) * 8);
#else
  size_t* index= m_indices + (this->getI(i,j,k) * 8);
#endif

  size_t nodeId;
  for (const size_t* it = beginId; it != endId; ++it,++index)
  {
    nodeId = *it;
    if (nodeId >= m_endNodeId) 
      m_endNodeId = nodeId + 1;
    if (nodeId < m_beginNodeId) 
      m_beginNodeId = nodeId;
    *index = nodeId;
  }
  this->m_timeStamp = MxTimeStamp::getTimeStamp();
}

//-----------------------------------------------------------------------------
template <MiMeshIjk::StorageLayout layout>
inline std::ostream&
MbHexahedronTopologyExplicitIjk<layout>::toStream(std::ostream& s) const
{
  s << "#Hexahedron topology explicitIjk " << std::endl;
  s << "# num cells" << std::endl;
  s << getNumCells() << std::endl;

  for(size_t i=0; i<this->m_numI; i++)
    for(size_t j=0; j<this->m_numJ; j++)
      for(size_t k=0; k<this->m_numK; k++)
      {
        s << "# cell #("<<i<<","<<j<<","<<k<<")";
        if (hasDeadCells() && isDead(i,j,k))
             s <<"(dead)";
        s << std::endl << "( ";
        size_t pos = this->getI(i,j,k) * 8;
        for (size_t n = 0; n < 8; ++n)
          s << m_indices[pos + n] << " ";
        s << ")" << std::endl;
      }
  return s;
}

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif


