/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#ifndef _MbPolyhedronCellsSet_H
#define _MbPolyhedronCellsSet_H

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4250)
#pragma warning(disable:4251)
#endif

#include <MeshVizXLM/mesh/topology/MiVolumeTopologyExplicitI.h>
#include <topology/MbTopologyI.h>
#include <cell/MbPolyhedronCell.h>
#include <MbError.h>

#include <Inventor/STL/algorithm>

/**
* @DTEXT Build a set of Polyhedron.
* 
* @ingroup MeshVizXLM_Implement_Topology
*/
class MbPolyhedronCellsSet : virtual public MiVolumeTopologyExplicitI, public MbTopologyI
{
public:

  /** 
  * Construct an empty set of Polyhedrons.
  */
  MbPolyhedronCellsSet();

  /** 
  * Defines a new list of Polyhedrons according to a list of facet indices,
  * the number of indices per facet and the number of facet per polyhedrons
  */
  template <typename _FacetNodeIter, typename _NumNodeIter, typename _NumFacetIter>
  void assign(_FacetNodeIter beginFacetNode, _FacetNodeIter endFacetNode,
              _NumNodeIter beginNumNode, _NumNodeIter endNumNode,
              _NumFacetIter beginNumFacet, _NumFacetIter endNumFacet);

  /** 
  * Defines a new list of Polyhedrons according to a list of facet indices,
  * the number of indices per facet and the number of facet per polyhedrons
  */
  template <typename _FacetNodeIter, typename _FacetRefIter>
  void assign(_FacetNodeIter beginFacetNode, _FacetNodeIter endFacetNode,
              _FacetRefIter beginFacetRef, _FacetRefIter endFacetRef,
              const std::vector<size_t>* cellFacets, size_t numCells);

  /**
  * Returns the i-th cell of this set.
  */
  virtual const MbPolyhedronCell* getCell(size_t cellId) const;

  /**
  * Returns the number of cells in the set.
  */
  virtual size_t getNumCells() const;

  /**
  * Returns the first node index in this polyhedral geometry.
  */
  virtual size_t getBeginNodeId() const
  {
    return m_beginNodeId;
  }

  /**
  * Returns the end node index in this polyhedral geometry.
  */
  virtual size_t getEndNodeId() const
  {
    return m_endNodeId;
  }

protected:
  std::vector<MbPolyhedronCell> m_cells;
  size_t m_beginNodeId;
  size_t m_endNodeId;
};


//-----------------------------------------------------------------------------
inline
MbPolyhedronCellsSet::MbPolyhedronCellsSet()
: m_beginNodeId(0), m_endNodeId(0)
{
}


//-----------------------------------------------------------------------------
inline const MbPolyhedronCell* 
MbPolyhedronCellsSet::getCell(size_t cellId) const
{
#ifndef MBMESH_DONT_USE_ASSERT
  if (hasDeadCells() && isDead(cellId)) 
    throw MbError("accessing dead cells");
#endif
  return &(m_cells[cellId]);
}

//-----------------------------------------------------------------------------
inline size_t
MbPolyhedronCellsSet::getNumCells() const 
{
  return m_cells.size();
}

//-----------------------------------------------------------------------------
template <typename _FacetNodeIter, typename _NumNodeIter, typename _NumFacetIter>
inline void 
MbPolyhedronCellsSet::assign(_FacetNodeIter beginFacetNode, _FacetNodeIter /*endFacetNode*/,
                             _NumNodeIter beginNumNode, _NumNodeIter /*endNumNode*/,
                             _NumFacetIter beginNumFacet, _NumFacetIter endNumFacet)
{
  size_t numCells = std::distance(beginNumFacet,endNumFacet);
  m_beginNodeId = std::numeric_limits<size_t>::max();

  m_cells.clear();
  m_cells.reserve(numCells);
  _FacetNodeIter beginFacet, itFacetNode = beginFacetNode;
  _NumNodeIter beginNum, itNumNode = beginNumNode;
  for (_NumFacetIter itNum = beginNumFacet; itNum != endNumFacet; ++itNum)
  { 
    beginNum = itNumNode;
    beginFacet = itFacetNode;
    while ( itNumNode != beginNum + *itNum )
    {
      for ( size_t i =0; i < *itNumNode; ++i)
      {
       if ( *itFacetNode + 1 > m_endNodeId) 
         m_endNodeId = *itFacetNode + 1;
        if ( *itFacetNode < m_beginNodeId) 
         m_beginNodeId = *itFacetNode;
        ++itFacetNode;
      }
      ++itNumNode;
    }
    m_cells.push_back(MbPolyhedronCell(beginFacet,itFacetNode,beginNum,itNumNode));    
  }
  m_timeStamp = MxTimeStamp::getTimeStamp();
}

//-----------------------------------------------------------------------------
template <typename _FacetNodeIter, typename _FacetRefIter>inline void 
  MbPolyhedronCellsSet::assign(_FacetNodeIter beginFacetNode, _FacetNodeIter /*endFacetNode*/,
  _FacetRefIter beginFacetRef, _FacetRefIter /*endFacetRef*/,
  const std::vector<size_t>* cellFacets, size_t numCells)
{
  m_cells.clear();
  m_cells.reserve(numCells);
  std::vector<size_t> facetNodes;
  std::vector<size_t> numFacetNodes;
  _FacetNodeIter itFacetNode;
  _FacetRefIter facetRef, lastFacetNode;
  std::vector<size_t>::const_iterator itCellFacets;

  m_beginNodeId = std::numeric_limits<size_t>::max();
  m_endNodeId = 0;

  for (size_t n = 0; n < numCells; ++n)
  { 
    facetNodes.clear();
    numFacetNodes.clear();
    numFacetNodes.reserve(cellFacets[n].size());
    for (itCellFacets = cellFacets[n].begin(); itCellFacets != cellFacets[n].end(); ++itCellFacets)
    {
      facetRef = beginFacetRef + *itCellFacets;
      itFacetNode = beginFacetNode + *facetRef++;
      lastFacetNode = beginFacetNode + *facetRef;
      numFacetNodes.push_back(0);
      while ( itFacetNode != lastFacetNode )
      {
        facetNodes.push_back(*itFacetNode);
        numFacetNodes.back()++;
        if ( *itFacetNode + 1 > m_endNodeId) 
          m_endNodeId = *itFacetNode + 1;
        if ( *itFacetNode  < m_beginNodeId) 
          m_beginNodeId = *itFacetNode;
        ++itFacetNode;
      }
    }
    m_cells.push_back(MbPolyhedronCell(facetNodes.begin(),facetNodes.end(),numFacetNodes.begin(),numFacetNodes.end()));    
  }
  m_timeStamp = MxTimeStamp::getTimeStamp();
}

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif

