#ifndef _PEBITopology_H
#define _PEBITopology_H

#include <MeshVizXLM/mesh/topology/MiVolumeTopologyExplicitI.h>
#include <topology/MbTopologyI.h>
#include <cell/MbPolygonCell.h>
#include <cell/PEBICell.h>

#include <Inventor/STL/vector>
#include <Inventor/STL/algorithm>

class PEBITopology : virtual public MiVolumeTopologyExplicitI, public MbTopologyI
{
public:
  PEBITopology(size_t numLayer);
  ~PEBITopology();

  /**
  * Returns a cell of this topology. 
  */
  const MiVolumeCell* getCell(size_t id) const ;

  void clear();
  void reservePolygonSet(size_t numPolygons);

  template < typename _nodeIter >
  void addPolygonCell(_nodeIter  begin, _nodeIter end);

  /**
  * Returns the max node id +1 used by this topology. This topology used only 
  * node index in the interval [beginNodeId,EndNodeId[. Thus the maximum node
  * index used by the topology is getEndNodeId() -1 and the number of 
  * nodes used by this topology is getEndNodeId()-getBeginNodeId().
  */
  size_t getEndNodeId() const;

  void setEndNodeId(size_t numNodes) { m_numNodes = numNodes; } 

  /**
  * Returns the number of cells.
  */
  size_t getNumCells() const;

  /** 
  * Returns true if the cell of index (i,j,k) should be ignored.
  * This value is ignored if the hasDeadCell() method returns false.
  * The default implementation returns always false.
  * @param (i,j,j) the index of the cell to check
  */ 
  bool isDead(size_t id) const 
  {
    return MbTopologyI::isDead(id);
  }

protected:
  std::ostream& toStream(std::ostream& s) const
  {
    return MiVolumeTopologyExplicitI::toStream(s);
  }

  //list of 2D cells
  std::vector<MbPolygonCell> m_polygonSet;
  size_t m_numLayer;
  size_t m_numNodes;
  size_t m_surfaceNumNodes;

  std::vector<PEBICell*> m_3DCellSet;

};

//-----------------------------------------------------------------------------
inline
PEBITopology::PEBITopology(size_t numLayer)
: m_numLayer(numLayer), m_surfaceNumNodes(0)
{
  PEBICell::setShift(m_surfaceNumNodes);
}

//-----------------------------------------------------------------------------
inline
PEBITopology::~PEBITopology()
{
  for ( size_t i = 0; i < m_3DCellSet.size(); ++i)
    delete m_3DCellSet[i];
}

//-----------------------------------------------------------------------------
inline size_t
PEBITopology::getNumCells() const 
{
  return m_3DCellSet.size();
}

//-----------------------------------------------------------------------------
inline const MiVolumeCell* 
PEBITopology::getCell(size_t id) const
{
  size_t layer = id / m_polygonSet.size();
  size_t polygonId = id % m_polygonSet.size();
  return m_3DCellSet[(polygonId*m_numLayer + layer)];
}

//-----------------------------------------------------------------------------
inline void 
PEBITopology::clear()
{
  m_surfaceNumNodes = 0;
  m_polygonSet.clear();
  for ( size_t i = 0; i < m_3DCellSet.size(); ++i)
    delete m_3DCellSet[i];
  m_3DCellSet.clear();
  m_timeStamp = MxTimeStamp::getTimeStamp();
}

//-----------------------------------------------------------------------------
inline void 
PEBITopology::reservePolygonSet(size_t numPolygons)
{
  m_polygonSet.reserve(numPolygons);
  m_3DCellSet.reserve(numPolygons*m_numLayer);
}

//-----------------------------------------------------------------------------
template < typename _nodeIter >
void PEBITopology::addPolygonCell(_nodeIter  begin, _nodeIter end)
{
  m_polygonSet.push_back(MbPolygonCell(begin,end));
  m_surfaceNumNodes = std::max(m_surfaceNumNodes,(size_t)(*std::max_element(begin,end))+1);
  for ( size_t l = 0; l < m_numLayer; ++l)
    m_3DCellSet.push_back(new PEBICell(m_polygonSet.back(),l));
  m_timeStamp = MxTimeStamp::getTimeStamp();
}

//-----------------------------------------------------------------------------
inline size_t
PEBITopology::getEndNodeId() const
{
  return m_surfaceNumNodes*(m_numLayer+1);
}

#endif


