﻿#include <Engine.h>

#include <iostream>

#if defined(linux) || defined(__linux)
#include <cstring>
#endif

#define CIRCLE_DIAMETER 40

#define GREEN_R 50
#define GREEN_G 120
#define GREEN_B 20

#define RGB_COMPONENT 3

//------------------------------------------------------------------------------
Engine::Engine(unsigned int width, unsigned int height) :
  m_isButton0Down(false),
  m_isTouchDown(false),
  m_mouseX(width / 2),
  m_mouseY(height / 2),
  m_width(width),
  m_height(height)
{ 
  memset(&m_frame, 0, sizeof(m_frame));

  resize(width, height);
}

//------------------------------------------------------------------------------
Engine::~Engine()
{
  if (m_frame.pixels != NULL)
  {
    delete[] m_frame.pixels;
    m_frame.pixels = NULL;
  }
}

//------------------------------------------------------------------------------
void Engine::resize(unsigned int width, unsigned int height)
{
  m_width = width;
  m_height = height;
  m_mouseX = width / 2;
  m_mouseY = height / 2;

  {
    std::lock_guard<std::mutex> lock(m_frameMutex);

    if (m_frame.pixels != NULL)
    {
      delete[] m_frame.pixels;
      m_frame.pixels = NULL;
    }

  // Create a pixel buffer
    m_frame.pixels = new unsigned char[m_width * m_height * RGB_COMPONENT];
  }

  // New render
  display(false);
}

//------------------------------------------------------------------------------
void Engine::display(bool interactiveFrame)
{
  std::lock_guard<std::mutex> lock(m_frameMutex);

  /* Draw a green circle */

  // Reset pixels color to black
  memset(m_frame.pixels, 0, m_width * m_height * RGB_COMPONENT);

  unsigned int stride;
  for (unsigned int i = 0; i < m_height; i++)
  {
    stride = i * m_width;
    for (unsigned int j = 0; j < m_width; j++)
    {
      // Calculate the Euclidean distance between the two points (j,i) and (mouseX,mouseY)
      int distance = (j - m_mouseX) * (j - m_mouseX) + (i - m_mouseY) * (i - m_mouseY);
      if ( distance < CIRCLE_DIAMETER * CIRCLE_DIAMETER )
      {
        m_frame.pixels[(j + stride) * RGB_COMPONENT] = GREEN_R;
        m_frame.pixels[(j + stride) * RGB_COMPONENT + 1] = GREEN_G;
        m_frame.pixels[(j + stride) * RGB_COMPONENT + 2] = GREEN_B;
      }
    }
  }

  m_frame.isInteractive = interactiveFrame;
  m_frame.id++;
}

//------------------------------------------------------------------------------
const Engine::Frame& Engine::getFrame() const
{
  return m_frame;
}

//------------------------------------------------------------------------------
std::mutex& Engine::getFrameMutex() const
{
  return m_frameMutex;
}

//------------------------------------------------------------------------------
bool Engine::mouseUp(int x, int y, SoMouseButtonEvent::Button button)
{
  // Update mouse position
  m_mouseX = x;
  m_mouseY = y;

  // Update left mouse button state
  if ( button == SoMouseButtonEvent::BUTTON1 )
  {
    m_isButton0Down = false;

    // New render
    display(false);
  }

  return true;
}

//------------------------------------------------------------------------------
bool Engine::mouseDown(int x, int y, SoMouseButtonEvent::Button button)
{
  // Update mouse position
  m_mouseX = x;
  m_mouseY = y;

  // Update left mouse button state
  if ( button == SoMouseButtonEvent::BUTTON1 )
  {
    m_isButton0Down = true;

    // New render
    display(true);
  }

  return true;
}

//------------------------------------------------------------------------------
bool Engine::mouseMove(int x, int y)
{
  // Update mouse position
  m_mouseX = x;
  m_mouseY = y;

  if ( m_isButton0Down )
  {
    // New render
    display(true);
  }

  return true;
}
