#include <IndependentRenderAreaListener.h>

#include <RemoteViz/Rendering/RenderArea.h>
#include <RemoteViz/Rendering/Connection.h>

#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/image/SbRasterImage.h>

#include <Engine.h>

//------------------------------------------------------------------------------
IndependentRenderAreaListener::IndependentRenderAreaListener(std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea) :
  m_renderArea(renderArea)
{
  // Instantiate the render engine
  m_engine = new Engine(renderArea->getWidth(), renderArea->getHeight());
}

//------------------------------------------------------------------------------
IndependentRenderAreaListener::~IndependentRenderAreaListener()
{
  delete m_engine;
}

//------------------------------------------------------------------------------
void IndependentRenderAreaListener::onClosedConnection(std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea, const std::string& connectionId, bool aborted)
{
  std::lock_guard<std::mutex> lock(m_listMutex);

  m_lastConnectionFrame.erase(connectionId);

  RenderAreaListener::onClosedConnection(renderArea, connectionId, aborted);
}

//------------------------------------------------------------------------------
void IndependentRenderAreaListener::onRequestedFrame(std::shared_ptr<RemoteViz::Rendering::RenderArea> /*renderArea*/, std::shared_ptr<RemoteViz::Rendering::Connection> sender, SbRasterImage* rasterImage, bool& isInteractive)
{
  std::lock_guard<std::mutex> lockFrame(m_engine->getFrameMutex());

  Engine::Frame frame = m_engine->getFrame();
  isInteractive = frame.isInteractive;

  std::lock_guard<std::mutex> lockList(m_listMutex);

  std::map<std::string, size_t>::iterator it = m_lastConnectionFrame.find(sender->getId());
  if ( it != m_lastConnectionFrame.end() )
  {
    // Check if a new frame is available
    if ( it->second != frame.id )
    {
      it->second = frame.id;
      copyImageBufferToRasterImage(rasterImage, frame.pixels);
    }
  }
  else // new connection
  {
    m_lastConnectionFrame.insert( std::pair<std::string, size_t>(sender->getId(), frame.id) );

    copyImageBufferToRasterImage(rasterImage, frame.pixels);
  }
}

//------------------------------------------------------------------------------
void IndependentRenderAreaListener::copyImageBufferToRasterImage(SbRasterImage* rasterImage, unsigned char* data)
{
  if ( data != NULL )
  {
    // Allocate memory and copy data into buffer
    size_t numBytes = rasterImage->getSize()[0] * rasterImage->getSize()[1] * rasterImage->getComponents();
    SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject();
    cpuBuffer->setSize( numBytes );
    unsigned char* buffer = (unsigned char*)cpuBuffer->map( SoBufferObject::SET );
    memcpy( buffer, data, numBytes );
    cpuBuffer->unmap();
    rasterImage->setBuffer(cpuBuffer.ptr());
  }
}

//------------------------------------------------------------------------------
void IndependentRenderAreaListener::onResize(std::shared_ptr<RemoteViz::Rendering::RenderArea> /*renderArea*/, unsigned int width, unsigned int height)
{
  // Resize engine
  m_engine->resize(width, height);
}

//------------------------------------------------------------------------------
bool IndependentRenderAreaListener::onMouseUp(std::shared_ptr<RemoteViz::Rendering::RenderArea> /*renderArea*/, std::shared_ptr<RemoteViz::Rendering::Connection> /*sender*/, int x, int y, SoMouseButtonEvent::Button button)
{
  // Update engine
  m_engine->mouseUp(x, y , button);

  return true;
}

//------------------------------------------------------------------------------
bool IndependentRenderAreaListener::onMouseDown(std::shared_ptr<RemoteViz::Rendering::RenderArea> /*renderArea*/, std::shared_ptr<RemoteViz::Rendering::Connection> /*sender*/, int x, int y, SoMouseButtonEvent::Button button)
{
  // Update engine
  m_engine->mouseDown(x, y , button);

  return true;
}

//------------------------------------------------------------------------------
bool IndependentRenderAreaListener::onMouseMove(std::shared_ptr<RemoteViz::Rendering::RenderArea> /*renderArea*/, std::shared_ptr<RemoteViz::Rendering::Connection> /*sender*/, int x, int y)
{
  // Update engine
  m_engine->mouseMove(x, y);

  return true;
}
