#include <iostream>
#include "RemoteViz/Rendering/Service.h"
#include "RemoteViz/Rendering/ServiceSettings.h"

#include "InventorApplicationServiceListener.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoMaterial.h>

using namespace RemoteViz::Rendering;

int main(int argc, char** argv)
{
  // Default value for service IP address
  std::string serviceIPaddress = "127.0.0.1";
  // Default value for service port
  unsigned short servicePort = 8080;

  // Instantiate a service settings class
  auto settings = std::make_shared<ServiceSettings>();

  // Use passed values for the service IP address and port
  if (argc == 3)
  {
   serviceIPaddress = argv[1];
   servicePort = atoi(argv[2]);
  }

  // Set the IP address used by the service
  settings->setIP(serviceIPaddress);
  // Set the port used by the service
  settings->setPort(servicePort);

  // set the run mode
  settings->setRunMode(ServiceSettings::INVENTOR_APPLICATION);

  // Instantiate a serviceListener class to manage the service events.
  auto serviceListener = std::make_shared<InventorApplicationServiceListener>();

  // Add the serviceListener class as listener
  Service::instance()->addListener(serviceListener);

  // To use an Open Inventor extension with RemoteViz, uncomment the following line and set the extension name.
  //settings.setUsedExtensions(ServiceSettings::VOLUMEVIZLDM);

  Widget myWindow = SoXt::init("InventorApplication"); // pass the app name
  if (myWindow == NULL) 
    return 1;

  // Make a scene containing a red cone
  SceneExaminer *root = new SceneExaminer;
  SoMaterial *myMaterial = new SoMaterial;
  root->ref();
  myMaterial->diffuseColor.setValue(1.0, 0.0, 0.0);   // Red
  root->addChild(myMaterial);
  root->addChild(new SoCone);

  // Create a renderArea in which to see our scene graph.
  // The render area will appear within the main window.
  SoXtRenderArea *myRenderArea = new SoXtRenderArea(myWindow);

  // Make the camera see everything.
  root->viewAll(myRenderArea->getViewportRegion());

  // Keep the root for onInstantiatedRenderArea callback
  InventorApplicationServiceListener::root = root;

  // Put our scene in myRenderArea, change the title
  myRenderArea->setSceneGraph(root);
  myRenderArea->setTitle("InventorApplication");
  myRenderArea->show();

  // Open the service by using the settings
  if (Service::instance()->open(settings))
  {
    std::cout << "IP : " << settings->getIP() << std::endl;

    std::cout << "Hostname : " << settings->getHostname() << std::endl;

    std::cout << "Port : " << settings->getPort() << std::endl;

    std::cout << "The InventorApplication Rendering Service is running. Press Ctrl + C to stop." << std::endl;

    SoXt::show(myWindow);  // Display main window
    SoXt::mainLoop();      // Main Inventor event loop

    // Close the service
    Service::instance()->close();
  }
  else
  {
    std::cout << "Error occurred during service initialization." << std::endl;
  }

  delete myRenderArea;

  root->unref();

  SoXt::finish();

  return 0;
}

