#pragma once
#include <httplib.h>
#include <json.hpp>
#include <string>
#include <iostream>

class LoadBalancerClient
{
public:

  /**
   *
   *  Register the instance to the load balancer from its IP address and its port number.
   *
   *  \param serviceIPaddress : IP address of the service to register.
   *
   *  \param servicePort : port number of the service to register.
   *
   *  \param[out] instanceID : ID of the registered instance.
   */
  static bool registerInstance(const std::string& serviceIPaddress, unsigned short servicePort, size_t& instanceID)
  {
    httplib::Client cli(s_registrationApiIP, s_registrationApiPort);

    nlohmann::json jsonRequest;
    jsonRequest["ip"] = serviceIPaddress;
    jsonRequest["port"] = servicePort;

    auto res = cli.Post("/instance", jsonRequest.dump(), "application/json");
    if (!res) // connection error
    {
      std::cout << "Unable to establish connection with the load balancer." << std::endl;
      return false;
    }
    else if (res->status == 200) // success
    {
      auto jsonResponse = nlohmann::json::parse(res->body);
      instanceID = jsonResponse["id"];
      return true;
    }
    else // message error
    {
      std::cout << "Unable to register the instance to the load balancer:" << std::endl;
      std::cout << "Error code: " << res->status << std::endl;
      std::cout << "Message: " << res->body << std::endl;
      return false;
    }
  }

  /**
   *
   *  Unregister the instance to the load balancer from its ID.
   *
   *  \param instanceID : ID of the service to unregister.
   */
  static bool unregisterInstance(size_t instanceID)
  {
    httplib::Client cli(s_registrationApiIP, s_registrationApiPort);
    const std::string path = "/instance/" + std::to_string(instanceID);
    auto res = cli.Delete(path.c_str());
    if (!res) // connection error
    {
      std::cout << "Unable to establish connection with the load balancer." << std::endl;
      return false;
    }
    else if (!(res->status == 200)) // message error
    {
      std::cout << "Error when unregistering the instance to the load balancer:" << std::endl;
      std::cout << "Error code: " << res->status << std::endl;
      std::cout << "Message: " << res->body << std::endl;
      return false;
    }

    return true;
  }

  /**
   *
   *  Delete the renderArea to the load balancer from its ID.
   *
   *  \param renderAreaId : ID identifying the renderArea
   */
  static bool deleteRenderArea(const std::string &renderAreaId)
  {
    httplib::Client cli(s_registrationApiIP, s_registrationApiPort);
    const std::string path = "/renderArea/" + renderAreaId;
    auto res = cli.Delete(path.c_str());
    if (!res)
    {
      std::cout << "Unable to establish connection with the load balancer." << std::endl;
      return false;
    }
    else if (!(res->status == 200))
    {
      std::cout << "Error when deleting the renderArea to the load balancer:" << std::endl;
      std::cout << "Error code: " << res->status << std::endl;
      std::cout << "Message: " << res->body << std::endl;
      return false;
    }

    return true;
  }

private:
  static constexpr char const* s_registrationApiIP = "127.0.0.1";
  static constexpr unsigned short s_registrationApiPort = 8081;
};
