#include <LoadBalancerClient.h>
#include <LoadBalancedServiceListener.h>
#include <LoadBalancedRenderAreaListener.h>
#include <RemoteViz/Rendering/RenderArea.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/touch/SoTouchManager.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoGradientBackground.h>

#include <Inventor/ViewerComponents/nodes/SceneExaminer.h>

LoadBalancedServiceListener::LoadBalancedServiceListener(std::shared_ptr<const RemoteViz::Rendering::ServiceSettings> settings)
{
  const std::string ip = settings->getIP();
  const std::string port = std::to_string( settings->getPort() );

  // Generate a random instance color, using a seed created with a hash of the "ip:port" string.
  const std::string key = ip + ":" + port;
  srand( (unsigned int)std::hash<std::string>{}(key) );
  const unsigned int random = rand();
  const unsigned short r = random & 0xF;
  const unsigned short g = (random >> 4) & 0xF;
  const unsigned short b = (random >> 8) & 0xF;
  m_color = SbColor( r / 15.f, g / 15.f, b / 15.f );
}

void LoadBalancedServiceListener::onInstantiatedRenderArea(std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea)
{
  // Instantiate a renderAreaListener class to manage the renderArea events (default behaviors).
  std::shared_ptr<RemoteViz::Rendering::RenderAreaListener> renderAreaListener(new LoadBalancedRenderAreaListener());

  // Add the renderAreaListener instance as renderArea listener
  renderArea->addListener(renderAreaListener);

  // Instantiate a sceneExaminer to interact with the camera
  SceneExaminer *examiner = new SceneExaminer();

  // Add recognizers for gesture events
  renderArea->getTouchManager()->addDefaultRecognizers();

  // Scene graph
  SoRef<SoMaterial> material = new SoMaterial;
  material->diffuseColor.setValue( m_color );
  examiner->addChild( material.ptr() );
  examiner->addChild(new SoCone());
  examiner->addChild(new SoGradientBackground());

  // Apply the sceneExaminer node as renderArea scene graph
  renderArea->getSceneManager()->setSceneGraph(examiner);

  // viewall
  examiner->viewAll(renderArea->getSceneManager()->getViewportRegion());
}

void LoadBalancedServiceListener::onDisposedRenderArea(const std::string &renderAreaId)
{
  LoadBalancerClient::deleteRenderArea(renderAreaId);
}
