#include <LoadBalancerClient.h>
#include <iostream>
#include <chrono>
#include <thread>
#include <atomic>
#include <csignal>
#include <RemoteViz/Rendering/Service.h>
#include <RemoteViz/Rendering/ServiceSettings.h>

#include <LoadBalancedServiceListener.h>

using namespace RemoteViz::Rendering;

std::atomic<bool> running;

void sighandler(int /*sig*/)
{
  running = false;
}

int main(int argc, char** argv)
{
  // Default value for service IP address
  std::string serviceIPaddress = "127.0.0.1";
  // Default value for service port
  unsigned short servicePort = 8080;

  // Instantiate a service settings class
  auto settings = std::make_shared<ServiceSettings>();

  // Use passed values for the service IP address and port
  if (argc == 3)
  {
   serviceIPaddress = argv[1];
   servicePort = atoi(argv[2]);
  }

  // Set the IP address used by the service
  settings->setIP(serviceIPaddress);
  // Set the port used by the service
  settings->setPort(servicePort);

  // Instantiate a serviceListener class to manage the service events.
  std::shared_ptr<LoadBalancedServiceListener> serviceListener(new LoadBalancedServiceListener(settings));
  // Add the serviceListener class as listener
  Service::instance()->addListener(serviceListener);

  // To use an Open Inventor extension with RemoteViz, uncomment the following line and set the extension name.
  //settings.setUsedExtensions(ServiceSettings::VOLUMEVIZLDM);

  // Open the service by using the settings
  if (Service::instance()->open(settings))
  {
    std::cout << "IP : " << settings->getIP() << std::endl;

    std::cout << "Hostname : " << settings->getHostname() << std::endl;

    std::cout << "Port : " << settings->getPort() << std::endl;

    std::cout << "The Load Balanced Rendering Service is running. Press Ctrl + C to stop." << std::endl;

    running = true;

    // Register the signal handler to stop the main loop
    std::signal(SIGABRT, &sighandler);
    std::signal(SIGTERM, &sighandler);
    std::signal(SIGINT, &sighandler);
    #ifdef SIGBREAK
      std::signal(SIGBREAK, &sighandler);
    #endif

    // Register the instance to the load balancer.
    size_t instanceID = 0;
    if (!LoadBalancerClient::registerInstance(Service::instance()->getSettings()->getIP(),
                                              Service::instance()->getSettings()->getPort(),
                                              instanceID))
    {
      // Close the service
      Service::instance()->close();
      return 1;
    }

    // Main loop
    while (running)
    {
      Service::instance()->dispatch();
      std::this_thread::sleep_for(std::chrono::milliseconds(1));
    }

    // Unregister the instance to the load balancer.
    LoadBalancerClient::unregisterInstance(instanceID);

    // Close the service
    Service::instance()->close();
  }
  else
  {
    std::cout << "Error occurred during service initialization." << std::endl;
  }

  return 0;
}

