var http = require('http'),
    service = require('./loadbalancer-service'),
    express = require('express'),
    bodyParser = require('body-parser'),
    log4js = require('log4js');

class LoadBalancer {

    /**
     * Constructor
     */
    constructor() {
        // Create http server and proxy HTTP requests.
        this.proxyServer = http.createServer(function (req, res) {
            service.proxyWebServer(req, res);
        });

        // Listen to the `upgrade` event and proxy the WebSocket requests.
        this.proxyServer.on('upgrade', function (req, socket, head) {
            service.proxyWebSocket(req, socket, head);
        });

        // Emitted when the server closes.
        this.proxyServer.on('close', function () {
            service.close();
        });

        // Create a HTTP server to handle REST requests and manage instance registration.
        this.registrationapi = express()

        // parse application/json
        this.registrationapi.use(bodyParser.json())

        this.registrationapi.post('/instance', service.registerInstance);

        this.registrationapi.delete('/instance/:id', service.unregisterInstance);

        this.registrationapi.delete('/renderArea/:id', service.deleteRenderArea);
    }

    /**
     * Starts the load balancer listening for connections.
     */
    listen() {
        // Load configuration file that stores IP address and port number.
        const config = require('./config/config.json');
        const loadBalancerConfig = config.loadbalancer;
        const registrationApiConfig = config.registrationapi;
        const logger = log4js.getLogger();
        logger.level = 'debug';

        this.proxyServer.listen(loadBalancerConfig.port, loadBalancerConfig.ip, () => {
            logger.info(`LoadBalancer running at http://${loadBalancerConfig.ip}:${loadBalancerConfig.port}/`);
        });

        this.registrationapi.listen(registrationApiConfig.port, registrationApiConfig.ip, () => {
            logger.info(`Registration API running at http://${registrationApiConfig.ip}:${registrationApiConfig.port}/`);
        });
    }
}

module.exports = LoadBalancer;
