#pragma once

#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <RemoteViz/Rendering/MetricsListener.h>

class CSVMetricsListener : public RemoteViz::Rendering::MetricsListener
{
public:

  /**
  * Constructor
  */
  CSVMetricsListener();

  /**
  * Save metrics in csv file.
  */
  void saveMetrics();

  /**
  *  Triggered when a new measure of network latency is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NETWORK_LATENCY.
  *
  *  \param time : network latency in milliseconds
  *
  *  \param client : The network latency is related to this client.
  */
  virtual void onMeasuredNetworkLatency(unsigned int time, std::shared_ptr<RemoteViz::Rendering::Client> client) override;

  /**
  *  Triggered when a new measure of frame decoding time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#DECODING_TIME.
  *
  *  \param time : frame decoding time in milliseconds
  *
  *  \param connection : The frame decoding time is related to this connection.
  */
  virtual void onMeasuredDecodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection ) override;

  /**
  *  Triggered when a new measure of frame rendering time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#RENDERING_TIME.
  *
  *  \param time : frame rendering time in milliseconds
  *
  *  \param renderArea : The frame rendering time is related to this render area.
  */
  virtual void onMeasuredRenderingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea ) override;

  /**
  *  Triggered when a new measure of frame encoding time is available.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#ENCODING_TIME.
  *
  *  \param time : frame encoding time in milliseconds
  *
  *  \param connection : The frame encoding time is related to this connection.
  */
  virtual void onMeasuredEncodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection ) override;

  /**
  *  Triggered when the number of clients changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CLIENTS.
  *
  *  \param number : number of clients
  */
  virtual void onChangedNumClients( unsigned int number ) override;

  /**
  *  Triggered when the number of connections changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_CONNECTIONS.
  *
  *  \param number : number of connections
  */
  virtual void onChangedNumConnections( unsigned int number ) override;

  /**
  *  Triggered when the number of renderArea changes.
  *  The metric must be enabled using Monitoring#enableMetrics with the value Monitoring#NUM_RENDERAREAS.
  *
  *  \param number : number of render areas
  */
  virtual void onChangedNumRenderAreas( unsigned int number ) override;

private:
  // write pairs of index-values in cvs file
  void writeInColumn(std::map<int, std::string> values);

  // string buffer
  std::stringstream sb;
  // join vector items
  std::string join(const std::vector<std::string>& vec, const char* separator);
  // get index of an item in headers
  int getIndexInHeaders(std::string value);
  // csv file
  std::ofstream outputFile;
  // csv separator
  const char* csvSeparator = ";";
  // headers
  const std::string timestampCsvField = "timestamp";
  const std::string clientIdCsvField = "client id";
  const std::string connectionIdCsvField = "connection id";
  const std::string renderAreaIdCsvField = "renderarea id";
  const std::string networkLatencyCsvField = "network latency";
  const std::string decodingTimeCsvField = "decoding time";
  const std::string renderingTimeCsvField = "rendering time";
  const std::string encodingTimeCsvField = "encoding time";
  const std::string numClientsCsvField = "number clients";
  const std::string numConnectionsCsvField = "number connections";
  const std::string numRenderAreasCsvField = "number render areas";

  std::vector<std::string> cvsHeaders;
};

