#include <iostream>
#include <iterator>
#include <chrono>
#include <algorithm>
#include <CSVMetricsListener.h>
#include <RemoteViz/Rendering/Client.h>
#include <RemoteViz/Rendering/Connection.h>
#include <RemoteViz/Rendering/RenderArea.h>

//--------------------------------------------------------------------------------
CSVMetricsListener::CSVMetricsListener()
{
  cvsHeaders = {
    timestampCsvField,
    clientIdCsvField,
    connectionIdCsvField,
    renderAreaIdCsvField,
    networkLatencyCsvField,
    decodingTimeCsvField,
    renderingTimeCsvField,
    encodingTimeCsvField,
    numClientsCsvField,
    numConnectionsCsvField,
    numRenderAreasCsvField
  };
  sb << join(cvsHeaders, csvSeparator) << "\n";

}

//--------------------------------------------------------------------------------
void CSVMetricsListener::saveMetrics()
{
  outputFile.open("monitoring.csv");
  if (outputFile.is_open())
  {
    outputFile << sb.str();
    outputFile.close();
  }
}

//--------------------------------------------------------------------------------
std::string CSVMetricsListener::join(const std::vector<std::string>& vec, const char* separator)
{
  std::stringstream res;
  copy(vec.begin(), vec.end(), std::ostream_iterator<std::string>(res, separator));
  return res.str();
}

//--------------------------------------------------------------------------------
int CSVMetricsListener::getIndexInHeaders(std::string value)
{
  auto it = std::find(cvsHeaders.begin(), cvsHeaders.end(), value);
  return (int)std::distance(cvsHeaders.begin(), it);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::writeInColumn(std::map<int, std::string> values)
{
  std::vector<std::string> line(cvsHeaders.size());
  line[0] = std::to_string(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count());
  for (const auto& value : values)
  {
    line[value.first] = value.second;
  }
  sb << join(line, csvSeparator) << "\n";
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onMeasuredNetworkLatency( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Client> client )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(clientIdCsvField), client->getId() },
    { getIndexInHeaders(networkLatencyCsvField), std::to_string(time) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onMeasuredDecodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(connectionIdCsvField), connection->getId() },
    { getIndexInHeaders(decodingTimeCsvField), std::to_string(time) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onMeasuredRenderingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::RenderArea> renderArea )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(renderAreaIdCsvField), renderArea->getId() },
    { getIndexInHeaders(renderingTimeCsvField), std::to_string(time) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onMeasuredEncodingTime( unsigned int time, std::shared_ptr<RemoteViz::Rendering::Connection> connection )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(connectionIdCsvField), connection->getId() },
    { getIndexInHeaders(encodingTimeCsvField), std::to_string(time) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onChangedNumClients( unsigned int number )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(numClientsCsvField), std::to_string(number) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onChangedNumConnections( unsigned int number )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(numConnectionsCsvField), std::to_string(number) }
  };
  writeInColumn(values);
}

//--------------------------------------------------------------------------------
void CSVMetricsListener::onChangedNumRenderAreas( unsigned int number )
{
  const std::map<int, std::string> values = {
    { getIndexInHeaders(numRenderAreasCsvField), std::to_string(number) }
  };
  writeInColumn(values);
}