/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Julien Sallanne (Jun 2011)
**=======================================================================*/

#include "utils.h"
#include "SoTransferFunction2D.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtColorWheel.h>

#include <Inventor/manips/SoTabBoxManip.h>

#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoRenderToTextureProperty.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoImageBackground.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoFaceSet.h>

#include <Inventor/nodes/SoLightModel.h>

#include <LDM/nodes/SoMultiDataSeparator.h>
#include <LDM/nodes/SoDataRange.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Inventor/drawers/SoPolygonScreenDrawer.h>
#include <Inventor/drawers/SoLassoScreenDrawer.h>

#include <Inventor/engines/SoCalculator.h>
#include <Inventor/actions/SoHandleEventAction.h>

static SoTopLevelDialog* pTopLevelDialog0 = NULL;
static SoMaterial*       pColor           = NULL;
static SoSeparator*      pRoot            = NULL;
static SoSeparator*      pToRender        = NULL;
static SoSeparator*      pRenderRoot      = NULL;
static SoRenderToTextureProperty* renderToTexProperty = NULL;

// Comment out the following line in order to use a second dataset for 2DTransferFunction.
//#define USE_DATA2

SoShape* createFrom2DPoints ( const std::vector<SbVec2f>& pointsInCam )
{
  std::vector<SbVec3f> polygon( pointsInCam.size() );
  for ( unsigned int i = 0; i < pointsInCam.size(); ++i )
  {
    polygon[i].setValue( pointsInCam[i][0], pointsInCam[i][1], 0.5f );
  }

  SoVertexProperty* polygonVertices = new SoVertexProperty();
  polygonVertices->vertex.setValues(0, (int)polygon.size(), &polygon[0]);

  SoFaceSet* polygonFaceSet = new SoFaceSet();
  polygonFaceSet->vertexProperty = polygonVertices;
  return polygonFaceSet;
}

void colFinalizedCallback(void *, const float hsv[3])
{
  SbColor color;
  color.setHSVValue(hsv);
  pColor->diffuseColor = color;
}


// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line, in camera space, and add a shape thanks to this points.
void lineFinalizedCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less that 2 points, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 3 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
  {
    lineInCam[i][0] = lineDrawer->point[i][0];
    lineInCam[i][1] = lineDrawer->point[i][1];
  }

  // Shape is put in a switch to easily enable/disable it thanks to a check box.
  SoSwitch* switchNode = new SoSwitch();

  SoSeparator *shapesRoot = searchName<SoSeparator>(pRoot, "shapesRoot");

  SoSeparator *manipSeparator = new SoSeparator;

  shapesRoot->addChild(manipSeparator);
  manipSeparator->addChild(switchNode);

  // Add colorEditor values into Material
  SoMaterial *material = new SoMaterial;
  material->diffuseColor = pColor->diffuseColor;
  material->transparency = pColor->transparency;

  // Add a manipulator
  SoTabBoxManip* manip = new SoTabBoxManip;

  // create checkbox
  SoDialogCheckBox* checkBox = new SoDialogCheckBox();
  checkBox->label = shapesRoot->getNumChildren()-1;
  checkBox->state = TRUE;
  checkBox->fixedWidth = TRUE;
  checkBox->fixedHeight = TRUE;
  SoColumnDialog *columnDialog = (SoColumnDialog *)pTopLevelDialog0->searchForAuditorId("columnDialogShape");
  columnDialog->addChild(checkBox);

  // Link checkbox to switch:
  // We use an engine to convert from TRUE/FALSE to SO_SWITCH_NONE/SO_SWITCH_ALL
  // TRUE/FALSE are respectively 0/1 and SO_SWITCH_NONE/SO_SWITCH_ALL are respectively -3/-1.
  // Using operation "oa = -2*a - 1", if a = 0, oa = -1 and if a = 1, oa = -3.
  // Not very clean but avoid defining a custom engine.
  SoCalculator* boolToInt = new SoCalculator();
  boolToInt->a.connectFrom(&checkBox->state);
  boolToInt->expression = "oa = -2*a - 1";
  switchNode->whichChild.connectFrom( &boolToInt->oa );

  switchNode->addChild(manip);
  switchNode->addChild(material);
  // add shape
  SoShape* shape = createFrom2DPoints(lineInCam);
  switchNode->addChild( shape );

  SoSeparator *shapesRoot_RenderToTex = searchName<SoSeparator>(pToRender, "shapesRoot_RenderToTex");

  SoSwitch* switchNode2 = new SoSwitch();

  SoSeparator *transformSeparator = new SoSeparator;

  shapesRoot_RenderToTex->addChild(transformSeparator);
  transformSeparator->addChild(switchNode2);

  // Link checkbox to switch:
  switchNode2->whichChild.connectFrom( &switchNode->whichChild );

  SoTransform* transform = new SoTransform;
  transform->translation.connectFrom( &manip->translation );
  transform->rotation.connectFrom( &manip->rotation );
  transform->scaleFactor.connectFrom( &manip->scaleFactor );
  transform->scaleOrientation.connectFrom( &manip->scaleOrientation );
  transform->center.connectFrom( &manip->center );

  switchNode2->addChild(material);
  switchNode2->addChild(transform);
  switchNode2->addChild( shape );

  // don't forget to clear line
  lineDrawer->clear();
  action->setHandled();
}


class auditorClass : public SoDialogAuditor
{
public:
private:
  void dialogRealSlider(SoDialogRealSlider* cpt)
  {
    float value = cpt->value.getValue();

    if (cpt->auditorID.getValue() == "transparency")
    {
      pColor->transparency.set1Value(0, value);
    }
  }
};

Widget
buildInterface(Widget window)
{
  pTopLevelDialog0 = (SoTopLevelDialog*) SoDialogViz::loadFromFile("$OIVHOME/examples/source/VolumeViz/2DTransferFunction/interface.iv");
  SoDialogCustom* customNode0 = nullptr;

  if (pTopLevelDialog0) {
    pTopLevelDialog0->buildDialog( window, TRUE );
    auditorClass* auditor0 = new auditorClass;
    pTopLevelDialog0->addAuditor(auditor0);
    customNode0 = (SoDialogCustom*)pTopLevelDialog0->searchForAuditorId(SbString("viewer0"));
    pTopLevelDialog0->label.setValue("TransferFunction2D");
    pTopLevelDialog0->show();
  }

  return customNode0 != nullptr ? customNode0->getWidget() : window;
}


SoVolumeData::LDMDataAccess::DataInfoBox getData(SoVolumeData* volData, SoCpuBufferObject *cpuBufferObject)
{
  SbVec3i32 dim  = volData->data.getSize();
  SbBox3f   volSize = volData->extent.getValue();
  SbBox3i32 box(SbVec3i32(0,0,0), SbVec3i32(dim[0]-1,dim[1]-1,dim[2]-1));

  // Call with NULL buffer (default 3rd parameter) to get number of bytes required to hold data
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  SoBufferObject* dummyPtr = NULL;
  info = volData->getLdmDataAccess().getData( 0, box, dummyPtr );
  // Fetch the data contained in the ROI in the original volume

  if (info.bufferSize) 
  {
    cpuBufferObject->setSize((size_t)info.bufferSize);
    info = volData->getLdmDataAccess().getData( 0, box, cpuBufferObject );
    return info;
  }
  return info;
}

void send2DHistogram(SoImageBackground *texHistogram)
{
  // Compute 2D Histogram
  SoRef<SoCpuBufferObject> cpuBufferObject1 = new SoCpuBufferObject();
  SoVolumeData::LDMDataAccess::DataInfoBox info1 = getData(searchName<SoVolumeData>(pRenderRoot, "volData1"), cpuBufferObject1.ptr());
  void *values1 = cpuBufferObject1->map(SoCpuBufferObject::READ_ONLY);
  const unsigned char* slicePtr1 = static_cast<unsigned char*>(values1);

#ifdef USE_DATA2
  SoRef<SoCpuBufferObject> cpuBufferObject2 = new SoCpuBufferObject();
  SoVolumeData::LDMDataAccess::DataInfoBox info2 = getData(searchName<SoVolumeData>(pRenderRoot, "volData2"), cpuBufferObject2.ptr());
  void *values2 = cpuBufferObject2->map(SoCpuBufferObject::READ_ONLY);
  const unsigned char* slicePtr2 = static_cast<unsigned char*>(values2);
#endif

  // 2DHistogram is only available for UNSIGNED_BYTE dataset
  if ( info1.errorFlag == SoVolumeData::LDMDataAccess::CORRECT && searchName<SoVolumeData>(pRenderRoot, "volData1")->getDataType() == SoDataSet::UNSIGNED_BYTE )
  {
    unsigned char histo[256][256];
    unsigned char value1, value2;
    memset(*histo, 0, sizeof(histo));
    for(int k=0; k<info1.bufferDimension[2]; k++)
    {
      for(int j=0; j<info1.bufferDimension[1]; j++)
      {
        for(int i=0; i<info1.bufferDimension[0]; i++)
        {
          int index = i+j*info1.bufferDimension[0]+k*info1.bufferDimension[0]*info1.bufferDimension[1];
          value1 = slicePtr1[index];
#ifdef USE_DATA2
          value2 = slicePtr2[index];
#else
          // Compute gradient into CPU with centralDiff
          int neighborsIndex[6];
          neighborsIndex[0] = (i==info1.bufferDimension[0]-1)? index : index + 1;
          neighborsIndex[1] = (i==0)? index : index - 1;

          neighborsIndex[2] = (j==info1.bufferDimension[1]-1)?
            i+(j)  *info1.bufferDimension[0]+k*info1.bufferDimension[0]*info1.bufferDimension[1] :
          i+(j+1)*info1.bufferDimension[0]+k*info1.bufferDimension[0]*info1.bufferDimension[1];

          neighborsIndex[3] = (j==0)?
            i+(j)  *info1.bufferDimension[0]+k*info1.bufferDimension[0]*info1.bufferDimension[1] :
          i+(j-1)*info1.bufferDimension[0]+k*info1.bufferDimension[0]*info1.bufferDimension[1];

          neighborsIndex[4] = (k==info1.bufferDimension[2]-1)?
            i+j*info1.bufferDimension[0]+(k)  *info1.bufferDimension[0]*info1.bufferDimension[1] :
          i+j*info1.bufferDimension[0]+(k+1)*info1.bufferDimension[0]*info1.bufferDimension[1];

          neighborsIndex[5] = (k==0)?
            i+j*info1.bufferDimension[0]+(k)  *info1.bufferDimension[0]*info1.bufferDimension[1] :
          i+j*info1.bufferDimension[0]+(k-1)*info1.bufferDimension[0]*info1.bufferDimension[1];

          SbVec3f gradient;
          gradient[0] = float(slicePtr1[neighborsIndex[0]]-slicePtr1[neighborsIndex[1]]);
          gradient[1] = float(slicePtr1[neighborsIndex[2]]-slicePtr1[neighborsIndex[3]]);
          gradient[2] = float(slicePtr1[neighborsIndex[4]]-slicePtr1[neighborsIndex[5]]);
          gradient *= 255.f/sqrt(3.f*255*255);
          value2 = (unsigned char)(sqrt( double((gradient[0]*gradient[0])+ (gradient[1]*gradient[1]) + (gradient[2]*gradient[2])) ));
#endif
          histo[value2][value1]++;
        }
      }
    }
    texHistogram->image.setValue(SbVec2i32(256,256), 1, SoSFImage::UNSIGNED_BYTE, histo);
    texHistogram->style = SoImageBackground::STRETCH;
  }

  cpuBufferObject1->unmap();
#ifdef USE_DATA2
  cpuBufferObject2->unmap();
#endif
}


int
main(int /*argc*/, char **argv)
{
  Widget mainWindow = SoXt::init(argv[0]);
  SoDialogViz::init();
  SoVolumeRendering::init();

  SoTransferFunction2D::initClass();

  SoOrthographicCamera *orthoCam = new SoOrthographicCamera;
  orthoCam->nearDistance.setValue(0.01f);

  SoSeparator* shapesRoot = new SoSeparator;
  shapesRoot->setName("shapesRoot");

  // Shape hint use to generate shapes normals.
  SoShapeHints* shapeHint = new SoShapeHints();
  shapesRoot->addChild(shapeHint);
  // must be less than PI/2: because of perspective projection, angle between body and top/bottom can be up to PI/2 - camFov/2
  shapeHint->creaseAngle = (float)(0.3 * M_PI);
  shapeHint->windingType = SoShapeHints::NO_WINDING_TYPE;
  shapeHint->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeHint->shapeType = SoShapeHints::SOLID;
  shapeHint->faceType = SoShapeHints::CONVEX;

  //------------------------------------------------------------------------------
  // Add line drawers
  SoSwitch* lineDrawerSwitch = new SoSwitch();
  lineDrawerSwitch->setName("LINE_DRAWER_SWITCH");
  lineDrawerSwitch->whichChild = 0;

  // lasso
  SoPolyLineScreenDrawer* lasso = new SoLassoScreenDrawer();
  lasso->setName("LASSO_SCREEN_DRAWER");
  lasso->onFinish.add( lineFinalizedCallback );
  lasso->simplificationThreshold = 3;
  lineDrawerSwitch->addChild(lasso);

  // polygon
  SoPolyLineScreenDrawer* polygon = new SoPolygonScreenDrawer;
  polygon->setName("POLYGON_SCREEN_DRAWER");
  polygon->onFinish.add( lineFinalizedCallback );
  lineDrawerSwitch->addChild(polygon);

  SoLightModel *lightModel = new SoLightModel;
  lightModel->model = SoLightModel::BASE_COLOR;

  SoSeparator* materialSep = new SoSeparator;
  pColor = new SoMaterial;
  materialSep->addChild(pColor);

  SoSeparator *sep = new SoSeparator;
  sep->addChild( new SoPerspectiveCamera );
  sep->addChild( orthoCam );
  sep->addChild(shapesRoot);
  sep->addChild(lineDrawerSwitch);
  sep->addChild(lightModel);
  sep->addChild(materialSep);

  SoSeparator* sep2 = new SoSeparator;

  sep2->addChild( new SoPerspectiveCamera );
  sep2->addChild( orthoCam );
  SoSeparator* shapesRoot2 = new SoSeparator;
  shapesRoot2->setName("shapesRoot_RenderToTex");
  shapesRoot2->addChild(shapeHint);
  sep2->addChild(shapesRoot2);
  sep2->addChild(lineDrawerSwitch);
  sep2->addChild(lightModel);
  sep2->addChild(materialSep);

  pRoot = new SoSeparator;
  pRoot->ref();
  pRoot->addChild(sep);

  pToRender = new SoSeparator;
  pToRender->ref();
  pToRender->addChild(sep2);

  // Set up viewer
  Widget parent = buildInterface(mainWindow);

  SoImageBackground *texHistogram = new SoImageBackground ();
  pRoot->addChild(texHistogram);

  SoDialogRealSlider *slider;
  slider = (SoDialogRealSlider *)pTopLevelDialog0->searchForAuditorId("transparency");
  pColor->transparency.set1Value(0, slider->getValue());

  // connect combobox to lineDrawer switch
  SoDialogComboBox *modeCombo;
  modeCombo = (SoDialogComboBox *)pTopLevelDialog0->searchForAuditorId("selectionMode");
  lineDrawerSwitch->whichChild.connectFrom(&modeCombo->selectedItem);

  // Set up viewer 1:
  SoXtExaminerViewer *transferFunction2DViewer = new SoXtExaminerViewer(parent);
  transferFunction2DViewer->setDecoration(FALSE);
  transferFunction2DViewer->setViewing(FALSE);
  transferFunction2DViewer->setSceneGraph(pRoot);
  transferFunction2DViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  transferFunction2DViewer->show();
  SoXt::show(mainWindow);

  SoXtExaminerViewer *transferFunction2DViewer2 = new SoXtExaminerViewer(parent);
  transferFunction2DViewer2->setDecoration(FALSE);
  transferFunction2DViewer2->setViewing(FALSE);
  transferFunction2DViewer2->setSceneGraph(pToRender);
  transferFunction2DViewer2->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);

  SoTransferFunction2D* tex2D = new SoTransferFunction2D;
  tex2D->setName("transferFunction2D");

  renderToTexProperty = new SoRenderToTextureProperty;
  renderToTexProperty->ref();
  tex2D->renderToTextureProperty.setValue(renderToTexProperty);

  renderToTexProperty->component = SoRenderToTextureProperty::RGB_TRANSPARENCY;
  renderToTexProperty->size.setValue(SbVec2s(256, 256));
  renderToTexProperty->updatePolicy = SoRenderToTextureProperty::WHEN_NEEDED;
  renderToTexProperty->backgroundColor.setValue(transferFunction2DViewer->getBackgroundColor());
  renderToTexProperty->setName("renderToTexProperty");

  renderToTexProperty->node.set1Value(0, (SoNode*) pToRender);

  /////////////////////////////////////////////////////////////
  // Data 1
  SoVolumeData* pVolData1 = new SoVolumeData;
  pVolData1->fileName     = "$OIVHOME/examples/data/VolumeViz/tooth.ldm";
  pVolData1->setName("volData1");

#ifdef USE_DATA2
  /////////////////////////////////////////////////////////////
  // Data 2
  SoVolumeData* pVolData2 = new SoVolumeData();
  pVolData2->fileName     = "$OIVHOME/examples/data/VolumeViz/tooth_gradient.ldm";
  pVolData2->dataSetId = 2;
  pVolData2->setName("volData2");
#endif

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
  pVolRender->numSlices = 400;

  tex2D->internalFormat.setValue(SoTransferFunction2D::RGBA_FORMAT);
  tex2D->minFilter = SoTransferFunction2D::NEAREST;
  tex2D->magFilter = SoTransferFunction2D::NEAREST;

  // Initialize and set the volume shader program
  SoVolumeRenderingQuality* pVolShader = new SoVolumeRenderingQuality;

  // Specify a fragment shader to combine the intensity and color values
  // First load the fragment shader code
  SoFragmentShader* fragmentShader = new SoFragmentShader;
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/VolumeViz/2DTransferFunction/customShader.glsl" );

  // Set the shader parameters.
  fragmentShader->addShaderParameter1i( "tex2D", SoPreferences::getInt("IVVR_TF2D_TEX_UNIT", 15) ); // for now force to 15
  fragmentShader->addShaderParameter1i( "data1", 1 );

  pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
  pVolShader->lighting = TRUE;

#ifdef USE_DATA2
  fragmentShader->addShaderParameter1i( "data2", 2 );
  fragmentShader->setDefine("USE_DATA2", 1);
#endif

  // Assemble the scene graph
  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  mds->addChild(tex2D);
  mds->addChild( pVolShader );
  mds->addChild( pVolData1 );

#ifdef USE_DATA2
  mds->addChild( pVolData2 );
#endif
  mds->addChild( pVolRender );

  pRenderRoot = new SoSeparator;
  pRenderRoot->ref();

  pRenderRoot->addChild( mds );

  // Set up viewer 2:
  SoDialogCustom *customNode1 = (SoDialogCustom*)pTopLevelDialog0->searchForAuditorId(SbString("viewer1"));
  SoXtExaminerViewer *renderViewer = new SoXtExaminerViewer(customNode1->getWidget());
  renderViewer->setSceneGraph(pRenderRoot);
  renderViewer->show();

  SoDialogCustom *customNode2 = (SoDialogCustom*)pTopLevelDialog0->searchForAuditorId(SbString("colorEditor"));
  SoXtColorWheel *colorWheel = new SoXtColorWheel(customNode2->getWidget());
  colorWheel->addFinishCallback(colFinalizedCallback);
  colorWheel->show();
  send2DHistogram(texHistogram);

  SoXt::mainLoop();

  renderToTexProperty->unref();
  pRenderRoot->unref();
  pRoot->unref();
  pToRender->unref();

  SoTransferFunction2D::exitClass();

  delete renderViewer;
  delete transferFunction2DViewer;
  delete transferFunction2DViewer2;

  // release modules
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


