/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Jan 2011)
**=======================================================================*/

/*******************************************************************************
* This example show how to combine SoScreenDrawer, SoCSGShape and SoVolumeClippingGroup
* to isolate region of interest of a volume data.
*
* The loaded scene can contains :
* - VOLUME_DATA_ROOT: Group representing volume data. If not specified, use camera near and far clip plan.
* - CSG_ROOT: Group that will contains result of CSG operation. Can be a VolumeClippingGroup.
* If not defined, CSG operation is added in main root node.
* - SHOW_SWITCH: Switch containing objects to show, for example a volume data or the result of CSGShape.
*******************************************************************************/

#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <DialogViz/SoDialogVizAll.h>
#include <Inventor/actions/SoSearchPathAction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/engines/SoCalculator.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoCube.h>


#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/drawers/SoLassoScreenDrawer.h>
#include <Inventor/drawers/SoEllipseScreenDrawer.h>
#include <Inventor/drawers/SoRectangleScreenDrawer.h>
#include <Inventor/drawers/SoPolygonScreenDrawer.h>

#include <Inventor/nodes/SoCSGShape.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>

#include "utils.h"


#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/CSGClippingGroup/volumeDataScene.iv"
//#define DEFAULT_SCENE "$OIVHOME/examples/source/VolumeViz/CSGClippingGroup/simpleScene.iv"

// root of scene graph.
SoSeparator* g_root;

// Node containing volume data. It will be used to retrieve boundingBox and 
// extrusion planes. Correspond to VOLUME_DATA_ROOT, or g_root if not specified.
SoGroup* g_volumeDataRoot;

// Root of CSGTree. Correspond to CSG_ROOT node, or g_root if CSG_ROOT not specified.
SoGroup* g_CSGShapeRoot;

// Switch to enable/disable VolumeClippingGroup node
SoSwitch* g_volumeClippingSwitch;

// Radio button to select current csg operation.
SoDialogRadioButtons* g_csgRadio;


//------------------------------------------------------------------------------
// Callback called when their are not anough pass to properly display VolumeClippingGroup.
void notEnoughPassCB( SoVolumeClippingGroup* /*mgr*/, void* /*userData*/ )
{
  std::cout << "not enough pass" << std::endl;
}


//------------------------------------------------------------------------------
// Auditor of "Show LDM Topology" checkBox
class ShowTopologyAuditor : public SoDialogCheckBoxAuditor 
{
  virtual void dialogCheckBox( SoDialogCheckBox * cpt )
  {
    SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, cpt->state.getValue());
  }
};

//------------------------------------------------------------------------------
// Add a new SoCSGShape which right child is specified shape. 
// By adding extruded shapes, we will build the following CSGTree:
//               S
//            S__|__
//         S__|__   |
//      R__|__   |  N4
//    __|__   |  N3
//   |     |  N2
//  NULL   N1
//
// Where S are SoCSGShape, N1, N2, ... 
// The first added is N1, the last is N4.
//
// The last added shape must be in top-right of tree to respect non-associativity 
// and non-commutativity of CSG Operations.
// For example, the following operations:
// N1 + N2 - N3, correspond to ((N1 + N2) - N3), which tree is (R represent RIGHT_ONLY):
//
//            -
//         +__|__
//      R__|__   |
//    __|__   |  N3
//   |     |  N2
//  NULL   N1
void
addShape(SoGroup* CSGTreeRoot, SoNode* newShape, SoCSGShape::CsgOperationType operation )
{
  SoCSGShape* firstCSGShape = find<SoCSGShape>(CSGTreeRoot);
  if ( firstCSGShape == NULL )
  {
    firstCSGShape = new SoCSGShape;
    firstCSGShape->rightOperand = newShape;
    firstCSGShape->csgOperation = SoCSGShape::RIGHT_ONLY;
    CSGTreeRoot->addChild( firstCSGShape );
  }
  else
  {
    SoCSGShape* newCSGShape = new SoCSGShape;
    newCSGShape->leftOperand = firstCSGShape;
    newCSGShape->csgOperation = operation;
    newCSGShape->rightOperand = newShape;
    CSGTreeRoot->replaceChild( firstCSGShape, newCSGShape );
  }
}

//------------------------------------------------------------------------------
// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line in camera space, current view volume, current model matrix
// and add an extruded shape to g_CSGShapeRoot.
void
lineFinalizedCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less than 1 point, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 1 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
    lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

  //------------------------------------------------------------------------------
  // create a new extruded shape :

  // retrieve path of extruded shape root
  SoSearchAction searchAction;
  searchAction.setNode( g_CSGShapeRoot );
  searchAction.apply( g_root );
  SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

  // retrieve bbox of volumeData node.
  SbBox3f bbox = getBbox( g_volumeDataRoot );

  // create an extruded shape from specified line. Line is extruded along view 
  // direction between bounding box enclosing planes.
  SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam, 
                                                                      pathToExtrudedShapeRoot, 
                                                                      SoViewVolumeElement::get(action->getState()), 
                                                                      bbox );
  if ( extrudedShape == NULL )
  {
    lineDrawer->clear();
    return;
  }

  // Clip extrudedShape with volumeData bbox.
  SoSeparator* bboxSep = createCube(bbox);
  SoCSGShape* extrudedShapeClipped = new SoCSGShape;
  extrudedShapeClipped->leftOperand = extrudedShape;
  extrudedShapeClipped->rightOperand = bboxSep;
  extrudedShapeClipped->csgOperation = SoCSGShape::INTERSECTION;


  // if it's the first time a CSGShape is added to g_CSGShapeRoot,
  // Initialize CSGTree with the box representing volumeData.
  if ( find<SoCSGShape>(g_CSGShapeRoot, SbString(), true) == NULL )
  {
    SoCSGShape* initialBox = new SoCSGShape;
    initialBox->rightOperand = bboxSep;
    initialBox->csgOperation = SoCSGShape::RIGHT_ONLY;
    g_CSGShapeRoot->addChild(initialBox);

    // and enable VolumeClippingGroup
    g_volumeClippingSwitch->whichChild = SO_SWITCH_ALL;
  }

  // Add this clippedShape to current CSGTree
  switch ( g_csgRadio->selectedItem.getValue() )
  {
  case 0:
    addShape( g_CSGShapeRoot, extrudedShapeClipped, SoCSGShape::INTERSECTION );
    break;
  case 1:
    addShape( g_CSGShapeRoot, extrudedShapeClipped, SoCSGShape::ADD );
    break;
  case 2:
    addShape( g_CSGShapeRoot, extrudedShapeClipped, SoCSGShape::SUB );
    break;
  }


  // don't forget to clear line
  lineDrawer->clear();

  action->setHandled();
}

 
//------------------------------------------------------------------------------
class ClearButtonAuditor : public SoDialogPushButtonAuditor
{
  virtual void dialogPushButton( SoDialogPushButton* )
  {
    g_CSGShapeRoot->removeAllChildren();
    g_volumeClippingSwitch->whichChild = SO_SWITCH_NONE;
  }
};


int
main(int argc, char **argv)
{

  SoPreferences::setValue("OIV_BUFFER_REGION_ENABLE", "0");

  Widget myWindow = SoXt::init(argv[0]); 
  if (myWindow == NULL)
    exit(1);

  // initialize some classes
  SoDialogViz::init();
  SoVolumeRendering::init();

  SoCSGShape::setRescueOperation( SoCSGShape::LEFT_ONLY );

  SoSeparator* viewerRoot = new SoSeparator;
  viewerRoot->ref();

  // Manually create a camera to view the scene:
  // because SoScreenDrawer contains an orthographic camera to draw line, 
  // SoXtExaminerViewer will not insert a camera in top of sceneGraph.
  SoCamera* camera = new SoPerspectiveCamera;
  viewerRoot->addChild( camera );


  //------------------------------------------------------------------------------
  // load scene

  SbString filename;
  if ( argc == 1 )
    filename = DEFAULT_SCENE;
  else
    filename = SbString(argv[1]);

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    std::cerr << "Cannot open " << filename.getString() << std::endl;
    exit(1);
  }

  g_root = SoDB::readAll(&myInput);
  if (g_root == NULL) {
    std::cerr << "Cannot create sceneGraph." << std::endl;
    exit(1);
  }
  viewerRoot->addChild( g_root );
  g_root->setName("ROOT");

  //------------------------------------------------------------------------------
  // retrieve volume data root to compute its bounding box

  g_volumeDataRoot = find<SoGroup>( g_root, "VOLUME_DATA_ROOT" );
  // If cannot find VOLUME_DATA_ROOT, use root.
  if ( g_volumeDataRoot == NULL )
    g_volumeDataRoot = g_root;
  g_volumeDataRoot->ref();

  //------------------------------------------------------------------------------
  // retrieve VolumeClippingGroup switch

  g_volumeClippingSwitch = find<SoSwitch>( g_root, "VOLUME_CLIPPING_ROOT" );

  //------------------------------------------------------------------------------
  // retrieve extruded shapes root

  g_CSGShapeRoot = find<SoGroup>( g_root, "CSG_ROOT" );
  // If cannot find CSG_ROOT, use root.
  if ( g_CSGShapeRoot == NULL )
    g_CSGShapeRoot = g_root;
  g_CSGShapeRoot->ref();

  //------------------------------------------------------------------------------
  // Add line drawers
  SoSeparator* drawers = new SoSeparator;
  drawers->fastEditing = SoSeparator::CLEAR_ZBUFFER;
  drawers->boundingBoxIgnoring = TRUE;
  drawers->setName("DRAWERS");
  SoSwitch* lineDrawerSwitch = new SoSwitch();
  lineDrawerSwitch->setName("LINE_DRAWER_SWITCH");
  lineDrawerSwitch->whichChild = 0;
  drawers->addChild(lineDrawerSwitch);
  g_root->addChild(drawers);

  // lasso
  SoLassoScreenDrawer* lasso = new SoLassoScreenDrawer();
  lasso->color.setValue(1, 0, 0);
  lasso->setName("LASSO_SCREEN_DRAWER");
  lasso->onFinish.add(lineFinalizedCallback);
  lasso->isClosed = TRUE;
  lineDrawerSwitch->addChild(lasso);

  // polygon
  SoPolygonScreenDrawer* polygon = new SoPolygonScreenDrawer;
  polygon->color.setValue(0, 1, 0);
  polygon->setName("POLYGON_SCREEN_DRAWER");
  polygon->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(polygon);

  // ellipse corner
  SoEllipseScreenDrawer* ellipseCorner = new SoEllipseScreenDrawer;
  ellipseCorner->color.setValue(0, 0, 1);
  ellipseCorner->method = SoEllipseScreenDrawer::CORNER_CORNER;
  ellipseCorner->setName("ELLIPSE_CORNER_SCREEN_DRAWER");
  ellipseCorner->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(ellipseCorner);

  // ellipse center
  SoEllipseScreenDrawer* ellipseCenter = new SoEllipseScreenDrawer;
  ellipseCenter->color.setValue(1, 1, 0);
  ellipseCenter->method = SoEllipseScreenDrawer::CENTER_CORNER;
  ellipseCenter->setName("ELLIPSE_CENTER_SCREEN_DRAWER");
  ellipseCenter->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(ellipseCenter);

  // Rectangle corner
  SoRectangleScreenDrawer* rectangleCorner = new SoRectangleScreenDrawer;
  rectangleCorner->color.setValue(1, 1, 1);
  rectangleCorner->method = SoRectangleScreenDrawer::CORNER_CORNER;
  rectangleCorner->setName("RECTANGLE_CORNER_SCREEN_DRAWER");
  rectangleCorner->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(rectangleCorner);

  // Rectangle center
  SoRectangleScreenDrawer* rectangleCenter = new SoRectangleScreenDrawer;
  rectangleCenter->color.setValue(0.5, 1, 0);
  rectangleCenter->method = SoRectangleScreenDrawer::CENTER_CORNER;
  rectangleCenter->setName("RECTANGLE_CENTER_SCREEN_DRAWER");
  rectangleCenter->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(rectangleCenter);

  //------------------------------------------------------------------------------
  // Build dialog.

  const SbString InterfaceName = "$OIVHOME/examples/source/VolumeViz/CSGClippingGroup/interface.iv";
  SoInput myInputInterface;
  if ( !myInputInterface.openFile(InterfaceName) ) 
  {
    std::cerr << "Cannot open " << InterfaceName.getString() << std::endl;
    exit(1);
  }

  SoGroup *myGroup = SoDB::readAll( &myInputInterface );
  if ( !myGroup )
  {  
    fprintf(stderr, "Cannot create interface.");
    exit(1);
  }
  myGroup->ref();

  SoTopLevelDialog* myTop = find<SoTopLevelDialog>( myGroup );
  myTop->ref();
  myTop->buildDialog( myWindow, true );

  SoDialogCustom* myCustom = find<SoDialogCustom>( myTop, "DIALOG_CUSTOM" );


  // combo box to select lineDrawer
  SoDialogComboBox* modeRadio = find<SoDialogComboBox>( myTop, "MODE_RADIO" );
  lineDrawerSwitch->whichChild.connectFrom(&modeRadio->selectedItem);

  // combo box to select csgOperation
  g_csgRadio = find<SoDialogRadioButtons>( myTop, "CSG_RADIO" );

  // CheckBox to clip inside/outside of csg group (only if there is a volume clipping group)
  SoDialogCheckBox* clipOutside = find<SoDialogCheckBox>( myTop, "CLIP_OUTSIDE_CHECK" );
  SoVolumeClippingGroup* volumeClippingGroup = find<SoVolumeClippingGroup>(g_root);
  if ( volumeClippingGroup != NULL )
  {
    volumeClippingGroup->setNotEnoughPassCallback(notEnoughPassCB, NULL);
    volumeClippingGroup->clipOutside.connectFrom(&clipOutside->state);
  }
  else
    clipOutside->enable = FALSE;

  // CheckBox to show/hide LDMTopology
  SoDialogCheckBox* showTopologyCheck = find<SoDialogCheckBox>( myTop, "ShowTopology" );
  ShowTopologyAuditor* showTopologyAuditor = new ShowTopologyAuditor;
  if ( find<SoVolumeRender>(g_root) != NULL )
    showTopologyCheck->addAuditor(showTopologyAuditor);
  else
    showTopologyCheck->enable = FALSE;


  // Combobox to select which object to show
  SoSwitch* showSwitch = find<SoSwitch>( g_root, "SHOW_SWITCH" );
  SoDialogComboBox* showSwitchCombo = find<SoDialogComboBox>( myTop, "SHOW_SWITCH_COMBO" );
  if ( showSwitch != NULL )
  {
    for ( int i = 0; i < showSwitch->getNumChildren(); ++i)
    {
      SoNode* child = showSwitch->getChild(i);
      showSwitchCombo->items.set1Value(i, child->getName().getString());
    }
    showSwitch->whichChild.connectFrom( &showSwitchCombo->selectedItem );
  }
  else
  {
    showSwitchCombo->enable = FALSE;
  }

  // Clear button
  SoDialogPushButton* clearButton = find<SoDialogPushButton>( myTop, "CLEAR_BUTTON" );
  ClearButtonAuditor* clearButtonAuditor = new ClearButtonAuditor();
  clearButton->addAuditor( clearButtonAuditor );

  myTop->buildDialog(myWindow, TRUE);
  myTop->show();

  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myCustom->getWidget());
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->setFastEditSavePolicy(SoGLRenderAction::WHEN_NEEDED);

  myViewer->setSceneGraph(viewerRoot);

  myViewer->viewAll();
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();

  delete myViewer;
  delete showTopologyAuditor;
  delete clearButtonAuditor;

  viewerRoot->unref();
  myTop->unref();
  myGroup->unref();
  g_volumeDataRoot->unref();
  g_CSGShapeRoot->unref();

  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


