/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Oct 2007)
**=======================================================================*/


#include <LDM/converters/SoLDMWriter.h>
#include <LDM/converters/SoConverterParameters.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/readers/SoVRLdmFileReader.h>
#include <LDM/readers/SoVolumeReader.h>
#include <LDM/SoLDMTopoOctree.h>
#include <Inventor/SbElapsedTime.h>

// Inventor algorithms
#include <Inventor/algorithms/SoAlgorithms.h>
#include <compute.h>


void
myXmlCB( FILE* /*f*/, void* /*data*/ )
{
  printf( "myXmlCB\n" );
}

int
main(int argc, char ** argv)
{
  // init VolumeViz module
  SoVolumeRendering::init();

  if ( argc >= 2) SoPreferences::setString("LDM_INPUT", argv[1]);
  if ( argc >= 3) SoPreferences::setInt("LDM_MAINMEM",atoi(argv[2]));
  if ( argc >= 4) SoPreferences::setInt("LDM_TEXMEM",atoi(argv[3]));

  SbString inputFileName = SoPreferences::getString("LDM_INPUT", "$OIVHOME/examples/data/VolumeViz/colt-float.ldm");
  if ( inputFileName == "" )
  {
    SoError::post(
      "You must define the input file name using\n"
      "LDM_INPUT environment variable\n"
      );
    exit(0);
  }

  // setup output parameters
  SbString outputFileName = SoPreferences::getString("LDM_OUTPUT","test.ldm");
  if ( outputFileName == "test.ldm" )
  {
    fprintf(stdout,
      "Warning: LDM_OUTPUT variables not defined, results will be saved in test.ldm\n");
  }

  int MaxMainMem = SoPreferences::getInt("LDM_MAINMEM",800);
  int MaxTexMem = SoPreferences::getInt("LDM_TEXMEM",400);

  SoLDMGlobalResourceParameters::setMaxMainMemory(MaxMainMem);
  SoLDMGlobalResourceParameters::setMaxTexMemory(MaxTexMem);

  fprintf(stdout,"LDM_INPUT set to %s\n",inputFileName.getString());
  fprintf(stdout,"LDM_MAINMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxMainMemory());
  fprintf(stdout,"LDM_TEXMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxTexMemory());

  SbBox3f inputSize;
  SoVolumeData::DataType inputType;
  SbVec3i32 inputDimension;
  SbVec3i32 inputTileSize;

  // creater reader for input volume
  SoVRLdmFileReader * inputReader;
  inputReader = new SoVRLdmFileReader;
  inputReader->setFilename(inputFileName);

  // get src data properties
  SoVRLdmFileReader::ReadError err = inputReader->getDataChar( inputSize, inputType, inputDimension );
  inputReader->getTileSize(inputTileSize);
  if ( err != SoVRLdmFileReader::RD_NO_ERROR )
  {
    SoError::post("unable to read input properties\n");
    exit(-1);
  }
  if (inputType!=SoDataSet::FLOAT)
  {
    SoError::post("Only FLOAT input is supported\n");
    exit(-1);
  }

  // Ok we can init the writer now
  SoConverterParameters* converterParams=SoConverterParameters::create(argc,argv);

  SoLDMWriter myLDMWriter;
  myLDMWriter.initialize(outputFileName, inputSize, inputDimension, inputType,converterParams);
  myLDMWriter.setXmlCallback( myXmlCB, NULL );


  // create the compute interface
  Compute* m_computeInterface = new Compute();

  // create a buffer to handle 1 Tile in CPU memory and 1 in GPU memory
  unsigned int tileDataSize = inputTileSize[0]*inputTileSize[1]*inputTileSize[2];

  // Create input and output tile buffer on the correct device
  // depending on compute interface used.
  SoBufferObject *inputTile  = m_computeInterface->createBufferObject(tileDataSize*inputReader->getNumBytesPerDatum());
  SoBufferObject *outputTile = m_computeInterface->createBufferObject(tileDataSize*inputReader->getNumBytesPerDatum());
  inputTile->ref();
  outputTile->ref();

  // how many tile in each direction ?
  SbVec3i32 nbTile;
  nbTile[0] = inputDimension[0]/inputTileSize[0] + ((inputDimension[0]%inputTileSize[0])?1:0);
  nbTile[1] = inputDimension[1]/inputTileSize[1] + ((inputDimension[1]%inputTileSize[1])?1:0);
  nbTile[2] = inputDimension[2]/inputTileSize[2] + ((inputDimension[2]%inputTileSize[2])?1:0);

  // init topo octree to do index and tileid computation
  SoLDMTopoOctree* topo=NULL;
  topo = new SoLDMTopoOctree;
  topo->init(inputDimension,inputTileSize[0],0);

  static double _writeTime=0.0;
  static double _readTime=0.0;
  static double _computeTime=0.0;

  for (int resolution=0; resolution<topo->getLevelMax();resolution++)
    for (int ztile=0; ztile<nbTile[2]; ztile++)
      for (int ytile=0; ytile<nbTile[1]; ytile++)
        for (int xtile=0; xtile<nbTile[0]; xtile++)
        {
          SbBox3i32 tilePosition;

          tilePosition.setBounds(
            (xtile+0)*inputTileSize[0],   (ytile+0)*inputTileSize[1],  (ztile+0)*inputTileSize[2],
            (xtile+1)*inputTileSize[0]-1, (ytile+1)*inputTileSize[1]-1, (ztile+1)*inputTileSize[2]-1
            );

          SoLDMTileID id = topo->getTileID(tilePosition.getMin(),resolution);
          int index = (int)topo->getFileID(id);


          ///////////////////////////////////////////////
          /// READ TILE FROM TILE TO DEVICE MEMORY
          ///////////////////////////////////////////////
          SbElapsedTime readTime;

          // read tile in memory
          SoBufferObject* inputBuffer = inputReader->readTile (index, tilePosition);

          _readTime += readTime.getElapsed();

          if ( inputBuffer != NULL)
          {
            // we do not know what kind of devices memory returned 
            // we then transfer it to our own input buffer.
            inputTile->getContext()->bind();
            inputTile->memcpy(inputBuffer);
            inputTile->getContext()->unbind();
            inputBuffer->ref();
            inputBuffer->unref();

            /////////////////////////////////////////
            /// COMPUTE RESULT
            /////////////////////////////////////////
            SbElapsedTime computeTime;

            // setup compute parameters
            struct ComputeParameters computeParameters;
            computeParameters.attribute = 2;
            computeParameters.inputData  = inputTile;
            computeParameters.inputDataType = inputType;
            computeParameters.outputData = outputTile;
            computeParameters.width = inputTileSize[2];                   // trace size
            computeParameters.height= inputTileSize[0]*inputTileSize[1];  // num trace
            // TODO: computeParameters.outputDataType = inputType;

            // launch computation
            SbBool ret = m_computeInterface->doCompute(&computeParameters);

            _computeTime += computeTime.getElapsed();

            if ( ret == TRUE )
            {
              ///////////////////////////////////////////////
              /// WRITE TILE FROM DEVICE MEMORY TO FILE
              ///////////////////////////////////////////////
              SbElapsedTime writeTime;

              // write tile to file.
              myLDMWriter.writeTile(tilePosition.getMin(), outputTile, resolution);

              _writeTime += writeTime.getElapsed();
            }
            else
            {
              fprintf(stderr,"doCompute return FALSE\n");
            }
          }
          else
          {
            fprintf(stderr,"readTile return FALSE\n");
          }
        }

  size_t processedSize = tileDataSize*inputReader->getNumBytesPerDatum()*nbTile[0]*nbTile[1]*nbTile[2];
  processedSize = processedSize / (1024*1024);
  fprintf(stdout,"read Time    : %.2f ( %.2f MB/sec)\n",_readTime,(processedSize/_readTime));
  fprintf(stdout,"compute Time : %.2f (%.2f MB/sec)\n",_computeTime,(processedSize/_computeTime));
  fprintf(stdout,"write Time   : %.2f (%.2f MB/sec)\n",_writeTime,(processedSize/_writeTime));

  delete topo;

  // release allocated Tiles for computing
  inputTile->unref();
  outputTile->unref();

  // release compute module
  delete m_computeInterface;

  // close writer
  myLDMWriter.finish();

  // finish VolumeViz module
  SoVolumeRendering::finish();

  return 0;
}


