/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Federico Gamba (Nov 2009)
**=======================================================================*/

#include "OivSceneGraph.h" 
#include "GuiDialogViz.h"

#include <Inventor/helpers/SbFileHelper.h>


extern void ROIChangedCB( void *data, SoDragger *dragger);

OivSceneGraph::OivSceneGraph(const SbString &LdmFilename)
{
   m_LDM_RAM = 800; //MBytes of RAM Cache
   m_LDM_GPU_3D = 400;  //MBytes of GPU 3D Cache
   m_LDM_GPU_2D = 40;  //MBytes of GPU 2D Textures

  createSceneGraph();
  setFilename(LdmFilename);
  
  loadAvizoColormap("$OIVHOME/examples/source/VolumeViz/Compute/ComputeHistogram/colormap.txt");
  assembleSceneGraph();
}

void OivSceneGraph::setFilename (const SbString &LdmFilename)
{
  m_LdmFilename = LdmFilename;
  m_VolumeData->fileName.setValue(m_LdmFilename);
  m_LdmDim = m_VolumeData->data.getSize();
  m_VolumeData->getMinMax(m_DataMin,m_DataMax);
  m_Extent= m_VolumeData->extent.getValue();

  m_DataRange->min.setValue(m_DataMin);
  m_DataRange->max.setValue(m_DataMax);

  m_ROIManip->box.setValue(SbVec3i32(0,0,0), m_LdmDim/4 );
  m_ROIManip->subVolume.setValue( SbVec3i32(0,0,0), m_LdmDim - SbVec3i32(1,1,1) );

  createBBox();

}
SoSeparator* OivSceneGraph::createSceneGraph() {

	char app_name[255];

	m_Root = new SoSeparator;
	m_Root->setName("Root");

	m_Background = new SoGradientBackground;
	m_Background->setName("Background");

#ifdef USE_POSCENEVIEW
	m_3DScene = new PoSceneView;
	m_3DCamera = new SoPerspectiveCamera;
#else
	m_3DScene = new SoSeparator;
#endif
	m_3DScene->setName("_3D_Scene");

	m_3DRoot = new SoSeparator;
	m_3DRoot->setName("_3D_Root");

	m_2DScene = new PoSceneView;
	m_2DScene->setName("_2D_Scene");
	m_2DRoot = new SoSeparator;
	m_2DRoot->setName("_2D_Root");
	m_2DCamera = new SoOrthographicCamera;

	m_Transform = new SoTransform;
	m_Transform->setName("ZScale");

	m_VolumeRoot = new SoSeparator;
	m_VolumeRoot->setName("VolumeRoot");

	m_VolumeData = new SoVolumeData;
	m_VolumeData->setName("LdmData");

	m_SwitchSlices = new SoSwitch;
	m_SwitchSlices->setName("SwitchSlices");

	for (int i=0; i<NUM_SLICE; ++i) {
		m_Slice[i] = new SoOrthoSlice;;
		sprintf(app_name,"Slice_%i",i);
		m_Slice[i]->setName(app_name);
		sprintf(app_name,"SwitchSlice_%i",i);
		m_SwitchSlice[i] = new SoSwitch;
		m_SwitchSlice[i]->setName(app_name);
	}

	m_TransferFunction = new SoTransferFunction;
	m_TransferFunction->setName("ColorMap");

	m_DataRange = new SoDataRange;
	m_DataRange->setName("DataRange");

	m_ROIManip = new SoROIManip;
	m_ROIManip->setName("ROI");


	m_SwitchVolRend = new SoSwitch;
	m_VolSkin = new SoVolumeSkin;

	m_VolRendQ = new SoVolumeRenderingQuality;
	m_VolRend = new SoVolumeRender;


	m_BoundingBox = new SoSeparator;
	m_BoundingBox->setName("BBox");

	m_Axes = new SoSeparator;
	m_Axes->setName("Axes");
	m_SwitchAxes = new SoSwitch;
	m_SwitchAxes->setName("Axes");

	createHistogram();

    return m_Root;
}

void
OivSceneGraph::assembleSceneGraph()
{

	//init LDM parameters
  SoLDMGlobalResourceParameters::setLoadNotificationRate(100);
  SoLDMGlobalResourceParameters::setTex3LoadRate(50);
  SoLDMGlobalResourceParameters::setMaxMainMemory(m_LDM_RAM);
  SoLDMGlobalResourceParameters::setMaxTexMemory(m_LDM_GPU_3D);
  SoLDMResourceParameters* pLDM_Pars = m_VolumeData->ldmResourceParameters.getValue();
  pLDM_Pars->tex2LoadRate.setValue(200);
  pLDM_Pars->max2DTexMemory.setValue(m_LDM_GPU_2D);
  m_Resolution=1;

  // Assemble the scene graph
  m_Root->ref();
  m_Root->addChild(m_Background);
  m_Root->addChild(m_3DScene);
  m_Root->addChild(m_2DScene);

#ifdef USE_POSCENEVIEW
   m_3DScene->setPart("scene", m_3DRoot);
   m_3DScene->sensitiveOnEvents(TRUE);
   m_3DScene->isBorderVisible=FALSE;
   m_3DScene->viewportOrigin.setValue(0.f, SCENE_SPLIT);
   m_3DScene->viewportSize.setValue(1.0f, 1.0f - SCENE_SPLIT);
   m_3DScene->setPart("cameraKit.camera", m_3DCamera);
#else
   m_3DScene->addChild(m_3DRoot);
#endif

   m_2DScene->setPart("scene", m_2DRoot);
   m_2DScene->sensitiveOnEvents(TRUE);
   m_2DScene->isBorderVisible=TRUE ;
   m_2DScene->viewportOrigin.setValue(0., 0.);
   m_2DScene->viewportSize.setValue(1.0f, SCENE_SPLIT);
   m_2DScene->setPart("cameraKit.camera", m_2DCamera);

  {
      m_3DRoot->addChild( m_Transform );
      m_3DRoot->addChild( m_VolumeRoot );
	   
	  {

		  m_VolumeRoot->addChild( m_VolumeData );
		  m_VolumeRoot->addChild( m_DataRange );
		  m_VolumeRoot->addChild( m_TransferFunction );
		  m_VolumeRoot->addChild( m_SwitchSlices ); 
				m_SwitchSlices->whichChild=SO_SWITCH_ALL;

		  {
			  for (int i=0;i<NUM_SLICE;i++) {
				SoOrthoSlice* app = (SoOrthoSlice*)(m_Slice[i]);
				app->axis.setValue(i%3);
				app->alphaUse = SoSlice::ALPHA_OPAQUE;

					m_SwitchSlices->addChild( m_SwitchSlice[i] ); 
							m_SwitchSlice[i]->whichChild=SO_SWITCH_ALL;
							m_SwitchSlice[i]->addChild(m_Slice[i]);

			  }

		  }

		  m_VolumeRoot->addChild(m_ROIManip);
					m_ROIManip->boxOn=true;
					m_ROIManip->constrained=true;

		  m_VolumeRoot->addChild(new SoGeometryPriority(0.9f));
		  m_VolumeRoot->addChild(m_SwitchVolRend);
				m_SwitchVolRend->addChild(m_VolSkin); //skin
					  m_VolSkin->alphaUse = SoVolumeSkin::ALPHA_OPAQUE;

					  SoSeparator* pVolRend = new SoSeparator; //node for VolRendering with Quality
					  pVolRend->addChild(m_VolRendQ);
					  pVolRend->addChild(m_VolRend);
					  
				m_SwitchVolRend->addChild(pVolRend);
				m_SwitchVolRend->whichChild = 0; //by default enable Skin


			// add SoPickStyle to avoid picking of Axes and BBox
			SoPickStyle *pUnpickable = new SoPickStyle;
			pUnpickable->style = SoPickStyle::UNPICKABLE;
				m_VolumeRoot->addChild( pUnpickable );
				m_VolumeRoot->addChild( m_BoundingBox );
				m_VolumeRoot->addChild( m_SwitchAxes ); 
					m_SwitchAxes->whichChild=SO_SWITCH_ALL;
					//axes not yet implemented
				m_VolumeRoot->addChild( new SoPickStyle );
		  
	  }
  }

    m_ROIManip->getDragger()->addValueChangedCallback ( (SoDraggerCB *)&ROIChangedCB, this );
    ROIChangedCB( (void*)this, NULL );


}

void 
OivSceneGraph::createBBox()
{
    float xmin,ymin,zmin,xmax,ymax,zmax;
    m_Extent.getBounds( xmin,ymin,zmin,xmax,ymax,zmax );

    SoDrawStyle *pStyle = new SoDrawStyle;
    pStyle->lineWidth = 2;
    SoLightModel *pLMod = new SoLightModel;
    pLMod->model = SoLightModel::BASE_COLOR;

    SoVertexProperty *pProp = new SoVertexProperty;
    pProp->vertex.setNum( 8 );
    pProp->vertex.set1Value( 0, xmin, ymin, zmin );
    pProp->vertex.set1Value( 1, xmax, ymin, zmin );
    pProp->vertex.set1Value( 2, xmax, ymax, zmin );
    pProp->vertex.set1Value( 3, xmin, ymax, zmin );
    pProp->vertex.set1Value( 4, xmin, ymin, zmax );
    pProp->vertex.set1Value( 5, xmax, ymin, zmax );
    pProp->vertex.set1Value( 6, xmax, ymax, zmax );
    pProp->vertex.set1Value( 7, xmin, ymax, zmax );
    pProp->orderedRGBA.set1Value( 0, SbColor(1,0,0).getPackedValue() );

    const int32_t indices[] = {
      0, 1, 2, 3, 0, -1,
        4, 5, 6, 7, 4, -1,
        0, 4, -1,
        1, 5, -1,
        2, 6, -1,
        3, 7, -1
    };
    SoIndexedLineSet *pLine = new SoIndexedLineSet;
    pLine->vertexProperty = pProp;
    pLine->coordIndex.setValues( 0, 24, indices );

    m_BoundingBox->addChild( pStyle );
    m_BoundingBox->addChild( pLMod  );
    m_BoundingBox->addChild( pLine );

}


//assume file is well formatted
//256 lines of 4 floats (RGBA)
void 
OivSceneGraph::loadAvizoColormap( const SbString &FileName )
{  
  float r, g, b, a;
  float* rgba;
  
  SbString colormapFilename = SbFileHelper::expandString(FileName);
  FILE* fp = fopen(colormapFilename.toLatin1(),"r");

  if (!fp) return;

  // Read values
  rgba = new float[256 * 4];
  memset(rgba,0,sizeof(float)*256*4);

  for (int i=0;i<256;i++) {
	fscanf( fp, "%g %g %g %g", &r, &g, &b, &a );
	rgba[i*4+0] = r;
	rgba[i*4+1] = g;
	rgba[i*4+2] = b;
	rgba[i*4+3] = a;
  }

  fclose(fp);

  m_TransferFunction->predefColorMap = SoTransferFunction::NONE;
  m_TransferFunction->colorMapType = SoTransferFunction::RGBA;
  m_TransferFunction->colorMap.setValues( 0, 256 * 4, rgba );

 }

void
OivSceneGraph::createHistogram() {


  SoSeparator *histo_root = new SoSeparator;
  m_2DRoot->addChild(histo_root);

  SoTranslation* pTrans = new SoTranslation;
  pTrans->translation.setValue(0.f,-0.9f,0.f);
  PoDomain    *pDomain = new PoDomain ;
  pDomain->min.setValue(0.f, 0.f,0.f) ;
  pDomain->max.setValue(1.8f, 1.f,1.f) ;
  

  int i;
  for (i=0;i<256;i++) {
	  m_histo256_f[i]=1;
  }
  float app = 1.2f*(1.0f/SCENE_SPLIT);
  m_histo = new PoSingleHistogram(SbVec2f(-app,0), app, PoSingleHistogram::X, 
		      256, (const float*)m_histo256_f, NULL) ;
  m_histo->valueVisibility = PoSingleHistogram::VISIBILITY_OFF;
  m_histo->nameVisibility = PoSingleHistogram::VISIBILITY_OFF;
  m_histo->barSpaceValue=0.;
  
  //PoLinearAxis *yAxis = new PoLinearAxis(SbVec2f(-app,0), 13, PoLinearAxis::YX) ; 
  //yAxis->gradFontSize = 0.06F ;
  //yAxis->titleString.setValue("Price%(M Dollars)") ;
  //yAxis->titleVisibility = PoLinearAxis::VISIBILITY_ON ;
  //yAxis->titleFontSize = 0.06F ;
  //yAxis->set("appearance.material", "diffuseColor 1 1 1") ;
  //yAxis->set("bodyApp.material", "diffuseColor 1 0 0") ;
  //yAxis->set("bodyApp.drawStyle", "lineWidth 2.0") ;


  histo_root->addChild(pTrans);
  histo_root->addChild(pDomain);
  histo_root->addChild(m_histo);
}

//convert histo into 256 entries normalized between 0 and 1
void NormalizeHisto256(int64_t* hvalues, size_t N, float histo256_f[256] ) {

   //move histo into 256 values
  int64_t hvalues256[256];
  if (N==256) {
	  memcpy(hvalues256,hvalues,sizeof(int64_t)*256);
  } else { //65536
	  int cc=0;
	  for (int i=0;i<256;i++) {
		  int64_t app = 0;
		  for (size_t j=0;j<(N/256);j++) {
			  app+=hvalues[cc];
			  cc++;
		  }
		  hvalues256[i] = app;
	  }
  }

  //normalize histo
  int64_t imin,imax,app;
  double dmin,ddelta,dapp;
  //determine min and max
  imax=imin=hvalues256[0];
  for (int i=0;i<256;i++) {
		app=hvalues256[i];
		if (app<imin) imin=app;
		if (app>imax) imax=app;
  }
  dmin = (double)imin;
  ddelta = (double)(imax-imin);

  for (int i=0;i<256;i++) {
	  dapp = (double)(hvalues256[i]);
	  dapp = (dapp - dmin)/ddelta;
	  histo256_f[i] = (float)(dapp);
  }
}

void
OivSceneGraph::updateHistogram() {
	
  //SbBox3i32   subvol = m_ROIManip->subVolume.getValue();
  SbBox3i32   subvol = m_ROIManip->box.getValue();

  // Display ROI dimensions in user interface
  int xmin,ymin,zmin,xmax,ymax,zmax;
  subvol.getBounds( xmin, ymin, zmin, xmax, ymax, zmax );
  //updateROIdisplay( xmin, ymin, zmin, xmax, ymax, zmax );

  // get data informations
  SbBox3i32 box(SbVec3i32(xmin, ymin, zmin), SbVec3i32(xmax, ymax, zmax));
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  SoLDMDataAccess& dataAccess(m_VolumeData->getLdmDataAccess());
  SoDataSet* DS = dataAccess.getDataSet();
  SoDataSet::DataType DT = DS->getDataType();
  //query size of data buffer
  info = dataAccess.getData(m_Resolution, box,(SoBufferObject*)NULL);
  SoRef<SoCpuBufferObject> bufferObj = new SoCpuBufferObject;
  bufferObj->setSize((size_t)info.bufferSize);

  info = dataAccess.getData(m_Resolution, box, bufferObj.ptr());
  if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {
	  //printf("buf: %d\n",info.bufferSize);
  } else
    SoError::post("Error while data access request");

  SbVec3i32 buffDim = info.bufferDimension;
  int32_t nx,ny,nz;
  buffDim.getValue(nx,ny,nz);
  int buffElements = nx*ny*nz;

  SoVolumeHistogram aHisto(DT);
  aHisto.setInputValueRange(m_DataMin,m_DataMax);
  aHisto.addValues(bufferObj.ptr(),buffElements);

  size_t numBins = aHisto.getHistoSize();
  int64_t* hvalues = aHisto.getHistogram();

  NormalizeHisto256(hvalues, numBins, m_histo256_f);

  float* app = m_histo->value.startEditing();
  memcpy(app,m_histo256_f,sizeof(float)*256);
  m_histo->value.finishEditing();
}


// recompute when ROI subvolume changes
void ROIChangedCB( void *data, SoDragger* /*dragger*/)
{
  OivSceneGraph* pSceneGraph = (OivSceneGraph*)data;
  pSceneGraph->updateHistogram();

}


