/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Oct 2007)
**=======================================================================*/


#include "colormaps.h"

#include <Inventor/nodes/SoColorMap.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/actions/SoSearchAction.h>

////////////////////////////////////////////////////////////////////////
//
int loadColormap( SoColorMap *pTF, const char *filename )
{
  FILE *fp = fopen( filename, "r" );
  if (!fp)
    return 0;

  float values[4*256];
  float *p = values;
  for (int i = 0; i < 256; ++i) {
    fscanf( fp, "%f %f %f %f", p, (p+1), (p+2), (p+3) );
    p += 4;
  }

  pTF->colorMap.setValues( 0, 4*256, values );
  fclose( fp );
  return 256;
}

SoSeparator *
loadColorTables(const char *filename)
{
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename))
  {
    fprintf(stderr, "Cannot open file %s\n", filename);
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator *fileTables = SoDB::readAll(&mySceneInput);
  if (fileTables == NULL)
  {
    fprintf(stderr, "Problem reading color tables file\n");
    return NULL;
  }
  fileTables->ref();

  SoSearchAction sa;
  sa.setSearchingAll(true);
  sa.setInterest(SoSearchAction::ALL);
  sa.setType(SoTransferFunction::getClassTypeId());
  sa.apply(fileTables);
  SoPathList &paths = sa.getPaths();

  SoSeparator *tables = NULL;
  if (paths.getLength())
  {
    tables = new SoSeparator;
    tables->ref();
    for (int ic=0; ic<paths.getLength(); ic++)
    {
      const SoPath *path = paths[ic];
      SoNode *ntail = path->getTail();
      SoTransferFunction *tf = (SoTransferFunction *)ntail;
      int numColors = tf->colorMap.getNum()/4;
      if ( (numColors > 256) )
      {
        int step = numColors/256;
        float *newcolors = tf->colorMap.startEditing();//new float[256*4];
        float *oldcolors = tf->colorMap.startEditing();
        for (int i=0; i<256; i++)
        {
          newcolors[4*i+0] = oldcolors[4*i*step+0];
          newcolors[4*i+1] = oldcolors[4*i*step+1];
          newcolors[4*i+2] = oldcolors[4*i*step+2];
          newcolors[4*i+3] = oldcolors[4*i*step+3];
        }
        //tf->colorMap.setNum(256);
        tf->colorMap.finishEditing();
        tf->colorMap.setNum(256*4);
      }
      tables->addChild(tf);
    }
  }

  mySceneInput.closeFile();
  fileTables->unref();
  if (tables != NULL)
    tables->unrefNoDelete();
  return tables;
}

SoTransferFunction *
getColorMap(SoSeparator *tables, SbString name)
{
  if (tables==NULL)
    return NULL;

  tables->ref();
  SoSearchAction sa;
  sa.setInterest(SoSearchAction::ALL);
  sa.setType(SoTransferFunction::getClassTypeId() );
  sa.apply(tables);
  SoPathList &paths = sa.getPaths();
  int np = paths.getLength();

  for (int i=0; i<np; i++)
  {
    const SoPath *path = paths[i];
    SoNode *ntail = path->getTail();
    SoTransferFunction *tf = (SoTransferFunction *)ntail;
    if (tf->getName().getLength() != 0)
    {
      if (tf->getName().getString() == name )
        return tf;
    }
  }
  tables->unrefNoDelete();
  return NULL;
}

SoTransferFunction *getTransferFunction(const char *filename, SbString name)
{
  SoSeparator *tables = loadColorTables(filename);
  return getColorMap(tables, name);
}


// load the colormap contained in SLB tables files eles 
// load the colormap in directFile
SoTexture2* setupColor(SoSeparator *tables,const char* SlbName, const char* directFile)
{
  SoTexture2* colormap_texture = new SoTexture2;

  SoTransferFunction *tf = getColorMap(tables,SlbName);
  if ( tf != NULL )
  {
    // create a texture that handle the colormap values
    colormap_texture->image.setValue(
      SbVec2s(tf->colorMap.getNum()/4,1),
      4,//tf->getNumComponents(),  // nb component
      SoSFImage::FLOAT,
      tf->colorMap.getValues(0)
    );
  }
  else
  {
    // load default colormap
    SoColorMap *localColor = new SoColorMap;
    loadColormap(localColor,directFile);

    // create a texture that handle the colormap values
    colormap_texture->image.setValue(
      SbVec2s(localColor->getNumColor(),1),
      localColor->colorMap.getNum()/localColor->getNumColor(),  // nb component
      SoSFImage::FLOAT,
      localColor->colorMap.getValues(0)
    );
  }

  return colormap_texture;
}

SoSwitch *
setupColortables(const char *filename)
{
  SoSeparator *tables = loadColorTables(filename);

  SoSwitch* colorTableSwitch = new SoSwitch;

  // colormap for original input data
  colorTableSwitch->addChild( setupColor(tables,"Seismic","colormap_amplitude.dat") );

  // colormap for hilbert 
  colorTableSwitch->addChild( setupColor(tables,"Seismic"  ,"colormap_amplitude.dat") );

  // colormap for enveloppe
  colorTableSwitch->addChild( setupColor(tables,"Envelope"  ,"colormap_amplitude.dat") );

  // colormap for Phase
  colorTableSwitch->addChild( setupColor(tables,"Phase"  ,"colormap_phase.dat") );

  colorTableSwitch->whichChild.setValue(0);

  return colorTableSwitch;
}

SoSwitch *
setupTransferFunction(const char *filename)
{
  SoSeparator *tables = loadColorTables(filename);

  SoSwitch* colorTableSwitch = new SoSwitch;

  // colormap for original input data
  colorTableSwitch->addChild( getColorMap(tables, "Seismic") );

  // colormap for hilbert 
  colorTableSwitch->addChild( getColorMap(tables,"Seismic") );

  // colormap for enveloppe
  colorTableSwitch->addChild( getColorMap(tables,"Envelope") );

  // colormap for Phase
  colorTableSwitch->addChild( getColorMap(tables,"Phase") );

  colorTableSwitch->whichChild.setValue(0);

  return colorTableSwitch;
}
