/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Oct 2007)
**=======================================================================*/


//#define SOQT

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoRenderToTextureProperty.h>
#include <Inventor/nodes/SoTextureUnit.h>
#include <Inventor/nodes/SoShaderProgram.h>
#include <Inventor/nodes/SoShaderParameter.h>
#include <Inventor/nodes/SoFragmentShader.h>

#include <Inventor/SbElapsedTime.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/sensors/SoNodeSensor.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoDataRange.h>

#include <VolumeViz/manips/SoROIManip.h>
#include <VolumeViz/LDM/SoLDMResourceManager.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/algorithms/SoAlgorithms.h>

#include <compute.h>
#include <colormaps.h>

#include <Inventor/helpers/SbFileHelper.h>

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

//---------------------------    Init DATA    ----------------------------------

SbString g_guiFilename = SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/Compute/ComputeSlice/ComputeSlice.iv");

SbString ColorTablesFile= SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/Compute/ComputeSlice/ColorTables.iv");

SbString ColormapVertFile= SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/Compute/ComputeSlice/colormapVert.glsl");
SbString ColormapFragFile= SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/Compute/ComputeSlice/colormapFrag.glsl");


int g_toggleResolution = 4;
int g_toggleAttribute = 0;

int g_animateDirection = 1;

bool g_toggleInterpolation = true;
bool g_isOn = true;

SoTransferFunction *g_transFunc;

//---------------------------      DATA       ----------------------------------

struct ComputeParameters g_computeParameters;
Compute* g_compute = NULL;

SoShaderParameter1i* g_transparentBorders;

SoBufferObject* g_inputBufferObject = NULL;

SoTexture2* g_dataTexture = NULL;
SoVolumeData* g_volData = NULL;
SoVertexProperty* g_sliceVertexProperty = NULL;
SoFaceSet* g_textureQuad = NULL;
SoFaceSet* g_view2TexturePolygon = NULL;
SoSwitch* g_colorMapsSwitch = NULL;
SoROIManip* g_ROIManip = NULL;
SoOrthoSlice* g_orthoSlice = NULL;

SoTopLevelDialog* g_topLevelDialog = NULL;
SoSwitch* g_displaySwitch = NULL;

SoTimerSensor* g_animationSensor = NULL;

int g_currentMinPercent=0;
int g_currentMaxPercent=100;

SbElapsedTime g_perfCounter;
double g_nbMPoints = 0;

//---------------------------   FUNCTIONS    ----------------------------------

Widget buildInterface( Widget, SoVolumeData* );
SoSeparator *makeVolBBox( const SbBox3f volSize );

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle interpolation
  if (SO_KEY_PRESS_EVENT(event, I)) 
  {
    g_toggleInterpolation = !g_toggleInterpolation;
    g_ROIManip->subVolume.touch();
  }

  // change loading Mode
  else if (SO_KEY_PRESS_EVENT(event, L)) 
  {
    if (g_isOn)
    {
      g_volData->ldmResourceParameters.getValue()->loadPolicy.setValue( SoVolumeData::SoLDMResourceParameters::NEVER );
      fprintf(stderr,"Mode is NEVER\n");
    }
    else
    {
      g_volData->ldmResourceParameters.getValue()->loadPolicy.setValue( SoVolumeData::SoLDMResourceParameters::ALWAYS );
      fprintf(stderr,"Mode is ALWAYS\n");
    }

    g_isOn = !g_isOn;
  }
  // YZ plane
  else if (SO_KEY_PRESS_EVENT(event, X))
  {
    g_orthoSlice->axis = SoOrthoSlice::X;
    g_orthoSlice->sliceNumber = g_volData->data.getSize()[0] / 2;

    // setup slice slider range
    SoDialogIntegerSlider *slider1 =
      (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("slice"));
    if (slider1)
    {
      SbVec3i32 voldim  = g_volData->data.getSize();
      slider1->min = 0;
      slider1->max = voldim[0] - 1;
      slider1->value = voldim[0] / 2;
    }

  }
  // XZ plane
  else if (SO_KEY_PRESS_EVENT(event, Y))
  {
    g_orthoSlice->axis = SoOrthoSlice::Y;
    g_orthoSlice->sliceNumber = g_volData->data.getSize()[1] / 2;
    // setup slice slider range
    SoDialogIntegerSlider *slider1 =
      (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("slice"));
    if (slider1)
    {
      SbVec3i32 voldim  = g_volData->data.getSize();
      slider1->min = 0;
      slider1->max = voldim[1] - 1;
      slider1->value = voldim[1] / 2;
    }

  }
  // XY plane
  else if (SO_KEY_PRESS_EVENT(event, Z))
  {
    g_orthoSlice->axis = SoOrthoSlice::Z;
    g_orthoSlice->sliceNumber = g_volData->data.getSize()[2] / 2;
    // setup slice slider range
    SoDialogIntegerSlider *slider1 =
      (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("slice"));
    if (slider1)
    {
      SbVec3i32 voldim  = g_volData->data.getSize();
      slider1->min = 0;
      slider1->max = voldim[2] - 1;
      slider1->value = voldim[2] / 2;
    }
  }
}


//------------------------------------------------------------------------------
void updateCompute()
{
  if ( !g_inputBufferObject )
    return;

  // convert inputType to SbDataType
  SbDataType dataType;

  switch ( g_volData->getDataType() )
  {
  case SoVolumeData::UNSIGNED_BYTE  : dataType = SbDataType::UNSIGNED_BYTE ; break;
  case SoVolumeData::UNSIGNED_SHORT : dataType = SbDataType::UNSIGNED_SHORT; break;
  case SoVolumeData::UNSIGNED_INT32 : dataType = SbDataType::UNSIGNED_INT32; break;
  case SoVolumeData::SIGNED_BYTE    : dataType = SbDataType::SIGNED_BYTE   ; break;
  case SoVolumeData::SIGNED_SHORT   : dataType = SbDataType::SIGNED_SHORT  ; break;
  case SoVolumeData::SIGNED_INT32   : dataType = SbDataType::SIGNED_INT32  ; break;
  case SoVolumeData::FLOAT          : dataType = SbDataType::FLOAT         ; break;
  case SoVolumeData::DOUBLE         : dataType = SbDataType::DOUBLE        ; break;
  }
  // setup Compute function parameters
  g_computeParameters.attribute = g_toggleAttribute;
  g_computeParameters.inputData = g_inputBufferObject;
  g_computeParameters.inputDataType = dataType;

  // update stats
  g_nbMPoints += (double)( g_computeParameters.width * g_computeParameters.height ) / 1000.0;

  // setup expected output data range
  // get source data range
  double mmin, mmax;
  g_volData->getMinMax(mmin,mmax);

  switch ( g_toggleAttribute ) 
  {
    case 0: /* NO attribute : output range = input range */
      g_computeParameters.dataMin = (float)mmin; g_computeParameters.dataMax = (float)mmax; break;
    case 1: /* Hilbert      : output range = input range */
      g_computeParameters.dataMin = (float)mmin; g_computeParameters.dataMax = (float)mmax; break;
    case 2: /* Enveloppe    : output range = [0..2*MAXINPUT] */
      g_computeParameters.dataMin = 0.0f; g_computeParameters.dataMax = (float)(sqrt(2.0f*mmax*mmax)); break;
    case 3: /* Phase        : output range = [-PI..PI] */
      g_computeParameters.dataMin = -3.1416f; g_computeParameters.dataMax = 3.1416f; break;
  } /* switch */

  float range = (g_computeParameters.dataMax - g_computeParameters.dataMin);
  // correct Min and max with user percent
  g_computeParameters.dataMin += range *((float)g_currentMinPercent)/100.0f;
  g_computeParameters.dataMax -= range * ( 100.0f - (float)g_currentMaxPercent ) / 100.0f;

  // setup the right colormap in the colormap switch
  if ( g_colorMapsSwitch )
    g_colorMapsSwitch->whichChild.setValue( g_toggleAttribute );

  // just set the parameters to the object, this will trigger
  // the render and then the compute call
  g_compute->doCompute( &g_computeParameters );

  g_dataTexture->image.setValue( SbVec2s( g_computeParameters.width,g_computeParameters.height ), 4, SoSFImage::UNSIGNED_BYTE, g_computeParameters.outputData , SoSFImage::NO_COPY );

  double elapsed = g_perfCounter.getElapsed();

  if ( elapsed>1.0 )
  {
    // setup dataType label
    SoDialogLabel *label =
      (SoDialogLabel*)g_topLevelDialog->searchForAuditorId(SbString("label_perf"));
    if (label)
    {
      double pps = g_nbMPoints/elapsed;
      SbString strLabel;
      strLabel = "";
      strLabel += (int)pps ;
      strLabel += " Kvalues/secs";
      label->label.setValue(strLabel);
      g_perfCounter.reset();
      g_nbMPoints = 0.0;
    }
  }
}

//------------------------------------------------------------------------------
void setComputeInterface()
{
  // reset inputBuffer
  SO_UNREF_RESET(g_inputBufferObject);

  // reset compute interface
  delete g_compute;
  g_compute = new Compute();
}


//------------------------------------------------------------------------------
SoSeparator *
viewer2_SceneGraph()
{
  SoSeparator *root = new SoSeparator;
  root->ref();

  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
  root->addChild( eventCB );

  SoLightModel *lightModel = new SoLightModel;
  lightModel->model = SoLightModel::BASE_COLOR;
  root->addChild( lightModel );
  root->addChild( new SoPerspectiveCamera );

  SoSeparator *root1 = new SoSeparator;
  root->addChild( root1 );

  // lineStyle
  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  root1->addChild( lineStyle );

  // Volume Bounding Box
  SoSeparator *volSep = new SoSeparator;
  root1->addChild( volSep );
  {
    SbVec3i32 dim = g_volData->data.getSize();
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( dim[0]/2.0f, dim[1]/2.0f, dim[2]/2.0f );
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = (float)dim[0];
    vol->height = (float)dim[1];
    vol->depth  = (float)dim[2];
    volSep->addChild( vol );
  }

  // setup slice vp
  g_sliceVertexProperty = new SoVertexProperty;
  {
    int i;
    SbVec3f vertices[6];
    SbVec2f texcoord[6];
    for (i = 0; i < 6; i++) {
      vertices[i] = SbVec3f(0,0,0);
      texcoord[i] = SbVec2f(0,1);
    }
    g_sliceVertexProperty->vertex.setValues(0, 6, vertices);
    g_sliceVertexProperty->texCoord.setValues( 0, 6, texcoord );
  }

  // slice BBox
  SoSeparator *sliceBBoxSep = new SoSeparator;
  root1->addChild( sliceBBoxSep  );
  {
    SoBaseColor *quadColor = new SoBaseColor;
    quadColor->rgb.setValue( 0.3f, 0.7f, 0.5f );
    sliceBBoxSep->addChild( quadColor );

    g_textureQuad = new SoFaceSet;
    g_textureQuad->vertexProperty = g_sliceVertexProperty;
    g_textureQuad->numVertices.set1Value( 0, 4 );

    sliceBBoxSep->addChild( g_textureQuad );
  }

  // slice texture data
  SoSeparator *sliceDataSep = new SoSeparator;
  root1->addChild( sliceDataSep );
  {
    // colormap texture to use with the shader
    SoTextureUnit* colormap_texture_unit = new SoTextureUnit;
    colormap_texture_unit->unit.setValue(0);
    sliceDataSep->addChild( colormap_texture_unit );

    // setup colorMap switch for each attribute

    g_colorMapsSwitch = setupColortables(ColorTablesFile.toLatin1());
    sliceDataSep->addChild(g_colorMapsSwitch);

    // setup texture that hold data
    SoTextureUnit* g_dataTexture_unit = new SoTextureUnit;
    g_dataTexture_unit->unit.setValue(1);
    sliceDataSep->addChild( g_dataTexture_unit );

    // Create Texture
    g_dataTexture = new SoTexture2;
    g_dataTexture->minFilter = SoTexture2::LINEAR;
    g_dataTexture->magFilter = SoTexture2::LINEAR;
    g_dataTexture->image.setHasTransparency( SoSFImage::ON );
    g_dataTexture->model.setValue(SoTexture::MODULATE);

    sliceDataSep->addChild( g_dataTexture );

    // create the Shader
    SoShaderParameter1i* p_colorMap = new SoShaderParameter1i;
    p_colorMap->name  = "colormap";
    p_colorMap->value = colormap_texture_unit->unit.getValue();

    g_transparentBorders = new SoShaderParameter1i;
    g_transparentBorders->name  = "transparentBorders";
    g_transparentBorders->value = 0;

    SoShaderParameter1i* p_data = new SoShaderParameter1i;
    p_data->name = "inputData";
    p_data->value = g_dataTexture_unit->unit.getValue();

    // Vertex shader
    SoVertexShader* VertexShader = new SoVertexShader;
    VertexShader->sourceProgram.setValue(ColormapVertFile);

    // Fragment shader
    SoFragmentShader* FragmentShader = new SoFragmentShader;
    FragmentShader->sourceProgram.setValue(ColormapFragFile);
    FragmentShader->parameter.set1Value(0,p_colorMap);
    FragmentShader->parameter.set1Value(1,p_data);
    FragmentShader->parameter.set1Value(2,g_transparentBorders);

    SoMaterial *texColor = new SoMaterial;
    texColor->diffuseColor.setValue( 1.0f, 1.0f, 1.0f );
    texColor->transparency = 0.5f;
    sliceDataSep->addChild( texColor );
    
    // Shader Program
    SoShaderProgram *shaderProgram = new SoShaderProgram;
    shaderProgram->shaderObject.set1Value(0, VertexShader);
    shaderProgram->shaderObject.set1Value(1, FragmentShader);
    sliceDataSep->addChild( shaderProgram  );

    SoDrawStyle *texStyle = new SoDrawStyle;
    texStyle->style = SoDrawStyle::FILLED;
    texStyle->lineWidth = 5.f;
    sliceDataSep->addChild( texStyle );

    g_view2TexturePolygon = new SoFaceSet;
    g_view2TexturePolygon->vertexProperty = g_sliceVertexProperty;
    g_view2TexturePolygon->numVertices.set1Value( 0, 4 );

    sliceDataSep->addChild( g_view2TexturePolygon );
  }

  return root;
}


//------------------------------------------------------------------------------
void
updateView2( void* /*userData*/ )
{
  // get OrthoSlice data Plane Coordinate
  SbPlane plane;
  {
    SbVec3f planeNormal;
    float planeDistance;
    SbVec3f volmin, volmax;

    g_volData->extent.getValue().getBounds( volmin, volmax );
    uint32_t slice = g_orthoSlice->sliceNumber.getValue();
    switch ( g_orthoSlice->axis.getValue() )
    {
      case SoOrthoSlice::X : planeNormal = SbVec3f(1.0f, 0.0f, 0.0f); break;
      case SoOrthoSlice::Y : planeNormal = SbVec3f(0.0f, 1.0f, 0.0f); break;
      case SoOrthoSlice::Z : planeNormal = SbVec3f(0.0f, 0.0f, 1.0f); break;
    } /* switch */
    planeDistance = (float)slice ;
    plane = SbPlane( planeNormal, planeDistance );
  }

  // get Slice information
  SbBox3i32 box( g_ROIManip->subVolume.getValue().getMin(), g_ROIManip->subVolume.getValue().getMax() );
  SoVolumeData::LDMDataAccess::DataInfoPlane info;
  SoBufferObject* dummyPtr = NULL;

  info = g_volData->getLdmDataAccess().getData( g_toggleResolution, box, plane, dummyPtr );

  // setup Slice VertexProperties
  if (info.numPolygonPoints > 0)
  {
    SbVec3i32 *quad = info.quadCoord;
    SbVec2f u( (float)(quad[1][info.uAxis]-quad[0][info.uAxis]), (float)(quad[1][info.vAxis]-quad[0][info.vAxis]) );
    SbVec2f v( (float)(quad[2][info.uAxis]-quad[1][info.uAxis]), (float)(quad[2][info.vAxis]-quad[1][info.vAxis]) );
    u /= u.dot(u);
    v /= v.dot(v);

    SbVec3f vertices[6];
    SbVec2f texcoord[6];

    for ( int i = 0; i < info.numPolygonPoints; i++ ) 
    {
      vertices[i][0] = (float)info.polygonCoord[i][0];
      vertices[i][1] = (float)info.polygonCoord[i][1];
      vertices[i][2] = (float)info.polygonCoord[i][2];
      SbVec2f uv( vertices[i][info.uAxis]-quad[0][info.uAxis], vertices[i][info.vAxis]-quad[0][info.vAxis] );
      texcoord[i][0] = uv.dot(u);
      texcoord[i][1] = uv.dot(v);
    }

    g_sliceVertexProperty->vertex.setValues(0, info.numPolygonPoints, vertices);
    g_sliceVertexProperty->texCoord.setValues(0, info.numPolygonPoints, texcoord);

    g_view2TexturePolygon->vertexProperty = g_sliceVertexProperty;
    g_view2TexturePolygon->numVertices.set1Value( 0, info.numPolygonPoints );
    g_textureQuad->vertexProperty = g_sliceVertexProperty;
  }

  // Allocate input buffer if needed
  if ( (g_inputBufferObject == NULL) || (info.bufferSize> (int)g_inputBufferObject->getSize()) )
  {
    SO_UNREF_RESET(g_inputBufferObject);

    g_inputBufferObject = g_compute->createBufferObject( (size_t)info.bufferSize );
  }

  g_computeParameters.width  = info.bufferDimension[0];
  g_computeParameters.height = info.bufferDimension[1];

  // setup dataType label
  SoDialogLabel *label =
    (SoDialogLabel*)g_topLevelDialog->searchForAuditorId(SbString("label_size"));
  if (label)
  {
    SbString strLabel;
    strLabel = "Slice size: ";
    strLabel += g_computeParameters.width;
    strLabel += " x ";
    strLabel += g_computeParameters.height;
    label->label.setValue(strLabel);
  }

  // really get the Slice data from LDM
  if ( info.bufferSize > 0 )
  {
    // get data 
    info = g_volData->getLdmDataAccess().getData( g_toggleResolution, box, plane, g_inputBufferObject );
    // update compute
    updateCompute();
  }
}


//------------------------------------------------------------------------------
// Animation Sensor Callback
void animateCB(void* /*userData*/, SoSensor*)
{
  int32_t slice = g_orthoSlice->sliceNumber.getValue();
  int32_t step  = (1<<g_toggleResolution) * g_animateDirection;

  if ( (slice + step) >= g_volData->data.getSize()[ g_orthoSlice->axis.getValue() ] || (slice + step) < 0 )
    g_animateDirection = -g_animateDirection;
  
  step  = (1<<g_toggleResolution) * g_animateDirection;
  slice += step;

  SoDialogIntegerSlider *slider1 =
    (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("slice"));

  if (slider1)
  {
    slider1->value.setValue(slice);
    // seems that there is a bug :
    // setting value of slider should trigger the auditor
    // and then change the sliceNumber...but..we have to force it...
    g_orthoSlice->sliceNumber.setValue(slice);
  }
}


//------------------------------------------------------------------------------
//main function
int main(int argc, char **argv)
{
  SoPreferences::setBool( "OIV_IMAGE_USE_GL_BUFFER_OBJECT", TRUE );


  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize VolumeViz and DialogViz extensions
  SoVolumeRendering::init();
  SoDialogViz::init();

  if ( argc >= 2 ) SoPreferences::setString("LDM_INPUT", argv[1]);
  if ( argc >= 3) SoPreferences::setInt("LDM_MAINMEM",atoi(argv[2]));
  if ( argc >= 4) SoPreferences::setInt("LDM_TEXMEM",atoi(argv[3]));

  SbString inputFileName = SoPreferences::getString("LDM_INPUT",SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/colt-float.ldm"));
  if ( inputFileName == "" )
  {
    SoError::post(
      "You must define the input file name using\n"
      "LDM_INPUT environment variable\n"
    );
    exit(0);
  }

  int MaxMainMem = SoPreferences::getInt("LDM_MAINMEM",100);
  int MaxTexMem = SoPreferences::getInt("LDM_TEXMEM",100);

  SoLDMGlobalResourceParameters::setMaxMainMemory(MaxMainMem);
  SoLDMGlobalResourceParameters::setMaxTexMemory(MaxTexMem);

  fprintf(stdout,"LDM_INPUT set to %s\n",inputFileName.getString());
  fprintf(stdout,"LDM_MAINMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxMainMemory());
  fprintf(stdout,"LDM_TEXMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxTexMemory());

  g_computeParameters.dumpToPng = false;

  setComputeInterface();

  // Node to hold the volume data
  g_volData = new SoVolumeData();
  g_volData->fileName = inputFileName;

  //getting data is faster in ALWAYS mode
  g_volData->ldmResourceParameters.getValue()->loadPolicy.setValue(SoVolumeData::SoLDMResourceParameters::ALWAYS);


  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Use a predefined colorMap with the SoTransferFunction
  g_transFunc = getTransferFunction(ColorTablesFile.toLatin1(),"Seismic");

  SoDataRange* pDataRange = new SoDataRange;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
  SbVec3i32 dimensions = g_volData->data.getSize();

  g_ROIManip = new SoROIManip();
  g_ROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  g_ROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  g_ROIManip->constrained = TRUE;
  g_ROIManip->boxOn = FALSE;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;

  // Node in charge of Ortho Slice
  g_orthoSlice = new SoOrthoSlice;
  g_orthoSlice->axis = SoOrthoSlice::Z;
  g_orthoSlice->sliceNumber = g_volData->data.getSize()[2] / 2;
  g_orthoSlice->clipping.setValue( TRUE );

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  root->addChild( eventCB );
  // Volume Bounding box
  root->addChild( makeVolBBox( g_volData->extent.getValue() ) ); 
  // Volume Data and manip
  root->addChild( g_volData );
  root->addChild( g_ROIManip );
  root->addChild( pDataRange );

  root->addChild( g_transFunc );

  g_displaySwitch = new SoSwitch;
  g_displaySwitch->addChild( g_orthoSlice );
  g_displaySwitch->addChild( pVolRender );
  g_displaySwitch->whichChild.setValue( 0 );

  root->addChild( g_displaySwitch );

  // Create a simple user interface to select attributes
  Widget parent = buildInterface( myWindow, g_volData );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(parent);
  myViewer1->setSceneGraph(root);
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  SoXtExaminerViewer* myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("Computed Attribute");
  myViewer2->setBackgroundColor( SbColor(0.4f,0.4f,0.4f) );
  myViewer2->setSceneGraph( viewer2_SceneGraph() );
  myViewer2->show();
  myViewer2->viewAll();

  // Animation Sensor
  g_animationSensor = new SoTimerSensor;
  g_animationSensor->setFunction(animateCB);
  g_animationSensor->setData(NULL);
  g_animationSensor->setInterval(SbTime(0.001));

  // feed backs
  SoFieldSensor *roiSensor = new SoFieldSensor( (SoSensorCB *)&updateView2, NULL );
  roiSensor->attach( &(g_ROIManip->subVolume) );

  SoFieldSensor *sliceSensor = new SoFieldSensor( (SoSensorCB *)&updateView2, NULL );
  sliceSensor->attach( &(g_orthoSlice->sliceNumber) );

  SoXt::show( myWindow );

  updateView2(NULL);

  SoXt::mainLoop();

  g_animationSensor->unschedule();

  delete myViewer1;
  delete myViewer2;
  root->unref();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}


//------------------------------------------------------------------------------
class myAuditorClass : public SoDialogAuditor
{
  void dialogIntegerSlider(SoDialogIntegerSlider* cpt);
  void dialogComboBox     (SoDialogComboBox* cpt);
  void dialogPushButton   (SoDialogPushButton* cpt);
  void dialogCheckBox     (SoDialogCheckBox* cpt);
};


//------------------------------------------------------------------------------
void
myAuditorClass::dialogIntegerSlider(SoDialogIntegerSlider* cpt)
{
  // Change resolution of data
  if (cpt->auditorID.getValue() == "resolution")
  {
    int value = cpt->value.getValue();
    if ( value != g_toggleResolution )
    {
      g_toggleResolution = value;
      updateView2(NULL);
    }
  }

  // Change slice number
  else if (cpt->auditorID.getValue() == "slice")
  {
    int value = cpt->value.getValue();
    g_orthoSlice->sliceNumber = value;
  }

  // Change data Min
  else if (cpt->auditorID.getValue() == "data_min")
  {
    int value = cpt->value.getValue();
    g_currentMinPercent = value;
    updateCompute();
  }

  // Change data Min
  else if (cpt->auditorID.getValue() == "data_max")
  {
    int value = cpt->value.getValue();
    g_currentMaxPercent = value;
    updateCompute();
  }

  // Change data Min
  else if (cpt->auditorID.getValue() == "src_data_min")
  {
    int value = cpt->value.getValue();
    g_transFunc->minValue = (int)(value * 2.55);
  }

  // Change data Min
  else if (cpt->auditorID.getValue() == "src_data_max")
  {
    int value = cpt->value.getValue();
    g_transFunc->maxValue = (int)(value * 2.55);
  }
  // LDM main memory
  else if (cpt->auditorID.getValue() == "ldmMaxMainMemory")
  {
    int value = cpt->value.getValue();
    SoLDMGlobalResourceParameters::setMaxMainMemory( value );
  }
}


//------------------------------------------------------------------------------
void
myAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
  // Select attribute to compute
  if (cpt->auditorID.getValue() == "attribute")
  {
    int value = cpt->selectedItem.getValue();
    if ( value != g_toggleAttribute )
    {
      g_toggleAttribute = value;
      updateView2( NULL );
    }
  }
  // Select displayed object
  else if ( cpt->auditorID.getValue() == "display")
  {
    int value = cpt->selectedItem.getValue();
    // None
    switch ( value )
    {
      case 0: g_displaySwitch->whichChild.setValue(SO_SWITCH_NONE); break;
      case 1: g_displaySwitch->whichChild.setValue(0); break;
      case 2: g_displaySwitch->whichChild.setValue(1); break;
      case 3: g_displaySwitch->whichChild.setValue(SO_SWITCH_ALL); break;
    }
  }
}


//------------------------------------------------------------------------------
void
myAuditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
  // manage slice animation animation
  if (cpt->auditorID.getValue() == "animation")
  {
    // start animation
    if ( cpt->buttonLabel.getValue() == "Start Animation" )
    {
      cpt->buttonLabel.setValue("Stop Animation");
      g_animationSensor->schedule();
    }
    // stop animation
    else
    {
      cpt->buttonLabel.setValue("Start Animation");
      g_animationSensor->unschedule();
    }
  }

  // dump to png push button
  if (cpt->auditorID.getValue() == "dump")
  {
    g_computeParameters.dumpToPng = true;
    updateCompute();
  }
}

//------------------------------------------------------------------------------
void
myAuditorClass::dialogCheckBox(SoDialogCheckBox* cpt)
{
  if (cpt->auditorID.getValue() == "transparentBorders")
  {
    g_transparentBorders->value = cpt->state.getValue();

    updateView2( NULL );
  }

  if (cpt->auditorID.getValue() == "dataOutline")
    SoLDMGlobalResourceParameters::setVisualFeedbackParam( SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, cpt->state.getValue() );
}


//------------------------------------------------------------------------------
Widget
buildInterface(Widget window, SoVolumeData* /*pVolData*/)
{
  SoInput myInput;
  if (! myInput.openFile( g_guiFilename.toLatin1() ))
    return NULL;

  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return NULL;

  g_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  myAuditorClass *myAuditor = new myAuditorClass;
  g_topLevelDialog->addAuditor(myAuditor);

  // setup slice slider range
  SoDialogIntegerSlider *slider1 =
    (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("slice"));
  if (slider1)
  {
    SbVec3i32 voldim  = g_volData->data.getSize();
    slider1->min = 0;
    slider1->max = voldim[2] - 1;
    slider1->value = voldim[2] / 2;
  }

  // setup resolution slider range
  SoDialogIntegerSlider *slider2 =
    (SoDialogIntegerSlider *)g_topLevelDialog->searchForAuditorId(SbString("resolution"));
  if (slider2)
  {
    slider2->min = 0;
    slider2->max = g_volData->getLDMTopoOctree()->getLevelMax();

    g_toggleResolution = slider2->value.getValue();
  }

  // setup dataType label
  SoDialogLabel *label =
    (SoDialogLabel*)g_topLevelDialog->searchForAuditorId(SbString("label_type"));
  if (label)
  {
    SbString strLabel;
    switch ( g_volData->getDataType() )
    {
    case SoVolumeData::UNSIGNED_BYTE  : strLabel= "UNSIGNED_BYTE"; break;
    case SoVolumeData::UNSIGNED_SHORT : strLabel= "UNSIGNED_SHORT"; break;
    case SoVolumeData::UNSIGNED_INT32 : strLabel= "UNSIGNED_INT32"; break;
    case SoVolumeData::SIGNED_BYTE    : strLabel= "SIGNED_BYTE"; break;
    case SoVolumeData::SIGNED_SHORT   : strLabel= "SIGNED_SHORT"; break;
    case SoVolumeData::SIGNED_INT32   : strLabel= "SIGNED_INT32"; break;
    case SoVolumeData::FLOAT          : strLabel= "FLOAT"; break;
    case SoVolumeData::DOUBLE         : strLabel= "DOUBLE"; break;
    };
    label->label.setValue(strLabel);
  }

  SoDialogCustom *customNode = (SoDialogCustom *)g_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  g_topLevelDialog->buildDialog( window, customNode != NULL );
  g_topLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}


//------------------------------------------------------------------------------
SoSeparator *makeVolBBox( const SbBox3f volSize )
{
  // The box will be easier to see without lighting and with wide lines
  SoLightModel *pLModel = new SoLightModel;
  pLModel->model = SoLightModel::BASE_COLOR;

  SoDrawStyle *pStyle = new SoDrawStyle;
  pStyle->lineWidth = 2;

  // The box should be unpickable so manip can be used inside it
  SoPickStyle *pPickable = new SoPickStyle;
  pPickable->style = SoPickStyle::UNPICKABLE;

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  volSize.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty *pProp = new SoVertexProperty;
  pProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  pProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  pProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  pProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  pProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  pProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  pProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  pProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );
  pProp->orderedRGBA.set1Value( 0, 0xFF0000FF );

  // Draw it with a line set
  int coordIndices[] = {0,1,2,3,0,-1,4,5,6,7,4,-1,
                        0,4,-1, 1,5,-1, 2,6,-1, 3,7};
  int numCoordIndices = sizeof(coordIndices)/sizeof(int);
  SoIndexedLineSet *pLines = new SoIndexedLineSet;
  pLines->vertexProperty = pProp;
  pLines->coordIndex.setValues( 0, numCoordIndices, coordIndices );

  // Assemble scene graph
  SoSeparator *pBoxSep = new SoSeparator;
  pBoxSep->addChild( pLModel );
  pBoxSep->addChild( pPickable );
  pBoxSep->addChild( pStyle );
  pBoxSep->addChild( pLines );
  return pBoxSep;
}

//------------------------------------------------------------------------------

