/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : M. Heck (Sep 2007)
**=======================================================================*/

#include "LdmMemoryReader.h"

#include <Inventor/errors/SoMemoryError.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/LDM/SoLDMTileID.h>

#include <LDM/SoLDMTopoOctree.h>

#include <stdio.h>  // for sprintf

const int DEBUG_TRACE = 0;

#ifdef WIN32
#  define atoll _atoi64
#endif

////////////////////////////////////////////////////////////////////////
LdmMemoryReader::LdmMemoryReader()
{ 
  m_initialized = FALSE;
  
  m_size.setBounds(-1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f);
  m_dim.setValue(0,0,0); 
  m_type = SoVolumeData::UNSIGNED_BYTE; 
  m_bytesPerVoxel = 1; 
  m_numSignificantBits = 8;

  // Pointer to "empty" tile
  // (data returned when an undefined tile is requested)
  m_hole = NULL;

  // Memory reader specific variables
  m_tileMap       = NULL;
  m_pTopoOctree   = NULL;
  m_numFileIds    = 0;
  m_volumeData    = NULL;
  m_hole          = NULL;
  m_memUsed       = 0;
  m_tilesCreated  = 0;
}

////////////////////////////////////////////////////////////////////////
LdmMemoryReader::~LdmMemoryReader() 
{
  // Delete all tile storage
  if (m_tileMap) {
    for (unsigned int i = 0; i < m_numFileIds; ++i) {
      if (m_tileMap[i])
        delete [] m_tileMap[i];
    }
  }

  // Delete tile map
  delete [] m_tileMap;
  m_tileMap = NULL;

  // Delete "empty" tile
  delete [] m_hole;

  // Release the volume data node
  if (m_volumeData) {
    m_volumeData->unref();
    m_volumeData = NULL;
  }

  delete m_pTopoOctree;

  m_initialized = FALSE;
}

////////////////////////////////////////////////////////////////////////
int
LdmMemoryReader::initialize( const SoVolumeData *pVolData )
{ 
  if (! pVolData) {
    fprintf( stderr, "*** NULL volume data pointer\n" );
    return -1;
  }

  // Save volume address and ref it to be safe
  m_initialized = TRUE;
  m_volumeData  = const_cast<SoVolumeData*>(pVolData);
  m_volumeData->ref();

  // Get characteristics of volume
  m_dim  = m_volumeData->data.getSize();
  m_size = m_volumeData->extent.getValue();
  m_type = m_volumeData->getDataType();
  m_bytesPerVoxel = m_volumeData->getDataSize();
  m_numSignificantBits = m_volumeData->numSigBits();
  m_tileSize = m_volumeData->ldmResourceParameters.getValue()->tileDimension.getValue();
  m_volumeData->getMinMax( m_dataMin, m_dataMax );
  m_volumeData->getMinMax( m_dataMinD, m_dataMaxD );

  // Create and initialize topoOctree object
  // Find out how many tiles (potentially) exist in this volume
  m_pTopoOctree = new SoLDMTopoOctree();
  m_pTopoOctree->init( m_dim, m_tileSize[0] );
  m_numFileIds = m_pTopoOctree->getNumFileIDs();

  // Note: Since we don't currently allow storing subsampled tiles it
  //       would be possible to use only the number of full resolution
  //       tiles instead of the total number of tiles.  However the
  //       memory saved in the lookup table would be minor compared to
  //       the memory we're using to store the tiles themselves.
  return 0;
}

////////////////////////////////////////////////////////////////////////
int
LdmMemoryReader::initialize( const SbBox3f & size, const SbVec3i32 & dimension,
                  SoDataSet::DataType dataType, int tileSize)
{
  // Store requested characteristics of volume
  m_dim  = dimension;
  m_size = size;
  m_type = dataType;
  m_bytesPerVoxel = SoDataSet::dataSize( dataType );
  m_numSignificantBits = 0;
  m_tileSize = SbVec3i32(tileSize,tileSize,tileSize);

  // Create and initialize topoOctree object
  // Find out how many tiles (potentially) exist in this volume
  m_pTopoOctree = new SoLDMTopoOctree();
  m_pTopoOctree->init( dimension, tileSize );
  m_numFileIds = m_pTopoOctree->getNumFileIDs();

  m_initialized = TRUE;
  return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Get the characteristics (file header) of the data volume. See SoVolumeData.
SoVolumeReader::ReadError
LdmMemoryReader::getDataChar( SbBox3f &size, SoVolumeData::DataType &type,  SbVec3i32 &dim )
{
  if (m_initialized) {
    size = m_size;
    type = m_type;
    dim  = m_dim;
    return RD_NO_ERROR;
  }
  else {
    return RD_UNKNOWN_ERROR;
  }
}

////////////////////////////////////////////////////////////////////////
//
// Return data for tile with specified fileID
//
// If we actually have data for the requested tile, copy that data.
//
// If we have data for a tile, then m_tileMap[fileID] contains a
// pointer to the data.  Else return the "holeData".
//
// Currently we will never have data for subsampled tiles.
SbBool 
LdmMemoryReader::readTile(int index, unsigned char*&buffer, const SbBox3i32& /*tilePosition*/)
{
  // Number of bytes in a tile
  size_t tileBytes = m_tileSize[0]*m_tileSize[1]*m_tileSize[2]*m_bytesPerVoxel;

  // If we don't have any data for this tile, return the "holeData"
  if (m_tileMap == NULL ||
      index < 0 ||
      (size_t)index >= m_numFileIds ||
      m_tileMap[index] == NULL) {

    if (DEBUG_TRACE)
      printf( "readTile: %d <no data>\n", index );

    unsigned char *pHoleData = getHoleData();
    if (pHoleData) {
      memmove( buffer, pHoleData, tileBytes );
      return TRUE;
    }
    else
      return FALSE;
  }

  if (DEBUG_TRACE)
    printf( "readTile: %d (have data)\n", index );

  // Else return data for this tile
  memmove( buffer, m_tileMap[index], tileBytes );
  return TRUE;
}


////////////////////////////////////////////////////////////////////////
//
unsigned char*
LdmMemoryReader::getHoleData()
{
  // If hole data does not exist yet, create an "empty" tile
  // and initialize its contents to zero.
  if (!m_hole) {
    size_t size = m_tileSize[0] * m_tileSize[1] * m_tileSize[2] * m_bytesPerVoxel;
    m_hole = new unsigned char[size];
    if (!m_hole) {
      char buf[128];
      sprintf( buf, "LdmMemoryReader::getHoleData\n"
        "Failed to allocate: %llu bytes", (unsigned long long)size );
      SoMemoryError::post( buf );
      return NULL;
    }
    memset((void*)m_hole, 0, size);
    m_memUsed += size;
  }
  return m_hole;
}

////////////////////////////////////////////////////////////////////////
// Store a stack of tiles (along the X axis)
//
// Special case interface for seismic data
// Caller guarantees that the data is precisely a stack of tiles,
// Specifically that:
//   - The region begins on a tile boundary in the X direction, and
//   - The region is exactly tileSize voxels in the Y and Z directions.
//
// This is appropriate for a tileSize by tileSize group of traces.
SbBool
LdmMemoryReader::putStackOfTiles( const SbBox3i32& region, SoCpuBufferObject* dataBuffer )
{
  if (dataBuffer == NULL)
    return FALSE;

  // Get bounds of region
  SbVec3i32 regmin, regmax;
  region.getBounds( regmin, regmax );

  // Must be a meaningful region
  if (regmax[0] < regmin[0])
    return FALSE;

  // Initialize tile map if necessary
  if (! m_tileMap) {
    m_tileMap = new unsigned char*[m_numFileIds];
    if (! m_tileMap) {
      char buf[128];
      sprintf( buf, "LdmMemoryReader::putStackOfTiles\n"
        "Failed to allocate tile map: %llu bytes",
        (unsigned long long)m_numFileIds * sizeof(unsigned char*) );
      SoMemoryError::post( buf );
      return FALSE;
    }
    m_memUsed += m_numFileIds * sizeof(unsigned char*);
    memset( m_tileMap, 0, m_numFileIds * sizeof(unsigned char*) );
  }

  // How many traces, voxels and bytes are we working with
  int numVoxelsY      = regmax[1] - regmin[1] + 1;
  int numVoxelsZ      = regmax[2] - regmin[2] + 1;
  int numTraces       = numVoxelsY * numVoxelsZ;
  int numVoxelPerTile = m_tileSize[0];
  int numVoxelInStack = regmax[0] - regmin[0] + 1;
  int numBytesPerTile = numVoxelPerTile * m_bytesPerVoxel;
  int numBytesInStack = numVoxelInStack * m_bytesPerVoxel;

  // Figure out what tiles are affected by this stack
  //
  // Note: We know this is a stack of tiles in X direction and at least
  //       for now we know Y and Z are exactly on tile boundaries.
  // For example, implies regmin[0] is always zero.
  // Does not guarantee that Y span and Z span are always tileSize!
  // The boundary tiles are normally incomplete.
  //
  // We can't just use NFM->getTiles because when there is overlap, it
  // returns all the neighbor (1 voxel overlap) tiles.
  // (This should be fixed in OIV 7.1)
  
  // Start with the first voxel inside the region.
  SbVec3i32 voxpos = regmin;
  int xmax = regmax[0];
  int xinc = m_tileSize[0];

  if (DEBUG_TRACE)
    printf( "writeStack: " );

  // Step down the stack, querying one voxel in each tile.
  // If no overlap, then stepping by tileSize gets us the first
  // voxel in each tile in the stack.
  // If there is overlap, then step by tileSize-1, but always
  // add 1 more to avoid the (ambiguous) overlap voxel.
  int itile = 0;
  void * data = dataBuffer->map(SoBufferObject::READ_ONLY);
  do {
    // Get id of tile containing this voxel
    SoLDMTileID tileId = m_pTopoOctree->getTileID( voxpos, 0 /* res */ );
    // Get fileID of this tile
    int fileId = m_pTopoOctree->getFileID( tileId );

    if (DEBUG_TRACE)
      printf( " %d", fileId );
    if (DEBUG_TRACE > 1) {
      SbBox3i32 tileBox;
      SbVec3i32 tileMin, tileMax;
      tileBox = m_pTopoOctree->getTilePos( tileId );
      tileBox.getBounds( tileMin, tileMax );
      printf( "voxpos %2d %2d %2d = tileid %2d fileid %2d : %d %d %d -> %d %d %d\n",
        voxpos[0],voxpos[1],voxpos[2], (int)tileId.getID(), fileId,
        tileMin[0],tileMin[1],tileMin[2], tileMax[0],tileMax[1],tileMax[2] );
    }

    // Create tile storage if necessary
    if (m_tileMap[fileId] == NULL) {
      size_t numBytes = m_tileSize[0] * m_tileSize[1] * m_tileSize[2] * m_bytesPerVoxel;
      m_tileMap[fileId] = new unsigned char[numBytes];
      if (! m_tileMap[fileId]) {
        char buf[128];
        sprintf( buf, "LdmMemoryReader::putStackOfTiles\n"
          "Failed to allocate tile data: %llu bytes", (unsigned long long)numBytes );
        SoMemoryError::post( buf );
        return FALSE;
      }
      m_memUsed += numBytes;
      m_tilesCreated++;
      memset( m_tileMap[fileId], 0, numBytes ); // Initialize tile values
    }

    // Setup to copy tile data
    // The best we can do is tileSize^2 calls to memcpy.
    // We'd like to copy the tile with one call but the data comes in as
    // a contiguous block with all the values in the first trace followed
    // by all the values in the second trace, etc.  We need each tile to
    // be a contiguous block with tileSize values from the first trace
    // followed by tileSize values from the second trace and so on.
    int xbeg = itile * xinc;
    int xend = xbeg + m_tileSize[0] - 1;
    if (xend > xmax)
      xend = xmax;
    int numBytesToCopy = (xend - xbeg + 1) * m_bytesPerVoxel;
    unsigned char *pSrc = (unsigned char*)data + (itile * xinc * m_bytesPerVoxel);
    unsigned char *pDst = m_tileMap[fileId];

    // Copy tile data
    for (int iTrace = 0; iTrace < numTraces; ++iTrace) {
      memcpy( pDst, pSrc, numBytesToCopy );
      pSrc += numBytesInStack;
      pDst += numBytesPerTile;
    }

    // Increment voxel position
    itile++;
    voxpos[0] = (itile * xinc) + 1;
  } while (voxpos[0] <= xmax);
  dataBuffer->unmap();

  if (DEBUG_TRACE)
    printf( "\n" );

  return TRUE;
}

////////////////////////////////////////////////////////////////////////
bool LdmMemoryReader::isLittleEndian()
{
  if (m_initialized)
    return m_littleEndian;
  return false;
}

////////////////////////////////////////////////////////////////////////
SbBool LdmMemoryReader::getMinMax(int& min, int& max)
{
  if (m_initialized) {
    if (m_isDataFloat) {
      min = (int)m_dataMinD;
      max = (int)m_dataMaxD;
    }
    else {
      min = (int)m_dataMin;
      max = (int)m_dataMax;
    }
    return TRUE;
  }
  return FALSE;
}

////////////////////////////////////////////////////////////////////////
SbBool LdmMemoryReader::getMinMax(int64_t & min, int64_t & max)
{
  if (m_initialized) {
    if (m_isDataFloat) {
      min = (int64_t)m_dataMinD;
      max = (int64_t)m_dataMaxD;
    }
    else {
      min = m_dataMin;
      max = m_dataMax;
    }
    return TRUE;
  }
  return FALSE;
}

////////////////////////////////////////////////////////////////////////
SbBool LdmMemoryReader::getMinMax(double & min, double & max)
{
  if (m_initialized) {
    if (m_isDataFloat) {
      min = m_dataMinD;
      max = m_dataMaxD;
    }
    else {
      min = (double)m_dataMin;
      max = (double)m_dataMax;
    }
    return TRUE;
  }
  return FALSE;
}


