/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : M. Heck (Sep 2007)
**=======================================================================*/
#ifndef  _LDM_MEM_READER_
#define  _LDM_MEM_READER_

#include <LDM/readers/SoVolumeReader.h>
#include <VolumeViz/nodes/SoVolumeData.h>

class SoLDMTopoOctree;

/**
 * @VREXT LDM memory reader.
 * 
 * @ingroup VolumeVizReaders
 * 
 * @DESCRIPTION
 *
 * This class implements the LDM volume reader interface.
 * It can be used to supply data to an SoVolumeData node in LDM mode.
 *
 * The purpose of this class is to allow creating an LDM volume in
 * system memory, i.e. without writing any data to disk.  This is
 * useful in order to visualize a computed volume before committing
 * the data to disk (which takes much longer than creating a memory
 * volume).  It is also useful for visualizing a computed subvolume
 * before committing to computation of the whole volume.  It is not
 * necessary to create every tile in the volume.  Tiles that are not
 * defined are considered to be filled with a constant value (zero by
 * default - see the getHoleData method).
 *
 * The maximum size of the volume is, of course, limited by the
 * available system memory.
 *
 * The volume can be initialized by passing a pointer to an existing
 * SoVolumeData object.  This is the most convenient method if you are
 * creating a new volume with the same characteristics as the "source"
 * volume.  The volume can also be initialized using specific values
 * for dimension, size, tileSize, etc.
 *
 * Blocks of data can added to the volume by calling the putStackOfTiles
 * method with a region and a block of data.  In the current (limited)
 * implementation the region must be a "stack" of tiles along the volume
 * X axis. In other words:
 *   1. The region begins on a tile boundary in the X direction, and
 *   2. The Y and Z dimensions of the region are exactly tileSize (e.g. 64).
 *
 * This is suitable for seismic data where the volume X axis is the time
 * (or sample) axis and the region is a tileSize x tileSize block of
 * complete traces. The class could be extended to support arbitrary regions.
 *
 * Currently it is only possible to store full resolution data.  All
 * tiles at subsampled levels of the hierarchy are considered to be filled
 * with a constant value (zero by default).  This is not a problem for
 * visualizing a small volume or a small subvolume of a large volume.
 *
 * If you use the initialize method that takes an exising volume data
 * object, please note that LDMMemoryReader will increment (in initialize)
 * and decrement (in the destructor) the reference count of this object.
 * If decrementing reduces the count to zero, the object is destroyed.
 * 
 * 
 */ 

class LdmMemoryReader : public SoVolumeReader  {

 public:
  /** Constructor. */
  LdmMemoryReader();

  /** Destructor. */
  virtual ~LdmMemoryReader();

  /**
   * Initialize from an existing volume 
   */
  int initialize( const SoVolumeData *pVolData );

  /** 
   * Initialize with specific parameters 
   */
  int initialize( const SbBox3f & size, const SbVec3i32 & dimension,
                  SoDataSet::DataType dataType, int tileSize = 64);

  /**
   * Store a stack of tiles identified by a region
   *
   * Only implemented for a stack along the volume X axis, therefore
   * caller guarantees that the region is exactly tileSize voxels in
   * the Y and Z directions.  
   */
  virtual SbBool putStackOfTiles( const SbBox3i32& region, SoCpuBufferObject* dataBuffer );

  /**
   * How much memory has the reader used so far (to store data)
   */
  int64_t getMemoryUsed() const { return m_memUsed; }

  /**
   * How many tiles has the reader created so far
   */
  int64_t getTilesCreated() const { return m_tilesCreated; }

  /**
   * This member method is inherited from SoVolumeReader
   * and tells VolumeViz that we are an LDM volume
   */
  virtual SbBool isDataConverted() const { return TRUE; }

  // The following methods are normally not important for the application.
  // They are used by VolumeViz to interface to the data.
 SoINTERNAL public:

  /**
   * Gets the characteristics (file header) of the data volume. See SoVolumeData.
   * @B size@b is the size of the actual volume. @B type@b is the type of the data.
   * @B dim@b is the dimension of the data.
   */
  ReadError getDataChar( SbBox3f &size, SoDataSet::DataType &type,  SbVec3i32 &dim );

  /**
   * 
   * Given an index, reads a tile if the data is organized in tiles (for LDM).
   * In the default LDM architecture, the LDM data is based on an octree
   * topology (see SoLDMFileReader). The index passed is 0 for the tile
   * of lowest resolution representing the entire volume (octree root node).
   * The index increments linearly going down through the octree.
   *
   * Indexing works as follows:
   *
   * Tile 1 is the lower back left corner of the cube.
   * The index increments on X, then Y, and finally Z.
   * So the back tiles are:@BR
   *
   * 3  4@BR
   * 1  2
   *
   * And the front tiles are:@BR
   * 
   * 7  8@BR
   * 5  6
   *
   * The tiles of full resolution are the leaf tiles.
   *
   * @B index@b specifies a fileID, the id of an existing tile (fileID=tileID in a cubical volume).
   * @B tilePosition@b specifies the position of the data in the associated volume data of the tile
   * corresponding to the given index. In the default SoVRLdmFileReader, the tilePosition isn't actually
   * used but it is passed as a convenience for customized reader (can be used for mapping to a
   * different index scheme).
   * The data is returned in the allocated buffer.
   */
  virtual SbBool readTile(int index, unsigned char*&buffer, const SbBox3i32& tilePosition);

  /**
   * Returns the minimum and maximum data values (obsolete).
   */
  virtual SbBool getMinMax(int & min, int & max);

  /**
   * Returns the minimum and maximum data values.
   */
   virtual SbBool getMinMax(int64_t & min, int64_t & max);

  /**
   * Returns the minimum and maximum data values (for float values).
   */
  virtual SbBool getMinMax(double & min, double & max);

  /**
   * 
   * Returns the size of a data tile.
   */
  virtual SbBool getTileSize(SbVec3i32& tile) {tile=m_tileSize; return TRUE;}

  /**
   * 
   * Returns the number of significant bits.
   */
  virtual int getNumSignificantBits() { return m_numSignificantBits; }

  /**
   * Returns whether the bytes of the data are stored in big or little endian order.
   * The order is little endian if the function returns true.
   */
  virtual bool isLittleEndian();

  ReaderType getReaderType() {return LDM;};

  /**
   * Sets user data. Usually this is a filename.
   * Just need a dummy version so VolumeReader doesn't try to open a file.
   */
  virtual void setUserData(void* /*data*/) {}

  /**
   * Lets VolumeViz know if the reader is thread safe or not.
   *
   * This class could easily be thread safe, but this has not been tested.
   */
  virtual SbBool isThreadSafe() { return FALSE; }

  /**
   * Returns the data used to render a hole (a tile which does not exist in the file).
   * This may occur when the original file was not fully converted.
   * By default, a tile of all zeros is returned.
   */
  virtual unsigned char* getHoleData();

  /**
   *  
   * Returns the distribution of data values, i.e., the number of voxels per data value.
   * numVox(0) is the number of voxels with value 0, numVox(1) is the number of voxels
   * with value 1, and so on.
   */
  virtual SbBool getHistogram(std::vector<int64_t>& /*numVox*/) { return FALSE; }

  //returns histo and values. bool returned whether some values are set or not
  virtual SbBool getHistogramAndValues(std::vector<int64_t>& /*numVox*/, std::vector<double>& /*values*/)
    { return FALSE; }

  /**
   * Method required by VolumeViz 3.0.
   * Must copy the rectangular part defined by @B subSlice @b of the XY slice @B sliceNumber @b
   * to the memory referenced by @B data@b.
   * Slices will not always be read sequentially.
   *
   * You can use the convenience method GetBuffer() to read data from file. Note:
   * SetFilename() must have been called previously.
   */
  virtual void getSubSlice( const SbBox2i32& /*subSlice*/, int /*sliceNumber*/, void* /*data*/ ) {};

 protected:

   SbBool m_initialized; // Characteristics set?

   size_t          m_numFileIds; // How many tiles actually exist
   unsigned char **m_tileMap;    // Pointers to tiles indexed by fileId

   // Volume characteristics
   SbVec3i32 m_tileSize;
   int m_numSignificantBits;

   unsigned char* m_hole;
   bool  m_littleEndian;
   bool  m_isDataSigned;
   bool  m_isDataFloat;

   SbBox3f   m_size;
   SbVec3i32 m_dim;
   SoDataSet::DataType m_type;

   // Data min/max
   int64_t m_dataMin , m_dataMax;
   double  m_dataMinD, m_dataMaxD;

   SoVolumeData    *m_volumeData;    // Volume we were initialized from
   SoLDMTopoOctree *m_pTopoOctree;   // For querying tile ids, etc.

   int64_t m_memUsed;       // Counts memory allocated (in bytes)
   int64_t m_tilesCreated;  // Counts tiles created
};

#endif // _LDM_MEM_READER_


