/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Mike Heck (Dec 2007)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
 * LoadData
 * Example: Load LDM data for computation
 *
 * Please see the associated programming guide document:
 *    PG-DataManagementVolVizLDM.pdf
 * located in the SDK directory .../examples/source/VolumeViz/Compute
 *
 * Note this example is optimized to load a data volume, e.g. seismic
 * amplitudes, already in LDM format.  The default data file below is
 * a SEGY file which is provided with the Open Inventor SDK.  The
 * example still works because VolumeViz will automatically convert
 * the data to LDM format "on the fly".  However this conversion will
 * reduce performance compared to loading an LDM format file.
 *
----------------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include <cinttypes>

#if !defined(__APPLE__)
#  include <malloc.h>
#endif

#include <Inventor/SoDB.h> 
#include <Inventor/SbElapsedTime.h> 

#include <VolumeViz/nodes/SoVolumeData.h> 
#include <VolumeViz/readers/SoVRSegyFileReader.h>

#include <Inventor/helpers/SbFileHelper.h>
////////////////////////////////////////////////////////////////////////
// Constants

// Default data file
#define VOLUMEFILE "$OIVHOME/examples/data/VolumeViz/Waha8.sgy"

// Enable more verbose output (will affect processing time)
static int debugFlag = 1;

// Estimated number of values we can compute per second
// (used to simulate computation time)
const float COMPUTE_PER_SEC = 5e6f;

// Data alignment requirement in bytes
// Typically 32 or 64 for cache lines, may be 128 for DMA transfers
const size_t ALIGNMENT = 64;

SbString outputFileName;
FILE* outputFILE = NULL;

////////////////////////////////////////////////////////////////////////
// Local functions

// Pretty print numbers for benchmark results
char *formatNumberApproximate( double _value, char *_buffer );

// Do the computation (whatever it is) on one block of data
int doComputation( SoCpuBufferObject* pSrcBuffer, SoCpuBufferObject* pDstBuffer,
                   SbVec3i32 actDim, SoVolumeData::DataType dataType );

////////////////////////////////////////////////////////////////////////

int main( int argc, char **argv )
{
  // Get filename and check that it exists
  const char *filename = NULL;
  
  int iarg;

  if (argc > 1)
  {
    filename = argv[1];
    
    for ( iarg = 2; iarg < argc; iarg++ ) 
    {
      if (argv[iarg][0] == '-') 
      {
        char option = argv[iarg][1];
        switch (option) 
        {
        case 'o':
          iarg++;
          outputFileName = SbString(argv[iarg]);
          outputFILE = SbFileHelper::open(outputFileName.getString(), "a");
          break;
        }
        continue;
      }
    }
  }

 if ( outputFILE == NULL )
    outputFILE = stderr; 

  if (filename == NULL) 
  {
    filename = VOLUMEFILE; // default
  }
  
  FILE *fp = SbFileHelper::open( filename, "r" );
  
  if (!fp) 
  {
    fprintf( stderr, "Unable to open input file '%s'\n", filename );
    return -1;
  }
  
  fclose( fp );

  fprintf(outputFILE, "Input: '%s'\n", filename );

  // Initialize (don't need any window system classes)
  SoDB::init();
  SoVolumeRendering::init();

  // Load the volume
  SoVolumeData *pVolData = new SoVolumeData();
  pVolData->ref();
  pVolData->fileName.setValue( filename );

  // Get some information about the volume
  SbBox3f   volSize  = pVolData->extent.getValue();
  SbVec3i32 volDim   = pVolData->data.getSize();
  int       bytesPV  = pVolData->getDataSize(); // BytesPerValue
  SbVec3i32 tileDim  = pVolData->ldmResourceParameters.getValue()->tileDimension.getValue();
  int       tileSize = tileDim[0];
  SoVolumeData::DataType dataType = pVolData->getDataType();

  fprintf(outputFILE, "Volume dimensions: %d %d %d\n"
          "       bytesPerValue: %d  tileSize: %d\n",
          volDim[0], volDim[1], volDim[2], bytesPV, tileSize );

  // Estimate number of full resolution tiles
  int xtiles = (int)ceil( (float)volDim[0] / tileDim[0]);
  int ytiles = (int)ceil( (float)volDim[1] / tileDim[1]);
  int ztiles = (int)ceil( (float)volDim[2] / tileDim[2]);

  // The number of values in one data block is the number of values in
  // a single tile times the number of tiles in a block, which in this
  // case is the number of tiles on the X axis of the volume.
  //
  // Similarly the number of data blocks we need to process is (in this
  // case) the number of Y tiles times the number of Z tiles.
  unsigned int valuesPerTile  = tileDim[0] * tileDim[1] * tileDim[2];
  unsigned int valuesPerBlock = xtiles * valuesPerTile;
  int64_t valuesTotal = (int64_t)volDim[0] * volDim[1] * volDim[2];
  unsigned int numBlocks = ytiles * ztiles;

  fprintf(outputFILE, "       Values in tile: %d  in block: %d  total: %" PRId64 "\n",
    valuesPerTile, valuesPerBlock, valuesTotal );
  fprintf(outputFILE, "       Tiles in block: %d  in volume: %d\n",
    xtiles, xtiles * ytiles * ztiles );
  fprintf(outputFILE, "       Blocks to process: %d\n", numBlocks );

  // Allocate memory for input data block
  //
  // The number of bytes in one stack is the number of data values
  // times the number of bytes in one value.  Note this number is very
  // conservative because it assumes there are no partial tiles.
  size_t numBytes = valuesPerBlock * bytesPV;
  SoRef<SoCpuBufferObject> pSrcBuffer = new SoCpuBufferObject();
  pSrcBuffer->setSize(numBytes);

  // Allocate memory for output data block
  SoRef<SoCpuBufferObject> pDstBuffer = new SoCpuBufferObject();
  pDstBuffer->setSize(numBytes);

  printf( "Processing data blocks...\n" );

  // Get the data access object and result object
  SoVolumeData::LDMDataAccess &dataAccess = pVolData->getLdmDataAccess();
  SoVolumeData::LDMDataAccess::DataInfoBox dataInfo;

  // Create and initialize timers
  SbElapsedTime loadTimer, compTimer, totalTimer;
  double loadTime, compTime, totalTime;
  loadTime = compTime = totalTime = 0;
  totalTimer.reset();
  int64_t bytesLoaded = 0;
  int     blocksLoaded = 0;

  // Temporary variables for loop
  SbBox3i32 subvol;
  SbVec3i32 actDim;

  // Setup for calculating data blocks on tile boundaries
  //
  // Tiles always contain exactly tileDim voxels.
  // We'll limit the region to the actual volume dimensions.
  //
  // We don't really need to precompute these values, the extra math
  // wouldn't slow down the loop much.  It's just clearer this way.
  int yinc   = tileDim[1];
  int zinc   = tileDim[2];
  int xLimit = volDim[0] - 1;
  int yLimit = volDim[1] - 1;
  int zLimit = volDim[2] - 1;

  // Loop over all stacks (data blocks)
  for (int iz = 0; iz < ztiles; ++iz) {
    for (int iy = 0; iy < ytiles; ++iy) {

      // Compute bounds of this stack of tiles
      loadTimer.reset();
      int xmin = 0;
      int xmax = xLimit;
      int ymin = iy * yinc;
      int ymax = ymin + tileDim[1] - 1;
      if (ymax > yLimit)
        ymax = yLimit;
      int zmin = iz * zinc;
      int zmax = zmin + tileDim[2] - 1;
      if (zmax > zLimit)
        zmax = zLimit;
      subvol.setBounds( xmin, ymin, zmin, xmax, ymax, zmax );

      if (debugFlag == 2)
        printf( "  Loading block %3d of %d : %d %d %d -> %d %d %d\n",
          blocksLoaded+1, numBlocks, xmin, ymin, zmin, xmax, ymax, zmax );
      else if (debugFlag > 0)
        printf( "  %d of %d\r", blocksLoaded+1, numBlocks );

      // Load data
      dataInfo = dataAccess.getData( 0, subvol, pSrcBuffer.ptr() );
      loadTime += loadTimer.getElapsed();

      if (dataInfo.errorFlag != SoVolumeData::LDMDataAccess::CORRECT) {
        printf( "*** ERROR loading block %d... error=%d\n",
          blocksLoaded, dataInfo.errorFlag );
        printf( "    Block %d : %d %d %d -> %d %d %d\n",
          blocksLoaded, xmin, ymin, zmin, xmax, ymax, zmax );
      }
      else {
        // Get the actual dimensions of the data loaded.
        // (Remember that tiles on the edges of the volume may be partial)
        // The compute engine will need this information to avoid
        // computing undefined values.
        actDim = dataInfo.bufferDimension;
        blocksLoaded++;
        bytesLoaded += dataInfo.bufferSize;

        // Do computation
        compTimer.reset();
        doComputation( pSrcBuffer.ptr(), pDstBuffer.ptr(), actDim, dataType );
        compTime += compTimer.getElapsed();
      }
    } // End of inner loop (over Z tiles)
  }

  // Report timing results
  totalTime = totalTimer.getElapsed();
  {
    double valuesLoaded = (double)volDim[0] * volDim[1] * volDim[2];
    double BytesPerSec = (double)bytesLoaded / loadTime;
    double ValsPerSec  = valuesLoaded / loadTime;
    char buf1[80], buf2[80], buf3[80];
    formatNumberApproximate( double(numBytes)   , buf1 );
    formatNumberApproximate( BytesPerSec, buf2 );
    formatNumberApproximate( ValsPerSec , buf3 );
    fprintf(outputFILE, "Data loading: <%g> sec for %sB : %sB/sec (%s-Values/sec)\n",
      loadTime, buf1, buf2, buf3 );
    ValsPerSec = valuesLoaded / compTime;
    formatNumberApproximate( valuesLoaded, buf1 );
    formatNumberApproximate( ValsPerSec, buf2 );
    fprintf(outputFILE, "Computation : <%g> sec for %s-Values : %s-Values/sec\n",
      compTime, buf1, buf2 );
    ValsPerSec = valuesLoaded / totalTime;
    formatNumberApproximate( ValsPerSec, buf2 );
    fprintf(outputFILE, "Total time  : <%g> sec for %s-Values : %s-Values/sec\n",
      totalTime, buf1, buf2 );
  }

  // Cleanup
  pSrcBuffer = NULL;
  pDstBuffer = NULL;

  pVolData->unref();
  SoVolumeRendering::finish();
  SoDB::finish();
  return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Do the computation on one block of data
//
//   pSrcBuffer : input data
//   pDstBuffer : output data
//   actDim     : dimensions of data block
//   dataType   : type of data (int, float, etc)
//
// In this example we're just looking at the actual data loading,
// so the computation doesn't do anything.
// However we simulate some computation time in order to see the
// balance between data loading time and compute time.
// This will be more interesting when we overlap loading and
// computation using the asynchronous access methods.

int doComputation( SoCpuBufferObject* /*pSrcBuffer*/, SoCpuBufferObject* /*pDstBuffer*/,
                   SbVec3i32 actDim, SoVolumeData::DataType /*dataType*/ )
{
  // How many values in this block?
  float numValues = (float)actDim[0] * actDim[1] * actDim[2];

  // Simulate appropriate amount of computation
  float numSec = numValues / COMPUTE_PER_SEC;
  int msec = (int)(numSec * 1000);

  SbTime::sleep( msec );
  return 0;
}

////////////////////////////////////////////////////////////////////////
//
// Reformat big numbers for display
// Static local utility

char *
formatNumberApproximate( double _value, char *_buffer )
{
  const char *units = "";
  double value = _value;
  if (value >= 1.e9) {
    value /= 1.e9;
    units = "G";
  }
  else if (value >= 1.e6) {
    value /= 1.e6;
    units = "M";
  }
  else if (value >= 1.e3) {
    value /= 1.e3;
    units = "K";
  }
  sprintf( _buffer, "<%.2f> %s", value, units );
  return _buffer;
}


