/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Damien DALLA ROSA (MMM yyyy)
**=======================================================================*/


#include <compute.h>

#include <Inventor/devices/SoCpuBufferObject.h>

#include <Inventor/devices/SoCpuContext.h>

#include <Inventor/errors/SoDebugError.h>

#include <Inventor/algorithms/SoAlgorithms.h>

#include <Inventor/algorithms/SoArithmetic.h>
#include <Inventor/algorithms/SoConversion.h>
#include <Inventor/algorithms/SoConvolution.h>
#include <Inventor/algorithms/SoSeismic.h>

#include <Inventor/image/SoPNGImageRW.h> 
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/image/SoRasterImageFile.h> 

Compute::Compute()
{
  // setup algorithm interface
  m_algorithmsInterface = new SoAlgorithms;

  if ( !m_algorithmsInterface )
    return;

  // setup algorithm interface context and sub Modules
  m_deviceContext = m_algorithmsInterface->createContext();

  if ( m_deviceContext.ptr() == NULL )
    return;

  m_deviceContext->bind();
  {
    m_convolutionAlgorithms = m_algorithmsInterface->createConvolutionInterface();
    m_conversionAlgorithms = m_algorithmsInterface->createConversionInterface();
    m_arithmeticAlgorithms = m_algorithmsInterface->createArithmeticInterface();
    m_seismicAlgorithms = m_algorithmsInterface->createSeismicInterface();
  }
  m_deviceContext->unbind();

  m_tmpBufferObject = NULL;
  m_outputBufferObject = NULL;

}

//------------------------------------------------------------------------------  
Compute::~Compute()
{
  if ( !m_algorithmsInterface )
    return;

  m_deviceContext->bind();
  {
    delete m_convolutionAlgorithms;
    delete m_conversionAlgorithms;
    delete m_arithmeticAlgorithms;
    delete m_seismicAlgorithms;

    SO_UNREF_RESET(m_outputBufferObject);
    SO_UNREF_RESET(m_tmpBufferObject);
  }
  m_deviceContext->unbind();

  delete m_algorithmsInterface;
}


//------------------------------------------------------------------------------  
SbBool 
Compute::doCompute( ComputeParameters* parameters )
{

  if ( !m_algorithmsInterface )
    return FALSE;

  int width = parameters->width;
  int height = parameters->height;

  // Setup the output buffer
  size_t outputSize = width * height * 4;

  if ( !m_outputBufferObject || (m_outputBufferObject->getSize() != outputSize) )
  {
    SO_UNREF_RESET(m_outputBufferObject);
    m_outputBufferObject = createBufferObject( outputSize );
  }  

  if ( !m_tmpBufferObject || (m_tmpBufferObject->getSize() != width * height * sizeof(float)) )
  {
    SO_UNREF_RESET(m_tmpBufferObject);
    m_tmpBufferObject = createBufferObject( width * height * sizeof(float) );
  }  
  
  parameters->outputData = m_outputBufferObject;

  // Ok we can compute the rame

  m_deviceContext->bind();
  {
    m_conversionAlgorithms->convert( parameters->inputData, parameters->inputDataType, m_tmpBufferObject, SbDataType::FLOAT, width * height );
     
    if ( parameters->attribute > 0 )
    {
      // compute Hilbert
      m_seismicAlgorithms->computeHilbert( m_tmpBufferObject, m_outputBufferObject, width, height );

      // Apply attribute computation if required
      if ( parameters->attribute == 2 || parameters->attribute == 3 )
      {
        if ( parameters->attribute == 2 )
          m_seismicAlgorithms->computeEnvelope( m_tmpBufferObject, m_outputBufferObject, width, height );
        else if ( parameters->attribute == 3 )
          m_seismicAlgorithms->computeInstPhase( m_tmpBufferObject, m_outputBufferObject, width, height );
      }
    }
    if ( parameters->attribute != 0 )
      m_conversionAlgorithms->convertFloatToRGBA( m_outputBufferObject, m_outputBufferObject, 
                                           width * height, parameters->dataMin, parameters->dataMax );
    else
      m_conversionAlgorithms->convertFloatToRGBA( m_tmpBufferObject, m_outputBufferObject, 
                              width * height, parameters->dataMin, parameters->dataMax );
  }
  m_deviceContext->unbind();
  
  if ( parameters->dumpToPng )
  {
    // We dump the input buffer.
    if ( parameters->inputDataType.getSize() == 1 )
    {
    // Here we dump
    SoCpuBufferObject* bufObj = new SoCpuBufferObject;
    bufObj->ref();

    parameters->inputData->map( bufObj, SoBufferObject::READ_ONLY );
    // - Create an SbRasterImage
    const SbVec2i32 size( width, height );
    SbRasterImage image( size, SbRasterImage::LUMINANCE, bufObj );

    // - Create a file IO object
    SoRasterImageFile imageFile( "dump-input.png" );

    // - Create an image writer and write the image
    SoPNGImageRW imageWriter;
    imageWriter.open( &imageFile, SoRasterImageRW::OPEN_WRITE );
    imageWriter.writeHeader( size );
    imageWriter.write( &image );
    imageWriter.writeFooter();
    imageWriter.close();    

    parameters->inputData->unmap( bufObj );
    bufObj->unref();
    }

    // We dump the RGBA output
    {
    // Here we dump
    SoCpuBufferObject* bufObj = new SoCpuBufferObject;
    bufObj->ref();

    m_outputBufferObject->map( bufObj, SoBufferObject::READ_ONLY );
    // - Create an SbRasterImage
    const SbVec2i32 size( width, height );
    SbRasterImage image( size, SbRasterImage::RGB_TRANSPARENCY, bufObj );

    // - Create a file IO object
    SoRasterImageFile imageFile( "dump-rgba.png" );

    // - Create an image writer and write the image
    SoPNGImageRW imageWriter;
    imageWriter.open( &imageFile, SoRasterImageRW::OPEN_WRITE );
    imageWriter.writeHeader( size );
    imageWriter.write( &image );
    imageWriter.writeFooter();
    imageWriter.close();    

    m_outputBufferObject->unmap( bufObj );
    bufObj->unref();
    }
  }

  return TRUE;
}


//------------------------------------------------------------------------------ 
SoBufferObject* 
Compute::createBufferObject( size_t size )
{
  SoBufferObject* bufferObject = NULL;

  if ( !m_algorithmsInterface )
    return NULL;

  m_deviceContext->bind();
  {
    bufferObject = m_algorithmsInterface->createBufferObject();
    bufferObject->ref();
    bufferObject->setSize( size );
  }
  m_deviceContext->unbind();

  return bufferObject;
}


//------------------------------------------------------------------------------ 

