/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David BEILLOIN (Sep 2008)
**=======================================================================*/

#define BUILDING_DLL 1
#include "SoDataAccessReader.h"

#include <string.h>

#include <Inventor/fields/SoSubFieldContainer.h>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/threads/SbThreadRWMutex.h>
#include <Inventor/SbElapsedTime.h>
#include <Inventor/SbMathHelper.h>
#include <Inventor/helpers/SbDataTypeMacros.h>

#include <LDM/SoLDMTopoOctree.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#if defined(WIN32)
#define strcasecmp _stricmp
#endif

SO_FIELDCONTAINER_SOURCE(SoDataAccessReader);

void SoDataAccessReader::initClass()
{
  SO_FIELDCONTAINER_INIT_CLASS(SoDataAccessReader, "SoDataAccessReader", SoVolumeReader);
}

void SoDataAccessReader::exitClass()
{
  SO__FIELDCONTAINER_EXIT_CLASS(SoDataAccessReader);
}


SoDataAccessReader::SoDataAccessReader()
{
	SO_FIELDCONTAINER_CONSTRUCTOR(SoDataAccessReader);
  SO_FIELDCONTAINER_ADD_FIELD(inputVolumeData,(NULL));
}

SoDataAccessReader::~SoDataAccessReader()
{
}

void SoDataAccessReader::notify(SoNotList *list)
{
  if (list->getLastRec()->getType() == SoNotRec::CONTAINER)
  {
    ////////////////////////////////////////////////////
    // input filename changed 
    if ( list->getLastField() == &inputVolumeData )
    {
      //Test if we are notified because the inputVolumeData changed
      //and not because some fields of inputVolumeData changed
      if ( list->getLastRec() == list->getFirstRec() )
        SoVolumeReader::notify(list);
    }
  }
}

SbBool
SoDataAccessReader::getMinMax(double & min, double & max)
{
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());
  
  if ( vd == NULL )
    return false;

  return vd->getMinMax(min,max);
}

SbBool
SoDataAccessReader::getMinMax(int64_t & min, int64_t & max)
{
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());
  
  if ( vd == NULL )
    return false;

  return vd->getMinMax(min,max);
}


SoVolumeReader::ReadError 
SoDataAccessReader::getDataChar( SbBox3f &size, SoVolumeData::DataType &type, SbVec3i32 &dim )
{
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());

  if ( vd == NULL )
    return RD_INVALID_DATA_ERROR;

  size = vd->extent.getValue();
  type = vd->getDataType();
  dim  = vd->data.getSize();

  return RD_NO_ERROR;
}

SbBool
SoDataAccessReader::getTileSize (SbVec3i32 &size)
{
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());

  if (vd==NULL)
    return FALSE;

  // Retrieve the tile size from the LDM ressource parameters
  size.setValue( vd->ldmResourceParameters.getValue()->tileDimension.getValue() );

  return true;
}

static SoLDMTopoOctree*topo = NULL;

int 
SoDataAccessReader::getTileIndex(const SbVec3i32& cellPos, int resolution )
{
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());

  if ( vd == NULL )
    return -1;

  if ( topo == NULL )
  {
    SbVec3i32 tileDim;
    getTileSize( tileDim );
    topo = new SoLDMTopoOctree;
    topo->init( vd->data.getSize(), tileDim[0]);
  }

  SoLDMTileID id = topo->getTileID(cellPos, resolution);
  return (int)( topo->getFileID(id) );
}

SbBox3i32
SoDataAccessReader::getTilePos(const SoLDMTileID tileID)
{
  SbVec3i32 cellPos;
  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());

  if ( topo == NULL )
  {
    SbVec3i32 tileDim;
    getTileSize( tileDim );
    topo = new SoLDMTopoOctree;
    topo->init( vd->data.getSize(), tileDim[0] );
  }

  return topo->getTilePos( tileID );
}


SoCpuBufferObject*
SoDataAccessReader::readTile (int index, const SbBox3i32& tilePosition)
{

  SbVec3i32 tileSize;
  getTileSize (tileSize);

  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());
  SoCpuBufferObject *tileBuffer = new SoCpuBufferObject();
  
  tileBuffer->setSize(tileSize[0]*tileSize[1]*tileSize[2]*vd->getDataSize());

  if ( vd == NULL )
    return FALSE;
  else
  {
    if ( topo == NULL )
    {
      SbVec3i32 tileDim;
      getTileSize( tileDim );
      topo = new SoLDMTopoOctree;
      topo->init( vd->data.getSize(), tileDim[0] );
    }
  }

  if ( index == 0 )
  {  
    void *temp = tileBuffer->map(SoBufferObject::READ_WRITE);
    memset(temp, 0, tileBuffer->getSize() );
    tileBuffer->unmap();
    return tileBuffer;
  }

  // Level and resolution are inverted inside LDM, so 
  // do the conversion to be sure to be consistent with DataAccess Api
  SoLDMTileID tileId = topo->getTileID(index);
  int resolution = topo->getLevelMax() - topo->level(tileId);

  // Call with NULL buffer (default 3rd parameter) to get number of bytes required to hold data
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  info = vd->getLdmDataAccess().getData( resolution, tilePosition, (SoBufferObject*)NULL );

  // If the tile is included in the whole volume, the DataAccessApi
  // Will return all the data, so just fill the input buffer
  if ( (size_t)info.bufferSize == tileBuffer->getSize() )
  {
    tileBuffer->ref();
    info = vd->getLdmDataAccess().getData( resolution, tilePosition, tileBuffer );
    tileBuffer->unrefNoDelete();
  }
  else
  {
    // Tile might be incomplete as it concerns a boder tile
    // DataAccess Api will just retrieve data contained in the volume
    // We will need to build the tile ourself

    // Match dimension to current resolution
    SbVec3i32 tileMin, tileMax;
    SbVec3i32 clampMin(tilePosition.getMin());
    tilePosition.getBounds(tileMin, tileMax);

    tileMin[0] >>= resolution;
    tileMin[1] >>= resolution;
    tileMin[2] >>= resolution;
    tileMax[0] >>= resolution;
    tileMax[1] >>= resolution;
    tileMax[2] >>= resolution;

    clampMin[0] >>= resolution;
    clampMin[1] >>= resolution;
    clampMin[2] >>= resolution;

    SbVec3i32 clampMax(clampMin + info.bufferDimension - SbVec3i32(1,1,1));

    // Get the data of the intersection (tile/volume)
    SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject;

    cpuBuffer->setSize((size_t)info.bufferSize);
    info = vd->getLdmDataAccess().getData( resolution, tilePosition, cpuBuffer.ptr() );
    void* tileData = cpuBuffer->map(SoBufferObject::READ_ONLY);

    SoRef<SoCpuBufferObject> cpuObj = new SoCpuBufferObject;

    tileBuffer->map(cpuObj.ptr(), SoBufferObject::SET);
    void* mapBuffer = cpuObj->map(SoBufferObject::SET);

    // Fill the whole buffer properly ny 
    // building the tile ourself
    buildTile( 
      SbBox3i32(clampMin, clampMax ), 
      tileData, 
      SbBox3i32(tileMin,tileMax), 
      mapBuffer, 
      SbBox3i32(clampMin, clampMax )
      );

    // Unmap bufferObject
    cpuBuffer->unmap();
    cpuObj->unmap();
    tileBuffer->unmap(cpuObj.ptr());
  }

  if (info.errorFlag != SoVolumeData::LDMDataAccess::CORRECT)
  {
    SoDebugError::post("SoDataAccessReader::readTile","LDMDataAccess::getData failed %d\n", info.errorFlag);
    
    return FALSE;
  }

  return tileBuffer;
}

void SoDataAccessReader::getSubSlice( const SbBox2i32& , int , void * )
{
  fprintf(stderr,"SoDataAccessReader::getSubSlice : Not Implemented\n");
  return;
}

SbBool
SoDataAccessReader::isThreadSafe() const
{
  return TRUE;
}

void
SoDataAccessReader::buildTile( const SbBox3i32& subVolume, const void* subVolumeBuffer, const SbBox3i32& tilePos, void* tileData, const SbBox3i32& intersection )
{
  SbVec3i32 intMin, intMax, tileMin, tileMax, subvMin, subvMax;
  intersection.getBounds( intMin, intMax );
  tilePos.getBounds( tileMin, tileMax );
  subVolume.getBounds( subvMin, subvMax );

  SbVec3i32 subvDim = subvMax - subvMin + SbVec3i32(1, 1, 1);

  SbVec3i32 srcMin = intMin - subvMin;
  SbVec3i32 dstMin = intMin - tileMin;
  SbVec3i32 dstMax = intMax - tileMin;

  SoVolumeData *vd = dynamic_cast<SoVolumeData*>(inputVolumeData.getValue());
  int m_dataSize = vd->getDataSize();
  SbVec3i32 m_tileDim = vd->getTileDimension();
  SbDataType m_dataTypeOut = vd->getDataType();

  unsigned char *src = (unsigned char*)subVolumeBuffer;
  unsigned char *dst = (unsigned char*)tileData;

# define tileOffset(_i,_j,_k) (((_k)*m_tileDim[1] + (_j))*m_tileDim[0] + (_i))
# define subvOffset(_i,_j,_k) (((_k)*subvDim[1] + (_j))*subvDim[0] + (_i))

  for (int kdst = dstMin[2], ksrc = srcMin[2]; kdst <= dstMax[2]; ++kdst, ++ksrc)
  {
    for (int jdst = dstMin[1], jsrc = srcMin[1]; jdst <= dstMax[1]; ++jdst, ++jsrc)
    {
      // optimization : use large memcpy instead of byte per byte copy.
      int dstOffset = tileOffset( dstMin[0], jdst, kdst ) * m_dataSize;
      int srcOffset = subvOffset( srcMin[0], jsrc, ksrc ) * m_dataSize;
      memcpy(
        dst + (dstOffset),
        src + (srcOffset),
        (dstMax[0]-dstMin[0]+1)*m_dataSize
      );
      // Fill the rest of the line of the tile with the last value of the volume
      if ( tileMax[0] > subvMax[0] )
      {
        SB_DATATYPE_CALL ( fillTile, ( dst + (tileOffset( dstMin[0], jdst, kdst ) + dstMax[0]+1)*m_dataSize, (tileMax[0] - tileMin[0]) - ( dstMax[0] - dstMin[0]), dst + (tileOffset( dstMin[0], jdst, kdst ) + dstMax[0])*m_dataSize ), m_dataTypeOut );
      }
    }
  }

  SbVec3i32 tileSize = tileMax - tileMin + SbVec3i32 (1, 1, 1);

  // In the following part:
  //  What's called valid coordinate is 
  //    dstMin[0] <= X <= dstMax[0]
  //    dstMin[1] <= Y <= dstMax[1]
  //    dstMin[2] <= Z <= dstMax[2]
  //
  //  What's called invalid coordinate is 
  //    dstmax[0] + 1 <= X <= tileMax[0]
  //    dstmax[1] + 1 <= Y <= tileMax[1]
  //    dstmax[2] + 1 <= Z <= tileMax[2]


  // Loop for volume with valid Z and invalid Y
  for (int kdst = dstMin[2]; kdst <= dstMax[2]; ++kdst )
  {
    for (int jdst = dstMax[1]+1; jdst < tileSize[1]; ++jdst )
    {
      int dstOffset = ( kdst * tileSize[0] * tileSize[1] + jdst * tileSize[0] )* m_dataSize;
      int srcOffset = ( dstMax[1] * tileSize[0] + kdst * tileSize[0] * tileSize[1] ) * m_dataSize;
  
      // Fill the subvolume with valid X
      memcpy (dst + dstOffset, dst + srcOffset, m_dataSize * (dstMax[0]-dstMin[0]+1) );

      // Fill the subvolume with invalid X
      if ( tileMax[0] > subvMax[0] )
      {
        SB_DATATYPE_CALL ( fillTile, 
                         ( dst + ((dstMax[0]+1) + ( kdst * tileSize[0] * tileSize[1] ) + ( jdst * tileSize[0] ))* m_dataSize, 
                           (tileMax[0] - tileMin[0]) - ( dstMax[0] - dstMin[0] ), 
                           dst + (dstMax[0] + ( dstMax[1] * tileSize[0] )+ ( kdst * tileSize[0] * tileSize[1] ))*m_dataSize
                         ), 
                         m_dataTypeOut
                       );
      }
    }
  } // end of Loop for volume with valid Z and invalid Y

  // Loop for volume with invalid Z and valid Y
  for (int kdst = dstMax[2]+1; kdst < tileSize[2]; ++kdst )
  {
    for (int jdst = dstMin[1]; jdst <= dstMax[1]; ++jdst )
    {
      int dstOffset = ( kdst * tileSize[0] * tileSize[1] + jdst * tileSize[0] ) * m_dataSize;
      int srcOffset = ( dstMax[2] * tileSize[1] * tileSize[0] + jdst * tileSize[0] ) * m_dataSize;

      // Fill the subvolume with valid X
      memcpy (dst + dstOffset, dst + srcOffset, m_dataSize * (dstMax[0]-dstMin[0]+1) );

      // Fill the subvolume with invalid X
      if ( tileMax[0] > subvMax[0] )
      {
        SB_DATATYPE_CALL ( fillTile, 
                         ( dst + ((dstMax[0]+1) + ( kdst * tileSize[0] * tileSize[1] ) + ( jdst * tileSize[0] ))* m_dataSize, 
                           (tileMax[0] - tileMin[0]) - ( dstMax[0] - dstMin[0]),
                           dst + (dstMax[0] + ( dstMax[2] * tileSize[0] * tileSize[1] ))*m_dataSize
                         ), 
                         m_dataTypeOut 
                       );
      }
    }
  } // end of loop for volume with invalid Z and valid Y

  // Loop for volume with invalid Z and invalid Y
  for (int kdst = dstMax[2]+1; kdst < tileSize[2]; ++kdst )
  {
    for (int jdst = dstMax[1]+1; jdst < tileSize[1]; ++jdst )
    {

      int dstOffset = ( kdst * tileSize[0] * tileSize[1] + jdst * tileSize[0] ) * m_dataSize;
      int srcOffset = ( dstMax[2] * tileSize[1] * tileSize[0] + jdst * tileSize[0] ) * m_dataSize;
      memcpy (dst + dstOffset, dst + srcOffset, m_dataSize * (dstMax[0]-dstMin[0]+1) );

      // Fill the subvolume with invalid X
      if ( tileMax[0] > subvMax[0] )
      {
        SB_DATATYPE_CALL ( fillTile,
                           ( dst + ((dstMax[0]+1) + (kdst * tileSize[0] * tileSize[1]) + (jdst * tileSize[0]))* m_dataSize ,
            (tileMax[0] - tileMin[0]) - (dstMax[0] - dstMin[0]),
            dst + (dstMax[0] + (dstMax[1] * tileSize[0]) + (dstMax[2] * tileSize[0] * tileSize[1]))* m_dataSize
            ),
            m_dataTypeOut
            );
      }

      // Fill the subvolume with valid X
      if ( tileMax[0] > subvMax[0] )
      {
        SB_DATATYPE_CALL ( fillTile,
                           ( dst + ((kdst * tileSize[0] * tileSize[1]) + (jdst * tileSize[0]))* m_dataSize,
            dstMax[0] - dstMin[0] +1,
            dst + (dstMax[0] + dstMax[1] * tileSize[0])* m_dataSize
            ),
            m_dataTypeOut
            );
      }
    }
  } // end of loop for volume with invalid Z and invalid Y
}

template <typename T>
void SoDataAccessReader::fillTile(void* ptrDst, int size, void* value )
{
  T* myPtr = (T*)ptrDst;
  T myValue = *(T*)value;

  std::fill(myPtr, myPtr + size, myValue);
}

