/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use a custom LDM reader.
author : Gamba Federico
December 2008
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <LDM/nodes/SoROI.h>
#include <LDM/manips/SoROIManip.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>

#include <LDM/nodes/SoGeometryPriority.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/events/SoKeyboardEvent.h>

#include <Inventor/helpers/SbFileHelper.h>

#include "CustomLDMReader.h"


SoSeparator* createBBox( SoVolumeData* pVolData);
SoROIManip* createROIManip( SoVolumeData* pVolData);
SoOrthoSliceDragger* createOrthoDragger( SoSeparator* pRoot, SoVolumeData* pVolData);

void EventCB (void *, SoEventCallback *eventCB);

SbString VolumeFile = "$OIVHOME/examples/source/VolumeViz/CustomReader/TestLDMCustomReader/data.txt";
#define LDM_RAM 256
#define LDM_GPU 128

SoSeparator* pRoot;
SoVolumeData* pVolData;
SoOrthoSlice* pOrtho;

//main function
int main(int argc, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize of VolumeViz extension
  SoVolumeRendering::init();
  CustomLDMReader::initClass();
  SbString envVar = SbString("OIV_USER_LIBS");
  SbString pathToCustomNode = "$OIVHOME/examples/bin/$OIVARCH/VolumeViz/CustomLDMReader";
  SoPreferences::setString(envVar, pathToCustomNode);

  if ( argc >= 2 ) SoPreferences::setString("LDM_INPUT", argv[1]);
  if ( argc >= 3) SoPreferences::setInt("LDM_MAINMEM",atoi(argv[2]));
  if ( argc >= 4) SoPreferences::setInt("LDM_TEXMEM",atoi(argv[3]));
  SbString inputFileName = SoPreferences::getString("LDM_INPUT",VolumeFile);

  if ( inputFileName == "" )
  {
    SoError::post(
      "You must define the input file name using\n"
      "LDM_INPUT environment variable\n"
    );
    exit(0);
  }

  // use our own custom reader

  CustomLDMReader *myReader = new CustomLDMReader;

  // Node to hold the volume data
  // In the first call we specify the voxel dimensions and the actual data.
  // In the second call we specify the 3D extent of the volume in modeling coords.
  pVolData = new SoVolumeData();
  pVolData->setReader(*myReader);
  pVolData->fileName.setValue(inputFileName);


  int MaxMainMem = SoPreferences::getInt("LDM_MAINMEM",LDM_RAM);
  int MaxTexMem = SoPreferences::getInt("LDM_TEXMEM",LDM_GPU);

  //size of 3D tiles cache memory in RAM (in MB)
  SoLDMGlobalResourceParameters::setMaxMainMemory(MaxMainMem);
  //size of 3D textures cache memory in GPU (in MB)
  SoLDMGlobalResourceParameters::setMaxTexMemory(MaxTexMem);


  SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TILE_OUTLINE, true);

  //this will force display to be updated every 10 time a 3D tile is loaded in RAM (only for debug purpose, in final application better to set to 50 at least)
  SoLDMGlobalResourceParameters::setLoadNotificationRate(10);
  //this will force display to be updated every 5 time a 3D texture is loaded in GPU (only for debug purpose, in final application better to set to 50 at least)
  SoLDMGlobalResourceParameters::setTex3LoadRate(5);


  fprintf(stdout,"LDM_INPUT set to %s\n",inputFileName.getString());
  fprintf(stdout,"LDM_MAINMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxMainMemory());
  fprintf(stdout,"LDM_TEXMEM set to %d MBytes\n",SoLDMGlobalResourceParameters::getMaxTexMemory());

  //getting data is faster in ALWAYS mode
  pVolData->ldmResourceParameters.getValue()->loadPolicy.setValue(SoVolumeData::SoLDMResourceParameters::ALWAYS);

  //set 2D textures properties
  //this will force display to be updated every 10 time a 2Dtexture is loaded in GPU (only for debug purpose, in final application better to set to 50 at least)
  pVolData->ldmResourceParameters.getValue()->tex2LoadRate.setValue(10);

  //size of 2D texture cache memory in GPU (in MByte)
  pVolData->ldmResourceParameters.getValue()->max2DTexMemory.setValue(40);


  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::STANDARD;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;

  // Assemble the scene graph
  pRoot = new SoSeparator;
  pRoot->ref();

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    pRoot->addChild( eventCB );

  pRoot->addChild(new SoGradientBackground);

  pRoot->addChild( pVolData );
  //must be done after setFilename on pVoldata
  pRoot->addChild(createBBox( pVolData) );
  pRoot->addChild( pTransFunc );

  pOrtho = new SoOrthoSlice;
  pRoot->addChild(pOrtho);
	pOrtho->axis.setValue(SoOrthoSlice::X);
	pOrtho->alphaUse.setValue(SoSlice::ALPHA_OPAQUE);

	SoOrthoSliceDragger* pOrthoDragger = createOrthoDragger( pRoot, pVolData);
    pRoot->addChild(pOrthoDragger);

  SoSeparator* pVolRendRoot = new SoSeparator;
  pRoot->addChild( pVolRendRoot );
    // Volume Rendering and Slice Rendering will have same priority
		pVolRendRoot->addChild(new SoGeometryPriority(0.9f));
		pVolRendRoot->addChild( createROIManip(pVolData) );
		pVolRendRoot->addChild( pVolRender );


  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
  myViewer->setSceneGraph(pRoot);
  myViewer->setTitle("LDM Custom Reader");
  myViewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  pRoot->unref();
  delete myViewer;

  CustomLDMReader::exitClass();
  SoVolumeRendering::finish();

  SoXt::finish();

  return 0;
}



SoSeparator* createBBox( SoVolumeData* pVolData)
{
	SoSeparator* m_BBox = new SoSeparator;
	m_BBox->ref();

	SbBox3f m_Extent = pVolData->extent.getValue();
    float xmin,ymin,zmin,xmax,ymax,zmax;
    m_Extent.getBounds( xmin,ymin,zmin,xmax,ymax,zmax );

    SoDrawStyle *pStyle = new SoDrawStyle;
    pStyle->lineWidth = 2;
    SoLightModel *pLMod = new SoLightModel;
    pLMod->model = SoLightModel::BASE_COLOR;

    SoVertexProperty *pProp = new SoVertexProperty;
    pProp->vertex.setNum( 8 );
    pProp->vertex.set1Value( 0, xmin, ymin, zmin );
    pProp->vertex.set1Value( 1, xmax, ymin, zmin );
    pProp->vertex.set1Value( 2, xmax, ymax, zmin );
    pProp->vertex.set1Value( 3, xmin, ymax, zmin );
    pProp->vertex.set1Value( 4, xmin, ymin, zmax );
    pProp->vertex.set1Value( 5, xmax, ymin, zmax );
    pProp->vertex.set1Value( 6, xmax, ymax, zmax );
    pProp->vertex.set1Value( 7, xmin, ymax, zmax );
    pProp->orderedRGBA.set1Value( 0, SbColor(1,0,0).getPackedValue() );

    const int32_t indices[] = {
      0, 1, 2, 3, 0, -1,
        4, 5, 6, 7, 4, -1,
        0, 4, -1,
        1, 5, -1,
        2, 6, -1,
        3, 7, -1
    };
    SoIndexedLineSet *pLine = new SoIndexedLineSet;
    pLine->vertexProperty = pProp;
    pLine->coordIndex.setValues( 0, 24, indices );

    m_BBox->addChild( pStyle );
    m_BBox->addChild( pLMod  );
    m_BBox->addChild( pLine );

	m_BBox->unrefNoDelete();

	return m_BBox;
}

SoROIManip* createROIManip( SoVolumeData* pVolData)
{

	SoROIManip* m_ROIManip = new SoROIManip;
	m_ROIManip->ref();

	m_ROIManip->setName("ROI");
	m_ROIManip->boxOn=true;
	m_ROIManip->constrained=true;

	SbVec3i32 m_LdmDim = pVolData->data.getSize();
    m_ROIManip->box.setValue(SbVec3i32(0,0,0), m_LdmDim/4 );
    m_ROIManip->subVolume.setValue( SbVec3i32(0,0,0), m_LdmDim - SbVec3i32(1,1,1) );

	m_ROIManip->unrefNoDelete();

	return m_ROIManip;
}

void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle data display
  if (SO_KEY_PRESS_EVENT(event, X)) {
    	pOrtho->axis.setValue(SoOrthoSlice::X);
  }

  // XZ plane
  else if (SO_KEY_PRESS_EVENT(event, Y)) {
    	pOrtho->axis.setValue(SoOrthoSlice::Y);
  }

  // XY plane
  else if (SO_KEY_PRESS_EVENT(event, Z)) {
    	pOrtho->axis.setValue(SoOrthoSlice::Z);
  }
  else {
    return;
  }
}



SoOrthoSliceDragger*
createOrthoDragger( SoSeparator* pRoot, SoVolumeData* pVolData)
{
  SoOrthoSliceDragger* app = new SoOrthoSliceDragger();

  //init dragger
  pOrtho->ref();
  SoSearchAction searchAction;
  searchAction.setSearchingAll(TRUE);
  searchAction.setNode( pOrtho );
  searchAction.apply( pRoot );
  SoPath *path = searchAction.getPath();
  app->orthoSlicePath.setValue(path);
  pOrtho->unrefNoDelete();

  return app;
}


