/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Aug 2011)
**=======================================================================*/

#include "SoCustomLdmWriter.h"
#include <Inventor/SoDB.h>
#include <Inventor/STL/cassert>
#include <Inventor/STL/sstream>
#include <LDM/converters/SoConverter.h>
#include <Inventor/helpers/SbFileHelper.h>

#if !defined _WIN32
#define _strdup strdup
#endif

SO_FIELDCONTAINER_SOURCE( SoCustomLdmWriter )

//------------------------------------------------------------------------------
//
SoCustomLdmWriter::SoCustomLdmWriter()
  : m_LDMTopo( new SoLDMTopoOctree )
  , m_isInitialized( FALSE )
{
  SO_FIELDCONTAINER_CONSTRUCTOR( SoCustomLdmWriter );

  SO_FIELDCONTAINER_ADD_FIELD( headerFilename, (SbString()) );
  SO_FIELDCONTAINER_ADD_FIELD( tileSize, (SbVec3i32(0, 0, 0)) );

  isBuiltIn = true;
}

//------------------------------------------------------------------------------
// 
void
SoCustomLdmWriter::initClass()
{
  SO_FIELDCONTAINER_INIT_CLASS( SoCustomLdmWriter, "CustomLdmWriter", SoVolumeWriter );
}

//------------------------------------------------------------------------------
// 
void
SoCustomLdmWriter::exitClass()
{
  SO__FIELDCONTAINER_EXIT_CLASS( SoCustomLdmWriter );
}

//------------------------------------------------------------------------------
// 
SoCustomLdmWriter::~SoCustomLdmWriter()
{
  delete m_LDMTopo;
  if ( m_dataFile != NULL )
    SbFileHelper::close(m_dataFile);
}

//------------------------------------------------------------------------------
// 
SbBool 
SoCustomLdmWriter::initialize()
{
  SbString dataFilename = headerFilename.getValue().getSubString(0, headerFilename.getValue().getLength()-4) + SbString("dat");
  m_dataFile = SbFileHelper::open(dataFilename, "w");

  m_LDMTopo->init( dimension.getValue(), tileSize.getValue()[0], 0 );


  SbString dataTypeXML;
  if ( dataType.getValue() == SoDataSet::UNSIGNED_BYTE )
    dataTypeXML = "byte";
  else if ( dataType.getValue() == SoDataSet::UNSIGNED_SHORT )
    dataTypeXML = "short";
  else if ( dataType.getValue() == SoDataSet::UNSIGNED_INT32 )
    dataTypeXML = "integer32";
  else if ( dataType.getValue() == SoDataSet::SIGNED_BYTE )
    dataTypeXML = "signed_byte";
  else if ( dataType.getValue() == SoDataSet::SIGNED_SHORT )
    dataTypeXML = "signed_short";
  else if ( dataType.getValue() == SoDataSet::SIGNED_INT32 )
    dataTypeXML = "signed_integer32";
  else if ( dataType.getValue() == SoDataSet::FLOAT )
    dataTypeXML = "float";

  //------------------------------------------------------------------------------
  // Generate header file

  std::stringstream headerStr;
  headerStr << "<?xml version=\"1.0\" encoding=\"utf-8\" ?>"                         << std::endl;
  headerStr << "<VolumeInformation>"                                                 << std::endl;
  headerStr << "  <DataType>" << dataTypeXML << "</DataType>"                        << std::endl;
  headerStr << "  <NumSignificantBits>8</NumSignificantBits>"                        << std::endl;
  headerStr << "  <Size>"                                                            << std::endl;
  headerStr << "    <U>" << dimension.getValue()[0] << "</U>"                        << std::endl;
  headerStr << "    <V>" << dimension.getValue()[1] << "</V>"                        << std::endl;
  headerStr << "    <W>" << dimension.getValue()[2] << "</W>"                        << std::endl;
  headerStr << "  </Size>"                                                           << std::endl;
  headerStr << "  <CoordinateSystem system=\"\">"                                    << std::endl;
  headerStr << "    <Origin>" << extent.getValue().getMin()[0] << " " << extent.getValue().getMin()[1] << " " << extent.getValue().getMin()[2] << " " << "</Origin>"           << std::endl;
  headerStr << "    <TotalExtent>"                                                   << std::endl;
  headerStr << "      <U> " << extent.getValue().getSize()[0] << " 0 0 </U>"         << std::endl;
  headerStr << "      <V> 0 " << extent.getValue().getSize()[1] << " 0 </V>"         << std::endl;
  headerStr << "      <W> 0 0 " << extent.getValue().getSize()[2] << " </W>"         << std::endl;
  headerStr << "    </TotalExtent>"                                                  << std::endl;
  headerStr << "  </CoordinateSystem>"                                               << std::endl;
  headerStr << "  <TileSize>"                                                        << std::endl;
  headerStr << "    <U>" << tileSize.getValue()[0] << "</U>"                         << std::endl;
  headerStr << "    <V>" << tileSize.getValue()[1] << "</V>"                         << std::endl;
  headerStr << "    <W>" << tileSize.getValue()[2] << "</W>"                         << std::endl;
  headerStr << "  </TileSize>"                                                       << std::endl;
  headerStr << "  <Border>0</Border>"                                                << std::endl;
  headerStr << "  <DataFilename>" << dataFilename << "</DataFilename>"               << std::endl;
  headerStr << "</VolumeInformation>"                                                << std::endl;

  FILE* headerFile = SbFileHelper::open( headerFilename.getValue().getString(), "w" );
  fprintf(headerFile, "%s", headerStr.str().c_str());
  SbFileHelper::close(headerFile);

  m_isInitialized = TRUE;

  return TRUE;
}

//------------------------------------------------------------------------------
SbBool 
SoCustomLdmWriter::closeAllHandles()
{
  if ( m_dataFile != NULL )
    SbFileHelper::close(m_dataFile);
  m_dataFile = NULL;
  return TRUE;
}

//------------------------------------------------------------------------------
SbBool 
SoCustomLdmWriter::restoreAllHandles()
{
  SbString dataFilename = headerFilename.getValue().getSubString(0, headerFilename.getValue().getLength()-4) + SbString(".dat");
  
  if ( m_dataFile == NULL )
    m_dataFile = SbFileHelper::open(dataFilename, "w");

  return TRUE;
}

//------------------------------------------------------------------------------
SbBool 
SoCustomLdmWriter::isDataConverted() const
{
  return TRUE;
}

//------------------------------------------------------------------------------
SbBool 
SoCustomLdmWriter::writeTile( const SoLDMTileID& tileId, SoBufferObject* buffer )
{
  if ( !m_isInitialized )
  {
    SoDebugError::post("SoCustomLdmWriter::writeTile", "writer is not initialized.");
    return FALSE;
  }

  // check that specified buffer has the right size.
  if ( buffer->getSize() != tileSize.getValue()[0] * tileSize.getValue()[1] * tileSize.getValue()[2] * SbDataType(dataType.getValue()).getSize() )
  {
    SoDebugError::post("SoCustomLdmWriter::writeTile", "buffer size does not correspond to tile size and datatype.");
    return FALSE;
  }


  // just retrieve a pointer to datas.
  SoCpuBufferObject* cpuBuffer = new SoCpuBufferObject;
  cpuBuffer->ref();
  buffer->map( cpuBuffer, SoBufferObject::READ_ONLY );
  void* bufferPtr = cpuBuffer->map( SoBufferObject::READ_ONLY );


  // retrieve fileId corresponding to tileId. It represent the offset of data in data file.
  int fileId = m_LDMTopo->getFileID(tileId);
  int64_t offset = (int64_t)buffer->getSize() * (int64_t)fileId;

  // write the data
  fseek( m_dataFile, (long)offset, 0 );
  fwrite( bufferPtr, 1, buffer->getSize(), m_dataFile );

  cpuBuffer->unmap();
  buffer->unmap(cpuBuffer);
  cpuBuffer->unref();
  return TRUE;
}

//------------------------------------------------------------------------------
SbBool 
SoCustomLdmWriter::finish()
{
  SbFileHelper::close( m_dataFile );
  m_dataFile = NULL;
  m_isInitialized = FALSE;
  return TRUE;
}

//------------------------------------------------------------------------------
void SoCustomLdmWriter::notify(SoNotList* list)
{
  if ( m_isInitialized )
    SoDebugError::post("SoCustomLdmWriter::notify", "Forbidden modification of initialized writer.");

  SoVolumeWriter::notify(list);
}

