/**
 * An example of a DICOM reader made with VolumeViz.
 *
 * Features focus are shown in the code an begin with @HIGHLIGHT.
 */

#include <Inventor/Qt/SoQtRenderArea.h>
#include <Inventor/Qt/viewers/SoQtPlaneViewer.h>
#include <Inventor/nodes/SoGradientBackground.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTranslation.h>
#include <LDM/nodes/SoDataRange.h>
#include <QAction>
#include <QDockWidget>
#include <QFileDialog>
#include <QHeaderView>
#include <QListView>
#include <QMenuBar>
#include <QObject>
#include <QStatusBar>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/readers/dicom/SiDicomSequence.h>
#include <VolumeViz/readers/dicom/SiDicomValue.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Medical/helpers/MedicalHelper.h>

#include "DicomViewer.h"
#include "ui_DicomViewer.h"

DicomViewer::DicomViewer( QWidget* parent )
  : QMainWindow( parent )
  , m_ui( new Ui::DicomViewer )
  , m_reader( new SoVRDicomFileReader )
  , m_volumeData( new SoVolumeData )
  , m_dataRange( new SoDataRange )
  , m_slice( new SoOrthoSlice )
  , m_sliceModel( new QStandardItemModel( this ) )
  , m_infoModel( new QStandardItemModel( this ) )
{
  m_ui->setupUi( this );
  init();

  m_infoModel->setRowCount( 0 );
  m_infoModel->setColumnCount( 3 );

  m_infoModel->setHeaderData( 0, Qt::Horizontal, QString( "Tag" ) );
  m_infoModel->setHeaderData( 1, Qt::Horizontal, QString( "Name" ) );
  m_infoModel->setHeaderData( 2, Qt::Horizontal, QString( "Value" ) );

  loadPath( "$OIVHOME/examples/data/VolumeViz/DicomSample/listOfDicomFiles.dcm" );
}

DicomViewer::~DicomViewer()
{
  delete m_ui;

  delete m_sliceModel;
  delete m_infoModel;
}

void
DicomViewer::init()
{
  m_ui->infoView->header()->setSectionResizeMode( QHeaderView::ResizeToContents );

  m_ui->sliceView->setModel( m_sliceModel );
  m_ui->infoView->setModel( m_infoModel );

  m_renderArea = new SoQtPlaneViewer( this );
  setCentralWidget( m_renderArea->getWidget() );
  m_renderArea->getWidget()->setSizePolicy( QSizePolicy( QSizePolicy::Maximum, QSizePolicy::MinimumExpanding ) );

  QObject::connect( m_ui->actionOpenFile, &QAction::triggered, this, &DicomViewer::openFile );
  QObject::connect( m_ui->actionOpenDir, &QAction::triggered, this, &DicomViewer::openDirectory );

  m_sliceSelectionModel = new QItemSelectionModel( m_sliceModel, this );
  m_ui->sliceView->setSelectionModel( m_sliceSelectionModel );

  QObject::connect( m_sliceSelectionModel, &QItemSelectionModel::currentChanged, this, &DicomViewer::changeActiveSlice );

  // build the scene graph
  SoRef<SoSeparator> root = new SoSeparator;

  SoRef<SoTransferFunction> transFunc( new SoTransferFunction );
  transFunc->predefColorMap = SoTransferFunction::INTENSITY;

  root->addChild( m_dataRange.ptr() );
  root->addChild( m_volumeData.ptr() );
  root->addChild( transFunc.ptr() );
  root->addChild( m_slice.ptr() );

  m_renderArea->setSceneGraph( root.ptr() );
  m_renderArea->viewAll();
}

void
DicomViewer::openFile()
{
  QString filePath = QFileDialog::getOpenFileName( this, "Open Image", QString() );
  if ( !filePath.isEmpty() )
  {
    loadPath( filePath );
  }
}

void
DicomViewer::openDirectory()
{
  QString dirPath = QFileDialog::getExistingDirectory( this, "Open Directory" );
  if ( !dirPath.isEmpty() )
  {
    loadPath( dirPath );
  }
}

void
DicomViewer::changeActiveSlice( const QModelIndex& index, const QModelIndex& /*old*/ )
{
  updateModel( index.row() );
}

void fillDataSetItem( const SiDicomDataSet& ds, QStandardItem* parent );
void fillSequenceItem( const SiDicomSequence& seq, QStandardItem* parent );

void
fillDataSetItem( const SiDicomDataSet& ds, QStandardItem* parent )
{
  for ( size_t i = 0; i < ds.getLength(); ++i )
  {
    const SiDicomElement* element = ds.getElement( i );

    //==========================================================================
    // @HIGHLIGHT: Extract element information:

    // The element tag; a struture where to get group and element id of the tag
    // and its name in the DICOM dictionary. See SoDicomTag.
    const SoDicomTag& tag = element->getTag();

    // The element value, an interface with different fields: see SiDicomValue.
    const SiDicomValue* value = element->getValue();

    // The datasets contained in a sequence element. This will be NULL if the
    // element value is set.
    const SiDicomSequence* seq = element->getSequence();

    //==========================================================================

    const QLatin1Char fillChar( '0' );
    const QString tagStr =
      QString( "(%1, %2)" ).arg( tag.getGroup(), 4, 16, fillChar ).arg( tag.getElement(), 4, 16, fillChar );
    QStandardItem* item = new QStandardItem( tagStr );
    parent->setChild( static_cast<int>( i ), 0, item );
    parent->setChild( static_cast<int>( i ), 1, new QStandardItem( tag.getName().toUtf8() ) );

    if ( value )
      parent->setChild( static_cast<int>( i ), 2, new QStandardItem( element->getValue()->asString().toUtf8() ) );

    if ( seq )
      fillSequenceItem( *seq, item );
  }
}

void
fillSequenceItem( const SiDicomSequence& seq, QStandardItem* parent )
{
  for ( size_t i = 0; i < seq.getLength(); ++i )
  {
    const SiDicomDataSet* ds = seq.getItem( i );
    int k = 0;
    if ( ds )
    {
      QStandardItem* item = new QStandardItem( QString( "Item#%1" ).arg( k + 1 ) );
      fillDataSetItem( *ds, item );
      parent->setChild( k, 0, item );
      k++;
    }
  }
}

void
DicomViewer::updateModel( size_t idx )
{
  m_slice->sliceNumber = static_cast<uint32_t>( idx );

  //============================================================================
  // @HIGHLIGHT: loading a DICOM slice dataset.

  // Step 1: Extract DICOM dataset from DICOM volume reader.
  const SiDicomDataSet* ds = m_reader->getDicomDataSet( idx );

  // Step 2: Iterate through DICOM elements.
  m_infoModel->setRowCount( static_cast<int>( ds->getLength() ) );
  for ( size_t i = 0; i < ds->getLength(); ++i )
  {
    const SiDicomElement* element = ds->getElement( i );
    const SoDicomTag& tag = element->getTag();
    const SiDicomValue* value = element->getValue();
    const SiDicomSequence* seq = element->getSequence();

    const QLatin1Char fillChar( '0' );
    const QString tagStr =
      QString( "(%1, %2)" ).arg( tag.getGroup(), 4, 16, fillChar ).arg( tag.getElement(), 4, 16, fillChar );
    QStandardItem* item = new QStandardItem( tagStr );
    m_infoModel->setItem( static_cast<int>( i ), 0, item );
    m_infoModel->setItem( static_cast<int>( i ) , 1, new QStandardItem( tag.getName().toUtf8() ) );

    if ( value )
      m_infoModel->setItem( static_cast<int>( i ) , 2, new QStandardItem( value->asString().toUtf8() ) );

    // recursive iteration into sequence items.
    if ( seq )
      fillSequenceItem( *seq, item );
  }

  // Step 3: do not forget to delete the dataset.
  delete ds;

  //============================================================================
}

void
DicomViewer::loadPath( const QString& path )
{
  m_sliceModel->clear();
  m_infoModel->clear();

  const SbString oivPath( SbFileHelper::expandString( path.toStdWString() ) );
  const QFileInfo info( QString().fromStdWString( oivPath.toStdWString() ) );

  // ===========================================================================
  // @HIGHTLIGHT: DICOM reader usage.
  if ( info.isFile() )
  {
    if ( m_reader->setFilename( oivPath ) != 0 )
      return;
  }
  else if( info.isDir() )
  {
    m_reader->setDirectory( oivPath );
  }
  // ===========================================================================

  else
  {
    //the file or folder does not exists
    return;
  }

  SbBox3f size;
  SoDataSet::DataType type;
  SbVec3i32 dim;

  m_reader->getDataChar( size, type, dim );

  int sliceCount = dim[2];
  m_sliceModel->setRowCount( static_cast<int>( sliceCount ) );
  for ( int i = 0; i < sliceCount; ++i )
  {
    m_sliceModel->setItem( i, 0, new QStandardItem( QString( "Slice #%1" ).arg( i + 1 ) ) );
  }

  if ( sliceCount > 0 )
  {
    m_volumeData->setReader( *m_reader.ptr() );
    m_renderArea->viewAll();
    m_sliceSelectionModel->setCurrentIndex( m_sliceModel->index( 0, 0 ), QItemSelectionModel::Select );

    MedicalHelper::dicomAdjustDataRange( m_dataRange.ptr(), m_volumeData.ptr() );
    MedicalHelper::orientView( MedicalHelper::AXIAL, m_renderArea->getCamera(), m_volumeData.ptr() );
    m_renderArea->saveHomePosition();
  }
}
