#include <OGLWindow.h>

#include <Inventor/SoDB.h>
#include <Inventor/devices/SoGLContext.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/sys/SoGLU.h>

#import <OpenGL/OpenGL.h>
#import <Cocoa/Cocoa.h>

@interface OGLView : NSOpenGLView
{
@public
  DrawCB m_cb;
  KeyCB m_keyCb;
  SoGLContext* m_context;
}

@end

@implementation OGLView
- (void)drawRect:(NSRect)theRect
{
  if (!m_context)
    return;


  m_context->bind();
  
  [[NSColor clearColor] set];

  NSRectFill([self bounds]);

  GLint opaque = NO;
  [[self openGLContext] setValues:&opaque forParameter:NSOpenGLCPSurfaceOpacity];

  [[self window] setOpaque:NO];

  [[self window] setAlphaValue:.999f];

  [[self window] setMovableByWindowBackground:YES];
  
  glViewport(0,0,[self bounds].size.width,[self bounds].size.height);
  glClearColor(0.f, 0.f, 0.f, 1.f);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  
  if (m_cb)
    m_cb();

 m_context->unbind();
 m_context->swapBuffers();

 [self setNeedsDisplay:YES];
}

- (void)keyUp:(NSEvent *)theEvent
{
  switch( [theEvent keyCode] ) {
        case 126: if (m_keyCb) m_keyCb(KEY_UP); break;      
        case 125: if (m_keyCb) m_keyCb(KEY_DOWN); break;
        case 124: if (m_keyCb) m_keyCb(KEY_RIGHT); break;
        case 123: if (m_keyCb) m_keyCb(KEY_LEFT); break;
        default: 
	  {
	    NSString *characters;
	    unsigned int characterIndex, characterCount;
	    
	    characters = [theEvent charactersIgnoringModifiers];
	    characterCount = [characters length];
	    for (characterIndex = 0; characterIndex < characterCount; characterIndex++) {
	      unichar v = [characters characterAtIndex:characterIndex];
	      if (v == 'j') if (m_keyCb) m_keyCb(KEY_J);
	      if (v == 'p') if (m_keyCb) m_keyCb(KEY_P);
		
	    }
	  }
	  break;
  }



}

- (BOOL) canBecomeKeyWindow
{
   return YES;
}

- (BOOL)acceptsFirstResponder
{
  return YES;
}

- (BOOL)becomeFirstResponder
{
  return YES;
}

- (BOOL)resignFirstResponder
{
  return YES;
}

- (void)timerFired:(NSTimer *)timer
{
  SoDB::processEvents();
}

- (void)setupTimer
{
 NSTimer* timer = [[NSTimer scheduledTimerWithTimeInterval:1.0f/120.0f target:self selector:@selector(timerFired:) userInfo:nil repeats:YES] retain];
}
@end


class OGLWindowImpl
{
public:
  OGLWindowImpl(int width, int height);

  ~OGLWindowImpl();

  void render();

  void mainLoop();

  void setDrawCallback(DrawCB callback);
  
  void setKeyCallback(KeyCB callback);

  void bindContext();
  
  void unbindContext();

  void update();

  static bool s_initNeeded;

  SoGLContext* m_context;
    
  DrawCB m_cb;
  KeyCB m_keyCb;
  
  NSAutoreleasePool* m_pool;

  OGLView* m_glView;
  NSWindow* m_container;
};

bool OGLWindowImpl::s_initNeeded = true;

OGLWindow::OGLWindow(int width, int height)
{
  m_impl = new OGLWindowImpl(width, height);
}

OGLWindow::~OGLWindow()
{
  delete m_impl;
}

void 
OGLWindow::render()
{
  m_impl->render();
}

void
OGLWindow::mainLoop()
{
  m_impl->mainLoop();
}

void 
OGLWindow::setDrawCallback(DrawCB callback)
{
  m_impl->setDrawCallback(callback);
}

void 
OGLWindow::setKeyCallback(KeyCB callback)
{
  m_impl->setKeyCallback(callback);
}

void
OGLWindow::update()
{
  m_impl->update();
}

void 
OGLWindow::bindContext()
{
  m_impl->bindContext();
}

void 
OGLWindow::unbindContext()
{
  m_impl->unbindContext();
}

OGLWindowImpl::OGLWindowImpl(int width, int height)
{
  m_context = NULL;
  m_cb = NULL;
  m_keyCb = NULL;

  if (s_initNeeded)
  {
    [NSApplication sharedApplication];
    s_initNeeded = false;
  }

  NSOpenGLPixelFormatAttribute attribs[] =
  {
    NSOpenGLPFANoRecovery,
    NSOpenGLPFAWindow,
    NSOpenGLPFAAccelerated,
    NSOpenGLPFADoubleBuffer,
    NSOpenGLPFAColorSize, 24,
    NSOpenGLPFAAlphaSize, 8,
    NSOpenGLPFADepthSize, 24,
    NSOpenGLPFAStencilSize, 8,
    NSOpenGLPFASingleRenderer,
    0
  };

  // Allocate the autorelease pool
  m_pool = [[NSAutoreleasePool alloc] init];

  NSOpenGLPixelFormat* windowedPixelFormat = [[NSOpenGLPixelFormat alloc] initWithAttributes:attribs];
  if ( windowedPixelFormat == nil )
  {
    //Couldn't initialize pixel format.
    SoDebugError::post( "SoGLWindow::createWindow", "Couldn't create NSOpenGLPixelFormat." );
    return;
  }

  NSRect windowFrame = NSMakeRect(0, 0, width, height);
  m_container = [[NSWindow alloc] initWithContentRect:windowFrame styleMask:NSTitledWindowMask | NSClosableWindowMask | NSMiniaturizableWindowMask | NSResizableWindowMask backing:NSBackingStoreRetained defer:NO];

  m_glView = [[OGLView alloc] initWithFrame:[m_container frame] pixelFormat:windowedPixelFormat];
  m_glView->m_context = NULL;
  m_glView->m_cb = m_cb;
  m_glView->m_keyCb = m_keyCb;

  if ( m_glView == nil )
  {
    //Couldn't initialize OpenGL view.
    SoDebugError::post( "SoGLWindow::createWindow", "Couldn't create NSOpenGLView." );
    return;
  }

  SbGlContextHelper::VisualInfo pix = (CGLPixelFormatObj)[[m_glView pixelFormat] CGLPixelFormatObj];
  SbGlContextHelper::GLContext ctx = (CGLContextObj)[[m_glView openGLContext] CGLContextObj];

  m_context = new SoGLContext( SbGlContextHelper::getCurrentDisplay(),pix, NULL, ctx );
  m_context->setNoGLContextDelete();

  m_glView->m_context = m_context;

  [((NSView*)m_glView) setHidden:NO];

  NSString* nsTitle = [[NSString alloc] initWithCString:"COCOA OpenGL window: VolumeRender"];
  [m_container setTitle:nsTitle];

  [m_container setContentView:m_glView];
  [m_container makeKeyAndOrderFront:nil];
  [m_container makeMainWindow];
  [((NSView*)m_glView) setNeedsDisplay:YES];

  [m_glView setupTimer];
}

OGLWindowImpl::~OGLWindowImpl()
{
  m_context->unref();
  
  [m_pool drain];
  m_pool = 0;
}

void
OGLWindowImpl::update()
{
  if (m_glView)
    [((NSView*)m_glView) setNeedsDisplay:YES];
}

void 
OGLWindowImpl::render()
{
  if (!m_context)
    return;
    
  SoDB::processEvents();

  if (m_cb)
    m_cb();

  m_context->swapBuffers();
}

void
OGLWindowImpl::mainLoop()
{
  // Enable this to get the keyboard events!!!
  [NSApp setActivationPolicy:NSApplicationActivationPolicyRegular];
  [NSApp run];
}

void 
OGLWindowImpl::setDrawCallback(DrawCB callback)
{
  m_cb = callback;
  if (m_glView)
    m_glView->m_cb = m_cb;
}

void 
OGLWindowImpl::setKeyCallback(KeyCB callback)
{
  m_keyCb = callback;
  if (m_glView)
    m_glView->m_keyCb = callback;

}

void 
OGLWindowImpl::bindContext()
{
  if (m_context)
    m_context->bind();
}

void 
OGLWindowImpl::unbindContext()
{
  if (m_context)
    m_context->unbind();
}
