#include <OGLWindow.h>

#include <windows.h>

#include <Inventor/SoDB.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/sys/SoGLU.h>

class OGLWindowImpl
{
public:
  OGLWindowImpl(int width, int height);

  ~OGLWindowImpl();

  void render();

  void mainLoop();

  void setDrawCallback(DrawCB callback);
  
  void setKeyCallback(KeyCB callback);

  void bindContext();
  
  void unbindContext();

  void update();

  static OGLWindowImpl* findWindow(HWND hwnd);

  SoRef<SoGLContext> m_context;
  HWND m_hwnd;
  HDC m_hdc;
  DrawCB m_cb;
  KeyCB m_keyCb;

  static std::map<HWND, OGLWindowImpl*> s_windows;
};


OGLWindow::OGLWindow(int width, int height)
{
  m_impl = new OGLWindowImpl(width, height);
}

OGLWindow::~OGLWindow()
{
  delete m_impl;
}

void 
OGLWindow::render()
{
  m_impl->render();
}

void
OGLWindow::update()
{
  m_impl->update();
}


void
OGLWindow::mainLoop()
{
  m_impl->mainLoop();
}

void 
OGLWindow::setDrawCallback(DrawCB callback)
{
  m_impl->setDrawCallback(callback);
}

void 
OGLWindow::setKeyCallback(KeyCB callback)
{
  m_impl->setKeyCallback(callback);
}


void 
OGLWindow::bindContext()
{
  m_impl->bindContext();
}

void 
OGLWindow::unbindContext()
{
  m_impl->unbindContext();
}


LRESULT CALLBACK mgrWindowProc( HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam )
{
  switch (message) 
  {  
    case WM_DESTROY:
      PostQuitMessage(WM_QUIT);
      break;
    case WM_PAINT:
      {
      OGLWindowImpl* win = OGLWindowImpl::findWindow(hwnd);
      if (win)
        win->render();
      break;
      }
    case WM_KEYDOWN:
      {
      OGLWindowImpl* win = OGLWindowImpl::findWindow(hwnd);
      if (win && win->m_keyCb)
      {
        int key = (int)(wParam);

        if (key == VK_LEFT) {win->m_keyCb(KEY_LEFT); break;}
        if (key == VK_RIGHT) {win->m_keyCb(KEY_RIGHT); break;}
        if (key == VK_SPACE) {win->m_keyCb(KEY_SPACE); break;}
        if (key == VK_UP) {win->m_keyCb(KEY_UP); break;}
        if (key == VK_DOWN) {win->m_keyCb(KEY_DOWN); break;}
        if (key == 'J') {win->m_keyCb(KEY_J); break;}
        if (key == 'P') {win->m_keyCb(KEY_P); break;}
        if (key == 'I') {win->m_keyCb(KEY_I); break;}
      
        if (key == VK_ESCAPE)
        {
         PostQuitMessage(WM_QUIT);
         break;
        }
      }
      }
    default:
      return DefWindowProc( hwnd, message, wParam, lParam );
  }
  return 0;
}


std::map<HWND, OGLWindowImpl*> OGLWindowImpl::s_windows;

OGLWindowImpl* 
OGLWindowImpl::findWindow(HWND hwnd)
{
  std::map<HWND, OGLWindowImpl*>::const_iterator it = s_windows.find(hwnd);

  if (it != s_windows.end())
    return it->second;
  return NULL;
}


OGLWindowImpl::OGLWindowImpl(int width, int height)
{
  LPCTSTR ClsName = "unitTest_GLVolumeViz";
  LPCTSTR WndName = "Simple GLVolumeViz";
  m_hwnd = NULL;
  m_hdc = NULL;

  // Create the window
  int pixelformat;
  HGLRC ctx;
  PIXELFORMATDESCRIPTOR pfd;

  WNDCLASSEX WndClsEx;

	// Create the application window
	WndClsEx.cbSize        = sizeof(WNDCLASSEX);
	WndClsEx.style         = CS_HREDRAW | CS_VREDRAW;
	WndClsEx.lpfnWndProc   = mgrWindowProc;
	WndClsEx.cbClsExtra    = 0;
	WndClsEx.cbWndExtra    = 0;
	WndClsEx.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
	WndClsEx.hCursor       = LoadCursor(NULL, IDC_ARROW);
	WndClsEx.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	WndClsEx.lpszMenuName  = NULL;
	WndClsEx.lpszClassName = ClsName;
	WndClsEx.hInstance     = 0;
	WndClsEx.hIconSm       = LoadIcon(NULL, IDI_APPLICATION);

  if (! RegisterClassEx(&WndClsEx))
  {
    MessageBox(0, "RegisterClass failed", "Error", MB_ICONEXCLAMATION | MB_OK);
    exit(1);
  }

  m_hwnd = CreateWindow( ClsName,
                          WndName,
                          WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS,
                          CW_USEDEFAULT,
                          CW_USEDEFAULT,
                          width,
                          height,
                          NULL,
                          NULL, 
                          0,
                          NULL);
  if (!m_hwnd)
  {
    MessageBox(0, "CreateWindow failed", "Error", MB_ICONEXCLAMATION | MB_OK);
    exit(2);
  }

  ShowWindow(m_hwnd, SW_SHOWNORMAL);
  UpdateWindow(m_hwnd);

  pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
  pfd.nVersion = 1;
  pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
  pfd.iPixelType = PFD_TYPE_RGBA;
  pfd.cColorBits = 24;
  pfd.cRedBits = 0;
  pfd.cRedShift = 0;
  pfd.cGreenBits = 0;
  pfd.cGreenShift = 0;
  pfd.cBlueBits = 0;
  pfd.cBlueShift = 0;
  pfd.cAlphaBits = 0;
  pfd.cAlphaShift = 0;
  pfd.cAccumBits = 0;
  pfd.cAccumRedBits = 0;
  pfd.cAccumGreenBits = 0;
  pfd.cAccumBlueBits = 0;
  pfd.cAccumAlphaBits = 0;
  pfd.cDepthBits = 32;
  pfd.cStencilBits = 0;
  pfd.cAuxBuffers = 0;
  pfd.iLayerType = PFD_MAIN_PLANE;
  pfd.bReserved = 0;
  pfd.dwLayerMask = 0;
  pfd.dwVisibleMask = 0;
  pfd.dwDamageMask = 0;

  // Setup pixel format
  m_hdc = GetDC(m_hwnd);
  if ((pixelformat = ChoosePixelFormat(m_hdc, &pfd)) == 0)
  {
    MessageBox(0, "ChoosePixelFormat failed", "Error", MB_ICONEXCLAMATION | MB_OK);
    exit(3);
  }
  if (SetPixelFormat(m_hdc, pixelformat, &pfd) == FALSE)
  {
    MessageBox(0, "SetPixelFormat failed", "Error", MB_ICONEXCLAMATION | MB_OK);
    exit(4);
  }
  if ((ctx = wglCreateContext(m_hdc)) == NULL)
  {
    MessageBox(0, "wglCreateContext failed", "Error", MB_ICONEXCLAMATION | MB_OK);
    exit(5);
  }

  m_context = new SoGLContext(m_hdc, &pfd, NULL, ctx);

  s_windows[m_hwnd] = this;
}

OGLWindowImpl::~OGLWindowImpl()
{
  if (m_context.ptr())
    m_context = NULL;

  if (m_hdc)
    ReleaseDC(m_hwnd, m_hdc);
  if (m_hwnd)
    DestroyWindow(m_hwnd);

  std::map<HWND, OGLWindowImpl*>::iterator it = s_windows.find(m_hwnd);

  if (it != s_windows.end())
    s_windows.erase(it);
}

void 
OGLWindowImpl::render()
{
  m_context->bind();
  if (m_cb)
    m_cb();
  m_context->unbind();
  m_context->swapBuffers();
}

void
OGLWindowImpl::update()
{
  if (m_hwnd)
    InvalidateRect(m_hwnd, NULL, FALSE);
}

void
OGLWindowImpl::mainLoop()
{
  MSG msg;
  while (GetMessage(&msg, 0, 0, 0))
  {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
}

void 
OGLWindowImpl::setDrawCallback(DrawCB callback)
{
  m_cb = callback;
}

void 
OGLWindowImpl::setKeyCallback(KeyCB callback)
{
  m_keyCb = callback;
}

void 
OGLWindowImpl::bindContext()
{
  m_context->bind();
}

void 
OGLWindowImpl::unbindContext()
{
  m_context->unbind();
}
