/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Mar 2011)
**=======================================================================*/

#include <Inventor/errors/SoDebugError.h>
#include <Inventor/SbElapsedTime.h>
#include <Inventor/SbTime.h>

#include <LDM/nodes/SoDataSet.h>
#include <LDM/tiles/SoCpuBufferBasicProperty.h>
#include <LDM/tiles/SoBufferAsyncNotifierInterface.h>
#include <LDM/readers/SoVRLdmFileReader.h>
#include <LDM/SoLDMTopoOctree.h>

#include "CustomAsyncBufferObject.h"
#include "CustomAsyncBufferCache.h"

//#define DEBUG_CACHE

/** init static vars */
uint64_t CustomAsyncBufferCache::CustomAsyncBufferCacheEntry::s_birthDate = 0;

/** */
CustomAsyncBufferCache::CustomAsyncBufferCache(const SbVec3i32& dataDimension, const SbVec3i32& tileDimension, const SoDataSet::DataType dataType, const size_t cacheSize)
:m_ldmReader(NULL),m_cacheLock()
{
  m_dataDimension = dataDimension;
  m_tileDimension = tileDimension;
  m_dataType = dataType;

  commonInit(cacheSize);
}

/** Constructor for LDM data set */
CustomAsyncBufferCache::CustomAsyncBufferCache(const SbString& filename, const size_t cacheSize)
:m_ldmReader(NULL),m_cacheLock()
{
  m_ldmReader = new SoVRLdmFileReader();
  m_ldmReader->setFilename(filename);
  m_ldmReader->ref();

  SbBox3f extent;
  m_ldmReader->getDataChar(extent,m_dataType,m_dataDimension);
  m_ldmReader->getTileSize(m_tileDimension);

  commonInit(cacheSize);
}

void
CustomAsyncBufferCache::commonInit(const size_t cacheSize)
{
  m_cacheSize = cacheSize;

  m_numRequest=0;
  m_numCacheRequest = 0;
  m_numCacheRequestMissed = 0;

  m_topologyOctree = new SoLDMTopoOctree;
  m_topologyOctree->init( m_dataDimension, m_tileDimension[0], 0 );

  asyncRequestThread = SbThread::create(s_asyncRequestThreadRoutine, this);
}

/** Destructor */
CustomAsyncBufferCache::~CustomAsyncBufferCache()
{
  // TODO : check if we need a barrier
  {
    SbThread::kill(asyncRequestThread);
    SbTime::sleep(1);
    SbThread::destroy(asyncRequestThread);
    SbTime::sleep(1);
  }
  delete m_topologyOctree;

  if ( m_ldmReader)
    m_ldmReader->unref();
  m_ldmReader = NULL;
}



void*
CustomAsyncBufferCache::s_asyncRequestThreadRoutine(void* externalCache)
{
  CustomAsyncBufferCache* curExternalCache = (CustomAsyncBufferCache*)(externalCache);
  while (!SbThread::isStopRequested())
  {
    curExternalCache->asyncRequestThreadRoutine();
  }
  return NULL;
}

void 
CustomAsyncBufferCache::asyncRequestThreadRoutine()
{
  // get a request from the list
  CustomAsyncRequest request;

  // we should implement a semaphore + signal
  // between this method and add addAsyncRequest
  if  ( m_asyncRequestList.size() == 0 )
    SbTime::sleep(100);

  m_asyncRequestLock.lock();
  if ( m_asyncRequestList.size() )
  {
    request = m_asyncRequestList.front();
    m_asyncRequestList.pop_front();
  }
  m_asyncRequestLock.unlock();

  int index = request.getId();
  if ( !(index<0) )
  {
     if (!isInCache(index))
     {

#if defined(DEBUG_CACHE)
       fprintf(stderr,"ASYNC: loading data entry %d\n",index);
#endif //DEBUG_CACHE

       // create a fake data buffer
       SoBufferObject* dataBuffer = createData(index);
       CustomAsyncBufferCacheEntry* cacheEntry = new CustomAsyncBufferCacheEntry(index,dataBuffer);

       m_cacheLock.lock();
       if ( !isInCache(index) )
       {
         checkCacheSize();
         m_cache[index]=cacheEntry;

         // trigger resource notification changes
         // TODO: could be done after a certain amount of request are loaded.
         request.getLDMAsyncNotifier()->refetchNotify();
       }
       else
       {
         fprintf(stderr,"ASYNC: loading a data entry %d already in cache\n",index);
         delete cacheEntry;
       }
       m_cacheLock.unlock();
     }
  }
}

void
CustomAsyncBufferCache::printStats() const
{
  static SbElapsedTime timing;
  if (timing.getElapsed()>1)
  {
    fprintf(stdout,"%llu read calls %llu cache call %.2f %% cache miss\n",
      (unsigned long long)m_numRequest,
      (unsigned long long)m_numCacheRequest,
      100.0f*float(m_numCacheRequestMissed)/float(m_numCacheRequest)
      );
    timing.reset();
  }
}

/** create a cache request  */
CustomAsyncBufferObject*
CustomAsyncBufferCache::createRequest(int index, const SbBox3i32&)
{
  CustomAsyncBufferObject* entry = new CustomAsyncBufferObject(this,index);
  ++m_numRequest;
  return entry;
}

void
CustomAsyncBufferCache::addAsyncRequest(SoBufferAsyncNotifierInterface* LDMAsyncNotifier, int index, const double /*weight*/)
{
  m_asyncRequestLock.lock();

  // first we check if it already exists in the list
  // if so then do nothing.
  bool found=false;
  std::list<CustomAsyncRequest>::const_iterator it = m_asyncRequestList.begin();
  for (;it!=m_asyncRequestList.end();++it)
  {
    if ( (*it).getId() == index && (*it).getLDMAsyncNotifier()==LDMAsyncNotifier )
    {
      found = true;
      break;
    }
  }
  if ( !found)
    m_asyncRequestList.push_back(CustomAsyncRequest(index,LDMAsyncNotifier));
  m_asyncRequestLock.unlock();
}

bool 
CustomAsyncBufferCache::isInCache(int index)
{
  SbThreadAutoLock autoLock(&m_cacheLock);
  std::map<int,CustomAsyncBufferCacheEntry*>::const_iterator it = m_cache.find(index);
  return ( (it!=m_cache.end()) && ((*it).second!=NULL) );
}

/** get an existing entry or create a new one */
SoBufferObject* 
CustomAsyncBufferCache::getData(int index, bool incRef)
{
  SbThreadAutoLock autoLock(&m_cacheLock);

  ++m_numCacheRequest;

  CustomAsyncBufferCacheEntry* cacheEntry = NULL;

  // Most of the time we should get data from the cache
  // of course if async mechanism works well
  std::map<int,CustomAsyncBufferCacheEntry*>::const_iterator it = m_cache.find(index);
  if ( (it!=m_cache.end()) && ((*it).second!=NULL) )
  {
    cacheEntry = (*it).second;
  }
  else
  {
    checkCacheSize();

    // create a fake data buffer
    SoBufferObject* dataBuffer = createData(index);

    cacheEntry = new CustomAsyncBufferCacheEntry(index,dataBuffer);
    m_cache[index]=cacheEntry;

#ifdef DEBUG_CACHE
    fprintf(stderr,"CACHE: adding entry %d\n",index);
#endif
    ++m_numCacheRequestMissed;
  }

  if ( incRef )
  {
    cacheEntry->ref();

#ifdef DEBUG_CACHE
    if (cacheEntry->getRefCount()>1 )
    {
      fprintf(stderr,"CACHE: refcount is greater %d\n",index);
    }
#endif
  }

  return cacheEntry->getBufferObject();
}

void CustomAsyncBufferCache::unref(int index)
{
  SbThreadAutoLock autoLock(&m_cacheLock);
  CustomAsyncBufferCacheEntry* cacheEntry = NULL;
  std::map<int,CustomAsyncBufferCacheEntry*>::const_iterator it = m_cache.find(index);
  if ( (it!=m_cache.end()) && ((*it).second!=NULL) )
  {
    cacheEntry = (*it).second;
    cacheEntry->unrefNoDelete();
  }
  else
  {
    SoDebugError::post("CustomAsyncBufferCache::unref","trying to unref a non-existing entry ??");
  }
}

void
CustomAsyncBufferCache::checkCacheSize()
{
  printStats();

  if ( m_cache.size() >= m_cacheSize )
  {
    // implement a very basic mechanism to get space in cache 
    // we try to remove 4 entries per call to reduce number of call to
    // this code (becasue of locks).

    // The really really basic strategy is to select entries to remove that
    // - is not currently used
    // - is older than other.
    //
    // A more realitic cache would select last recently used entry
    int numToDelete = 4;
    while ( numToDelete>0 )
    {
      // store the current oldest
      std::map<int,CustomAsyncBufferCacheEntry*>::iterator itToDelete = m_cache.end();

      // search for oldest
      std::map<int,CustomAsyncBufferCacheEntry*>::iterator it=m_cache.begin();
      while ( it!=m_cache.end() )
      {
        CustomAsyncBufferCacheEntry* curEntry = (*it).second;
        if ( curEntry && curEntry->getRefCount() == 0 )
        {
          if ( itToDelete==m_cache.end() || (curEntry->getBirthDay() < (*itToDelete).second->getBirthDay()) )
            itToDelete = it;
        }
        ++it;
      }

      // if we found the oldest remove it !
      if ( itToDelete!=m_cache.end() )
      {
#ifdef DEBUG_CACHE
        int index = (*itToDelete).first;
        fprintf(stderr,"CACHE: removing entry %d\n",index);
#endif
        delete (*itToDelete).second;
        m_cache.erase(itToDelete);
        --numToDelete;
      }
      else
      {
        // There is no entry to delete, probably becasue the cache is too small
        SoDebugError::post("CustomAsyncBufferCache::checkCacheSize()","Unable to get space in cache!\n  cache is too small");
        break;
      }
    }
  }
}


SoBufferObject*
CustomAsyncBufferCache::createData(int index)
{
  if ( m_ldmReader )
  {
    SoLDMTileID tileId = m_topologyOctree->getTileID(index);
    SbBox3i32 tilePosition = m_topologyOctree->getTilePos( tileId );
    SoBufferObject* data = m_ldmReader->readTile(index,tilePosition);
    data->ref();
    return data;
  }
  else
  {
    SoCpuBufferBasicProperty* dataBuffer = new SoCpuBufferBasicProperty();
    dataBuffer->ref();

    SbVec3i32 tileDim = getTileDimension();
    size_t tileByteSize = tileDim[0]*tileDim[1]*tileDim[2]*SbDataType(getDataType()).getSize();

    int resolution = m_topologyOctree->level(index);

    dataBuffer->setSize(tileByteSize);
    void *dataPtr = dataBuffer->map(SoBufferObject::SET);

    memset(dataPtr, resolution*10, tileByteSize);
    //dataBuffer->setMinMax(0,0);

    dataBuffer->unmap();

    return dataBuffer;
  }

  return NULL;
}

SbBool 
CustomAsyncBufferCache::getMinMax(int64_t & min, int64_t & max)
{
  if ( m_ldmReader )
    return m_ldmReader->getMinMax(min, max);
  return FALSE;
}

SbBool 
CustomAsyncBufferCache::getMinMax(double & min, double & max)
{
  if ( m_ldmReader )
    return m_ldmReader->getMinMax(min, max);
  return FALSE;
}


