/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Mar 2011)
**=======================================================================*/

#ifndef _CUSTOM_ASYNC_BUFFER_CACHE_H_
#define _CUSTOM_ASYNC_BUFFER_CACHE_H_

#include "common.h"
#include <Inventor/SbVec.h>
#include <Inventor/SbDataType.h>
#include <Inventor/threads/SbThreadMutex.h>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <Inventor/misc/SoRefCounter.h>
#include <LDM/nodes/SoDataSet.h>
#include <LDM/tiles/SoBufferAsyncNotifierInterface.h>

#include <Inventor/STL/map>
#include <Inventor/STL/list>

class SoCpuBufferObject;
class CustomAsyncBufferObject;
class SoVRLdmFileReader;
class SoLDMTopoOctree;

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/** Try to mimic application defined external cache */
class READER_DLLEXPORT CustomAsyncBufferCache
{
public:
  /** Constructor for fake data set */
  CustomAsyncBufferCache(
    const SbVec3i32& dataDimension,
    const SbVec3i32& tileDimension,
    const SoDataSet::DataType dataType,
    const size_t cacheSize
    );

  /** Constructor for LDM data set */
  CustomAsyncBufferCache(
    const SbString& filename,
    const size_t cacheSize
    );

  /** Destructor */
  ~CustomAsyncBufferCache();

  /** **/
  SbVec3i32 getTileDimension() const
  { return m_tileDimension; }

  /** **/
  SbVec3i32 getDimension() const
  { return m_dataDimension; }

  /** **/
  SoDataSet::DataType getDataType() const
  { return m_dataType; }

  /** */
  SbBool getMinMax(int64_t & min, int64_t & max);

  /** */
  SbBool getMinMax(double & min, double & max);

  /**
   * Create a buffer cache request
   * Note: called by the custom reader 
   */
  CustomAsyncBufferObject* createRequest(int index, const SbBox3i32& tilePosition);

  /**
   * Ask cache to reload data attached to an existing tile
   * Note: called by the custom bufferObject::refetch method
   */
  void addAsyncRequest(SoBufferAsyncNotifierInterface* LDMAsyncNotifier, int index, const double weight);

SoINTERNAL public:

  /** return TRUE if the given tile is in cache */
  bool isInCache(int index);

  /** add a tile in real cache */
  void addInCache(int index, SoCpuBufferObject* bufferObject);

  /** 
   * load in cache and return the data buffer of id index
   * if incRef is true the cacheEntry is locked and cannot be removed
   * until unref is called
  */
  SoBufferObject* getData(int index, bool incRef);

  void unref(int index);

  /** create a fake data tile */
  SoBufferObject* createData(int index);

  /** print statistics on console */
  void printStats() const;

private:

  void commonInit(const size_t cacheSize);

  SoVRLdmFileReader* m_ldmReader;

  /** properties of the dataset in application defined external cache */
  SbVec3i32 m_dataDimension;
  SbVec3i32 m_tileDimension;
  SoDataSet::DataType m_dataType;
  SoLDMTopoOctree* m_topologyOctree;

  //
  // Internal cache management
  //
  /** control the memory size that will be handled in the cache */
  size_t m_cacheSize;
  /** locking object on m_cacheMap */
  SbThreadMutex m_cacheLock;

  void checkCacheSize();

  /** internal storage for the cache */
  class CustomAsyncBufferCacheEntry : public SoRefCounter
  {
  public:
    /** constructor */
    CustomAsyncBufferCacheEntry(int tileID,SoBufferObject* bufferObject)
      :m_tileID(tileID),m_bufferObject(bufferObject),m_birthDate(s_birthDate++)
    {}
    /** destructor */
    ~CustomAsyncBufferCacheEntry()
    { 
      m_bufferObject->unref();
    }

    /** buffer accessor */
    SoBufferObject* getBufferObject() const
    { return m_bufferObject; }

    /** lifetime accessor */
    uint64_t getBirthDay() const
    { return m_birthDate; }

  private:
    int m_tileID;
    SoBufferObject* m_bufferObject;
    uint64_t m_birthDate;

    static uint64_t s_birthDate;
  };

  /** cache storage */
  std::map<int,CustomAsyncBufferCacheEntry*> m_cache;

  //
  // Asynchronous request management
  //
  class CustomAsyncRequest {
  public:
    /** Constructor */
    CustomAsyncRequest()
      : m_id(-1),m_asyncManager(NULL){};
    CustomAsyncRequest(int index,SoBufferAsyncNotifierInterface* asyncManager)
      : m_id(index),m_asyncManager(asyncManager){};

    /** accessors */
    int getId() const
    { return m_id; }

    /** accessors */
    SoBufferAsyncNotifierInterface* getLDMAsyncNotifier() const
    { return m_asyncManager.ptr(); }

  private:
      int m_id;
      SoRef<SoBufferAsyncNotifierInterface> m_asyncManager;
  };

  /** list of current async request */
  SbThreadSpinlock m_asyncRequestLock;
  std::list<CustomAsyncRequest> m_asyncRequestList;

  SbThread* asyncRequestThread;
  static void* s_asyncRequestThreadRoutine(void* userdata);
  void asyncRequestThreadRoutine();

  //
  // Statistics Info
  //
  size_t m_numRequest;
  size_t m_numCacheRequest;
  size_t m_numCacheRequestMissed;
};

#ifdef WIN32
#pragma warning( pop )
#endif

#endif // _CUSTOM_ASYNC_BUFFER_CACHE_H_


