/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Mar 2011)
**=======================================================================*/

#include <Inventor/errors/SoDebugError.h>
#include <Inventor/devices/SoCpuBufferObject.h>
#include <Inventor/devices/SoGLBufferObject.h>

#include "CustomAsyncBufferObject.h"
#include "CustomAsyncBufferCache.h"


/** Constructor */
CustomAsyncBufferObject::CustomAsyncBufferObject()
{
  assert(0);
};

CustomAsyncBufferObject::CustomAsyncBufferObject(CustomAsyncBufferCache* cache, int tileID)
{
  m_useCount = 0;
  m_customCache = cache;
  m_tileID = tileID;
  SoBufferObject::setSize( getRealSize() );
}

/** Destructor */
CustomAsyncBufferObject::~CustomAsyncBufferObject()
{
};


bool
CustomAsyncBufferObject::isInMemory()
{
  return m_customCache->isInCache(m_tileID);
}

void
CustomAsyncBufferObject::requestRefetch(SoBufferAsyncNotifierInterface* LDMAsyncNotifier,const double weight)
{
  m_customCache->addAsyncRequest(LDMAsyncNotifier,m_tileID,weight);
}


void 
CustomAsyncBufferObject::setupBuffer()
{
  if ( m_useCount == 0 )
  {
    // we manage only buffer inhertiting SoCpuBufferObject class
    SoCpuBufferObject* rawTile = dynamic_cast<SoCpuBufferObject*>(m_customCache->getData(m_tileID, true));
    assert(rawTile!=NULL);
    void* p = rawTile->map(SoCpuBufferObject::READ_ONLY);
    assert(p!=NULL);
    setBuffer(p,rawTile->getSize());
  }
}


void 
CustomAsyncBufferObject::resetBuffer()
{
  if (m_useCount == 0)
  {
    // we manage only buffer inhertiting SoCpuBufferObject class
    SoCpuBufferObject* rawTile = dynamic_cast<SoCpuBufferObject*>(m_customCache->getData(m_tileID, false));
    assert(rawTile!=NULL);
    setBuffer(NULL,rawTile->getSize());
    rawTile->unmap();
    m_customCache->unref(m_tileID);
  }
}



////////////////////////////////////////////////////////////////////////////
// Returns a pointer to the memory array used for the storage of the buffer.
void* 
CustomAsyncBufferObject::map(AccessMode accessMode, size_t offset, size_t count)
{
  assert(checkAccessMode(accessMode));
  setupBuffer();

  ++m_useCount;

  return SoCpuBufferObject::map(accessMode, offset, count);
}

//////////////////////////////////////////////////////////////////////
// Map the current buffer object into the specified CPU buffer object.
void 
CustomAsyncBufferObject::map( SoCpuBufferObject* targetBufferObject, AccessMode accessMode, size_t startPosition, size_t mappingSize )
{
  assert(checkAccessMode(accessMode));
  setupBuffer();
  ++m_useCount;
  SoCpuBufferObject::map ( targetBufferObject, accessMode, startPosition, mappingSize );
}

////////////////////////////////////////////////////////////////////
// Map the current buffer object into the specified GLbuffer object.
void 
CustomAsyncBufferObject::map( SoGLBufferObject* targetBufferObject, AccessMode accessMode, size_t startPosition, size_t mappingSize )
{
  assert(checkAccessMode(accessMode));
  setupBuffer();
  ++m_useCount;
  SoCpuBufferObject::map ( targetBufferObject, accessMode, startPosition, mappingSize );
}

//////////////////////////////////////////////////////////////////
// Map the current buffer object into the specified buffer object.
void 
CustomAsyncBufferObject::map( SoBufferObject* targetBufferObject, AccessMode accessMode, size_t startPosition, size_t mappingSize )
{
  assert(checkAccessMode(accessMode));
  setupBuffer();
  ++m_useCount;
  SoCpuBufferObject::map(targetBufferObject, accessMode, startPosition, mappingSize);
}


////////////////////////////////////////////////////////////////////////////
// Unmap buffer
void 
CustomAsyncBufferObject::unmap()
{
  assert(m_useCount>0);
  --m_useCount;
  resetBuffer();
}

void
CustomAsyncBufferObject::unmap( SoCpuBufferObject* targetBufferObject )
{
  SoCpuBufferObject::unmap ( targetBufferObject );
  assert(m_useCount>0);
  --m_useCount;
  resetBuffer();
}

void 
CustomAsyncBufferObject::unmap( SoGLBufferObject* targetBufferObject )
{
  SoCpuBufferObject::unmap ( targetBufferObject );
  assert(m_useCount>0);
  --m_useCount;
  resetBuffer();
}

void 
CustomAsyncBufferObject::unmap( SoBufferObject* bufferObject )
{
  SoCpuBufferObject::unmap ( bufferObject );
  assert(m_useCount>0);
  --m_useCount;
  resetBuffer();
}

bool 
CustomAsyncBufferObject::setSize( size_t )
{
  assert(0);
  return false;
}

bool
CustomAsyncBufferObject::getMinMax(double& , double& )
{
  return false;
}

size_t 
CustomAsyncBufferObject::getRealSize() const
{
  SbVec3i32 tileDim = m_customCache->getTileDimension();
  return tileDim[0]*tileDim[1]*tileDim[2]*SbDataType(m_customCache->getDataType()).getSize();
}


