/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Mar 2011)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

// DialogViz Interface
#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/actions/SoSearchAction.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include "CustomAsyncDataReader.h"


#define DEMOROOT "$OIVHOME/examples/source/VolumeViz/LDMAsyncCache/"

/** Global root scene graph loaded from scenegraph.iv */
SoSeparator *g_rootSceneGraph = NULL;

/** */
SoGroup *g_myGroup  = NULL;

/** */
SoTopLevelDialog *g_myTopLevelDialog=NULL;

/** helper to search a node by type */
template<typename T> T* searchNode(SoNode* scene);

/** helper to search a node by name */
template<typename T> T* searchName(SoNode* scene, SbName name);

/** build and link the dialogviz interface */
Widget buildInterface(Widget window, const char* filename, const char* viewer, SoTopLevelDialog** myTopLevelDialog);

/** Release resources allocated by buildInterface */
void releaseInterface();

//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow)
    return 0;
  SoDialogViz::init();
  SoVolumeRendering::init();
  CustomAsyncDataReader::initClass();

  // setup path where to find our CustomAsyncDataReader custom node
  // this is based on standard Open Inventor output directory rules
  SoPreferences::setString("OIV_USER_LIBS", "$OIVHOME/examples/bin/$OIVARCH/VolumeViz/CustomAsyncDataReader");

  // Enable LDM SoBufferAsyncInterface support (it is false by default)
  SoPreferences::setBool("IVVR_ASYNC_SUPPORT",TRUE);

  //SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TILE_OUTLINE, true);
  //SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::USE_FAKE_DATA, true);

  // Read the global scene graph
  {
    SoInput myInput;
    myInput.openFile( SbString(DEMOROOT) + "scenegraph.iv" );
    g_rootSceneGraph = SoDB::readAll(&myInput);
    g_rootSceneGraph->ref();
  }

  // search for SoVolumeData node in the scene graph and assign it our custom reader
  CustomAsyncDataReader* customReader = new CustomAsyncDataReader;
  SoVolumeData* volumeData = searchNode<SoVolumeData>(g_rootSceneGraph);
  if ( volumeData )
  {
    SbString filename = volumeData->fileName.getValue();
    volumeData->setReader(*customReader, TRUE);
    customReader->setFilename(filename);
  }

  // Interface setup
  SbString InterfaceFile = SbString(DEMOROOT) + "interface.iv";
  Widget parent = buildInterface(myWindow, InterfaceFile.toLatin1(), "Viewer", &g_myTopLevelDialog);

  // launch the viewer
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
  myViewer->setTitle("Custom Async buffer cache");
  myViewer->setSceneGraph(g_rootSceneGraph);
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();

  // release allocated object
  g_rootSceneGraph->unref();

  // release viewer
  delete myViewer;

  // release interface resrouces
  releaseInterface();

  CustomAsyncDataReader::exitClass();
  SoVolumeRendering::finish();

  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


////////////////////// BEGIN INTERFACE MANAGEMENT ////////////////////////////

// DialogViz auditor class to handle user input
class myAuditorClass : public SoDialogAuditor
{
  virtual void dialogCheckBox (SoDialogCheckBox* cpt);
  virtual void dialogIntegerSlider (SoDialogIntegerSlider* cpt);
};

// Auditor method for checkBox input
void
myAuditorClass::dialogCheckBox(SoDialogCheckBox* cpt)
{
  if (cpt->auditorID.getValue() == "ShowTopology")
  {
    SbBool value = cpt->state.getValue();
    SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, value);
  }
}

// Auditor method for IntegerSlider input
void
myAuditorClass::dialogIntegerSlider(SoDialogIntegerSlider* cpt)
{
  // change current max texture memory
  if (cpt->auditorID.getValue() == "gpuMemory")
    SoLDMGlobalResourceParameters::setMaxTexMemory(cpt->value.getValue());
  // change current max cpu memory
  else if (cpt->auditorID.getValue() == "cpuMemory")
    SoLDMGlobalResourceParameters::setMaxMainMemory(cpt->value.getValue());
}

void releaseInterface()
{
  if (g_myGroup)
    g_myGroup->unref();
  g_myGroup = NULL;

}
Widget
buildInterface(Widget window, const char* filename, const char* viewer, SoTopLevelDialog** myTopLevelDialog)
{
  SoInput myInput;
  if (! myInput.openFile( filename ))
    return NULL;

  g_myGroup = SoDB::readAll( &myInput );
  if (! g_myGroup)
    return NULL;
  g_myGroup->ref();

  *myTopLevelDialog = (SoTopLevelDialog *)g_myGroup->getChild( 0 );

  SoDialogCustom *customNode = (SoDialogCustom *)(*myTopLevelDialog)->searchForAuditorId(SbString(viewer));

  // Create and register auditor to handle user input
  myAuditorClass *myAuditor = new myAuditorClass;
  (*myTopLevelDialog)->addAuditor(myAuditor);

  (*myTopLevelDialog)->buildDialog( window, customNode != NULL );
  (*myTopLevelDialog)->show();

  // realSlider init with current interface file 
  SoDialogIntegerSlider* cpuMemory = (SoDialogIntegerSlider*)( (*myTopLevelDialog)->searchForAuditorId(SbString("cpuMemory")) );
  if ( cpuMemory )
    SoLDMGlobalResourceParameters::setMaxMainMemory(cpuMemory->value.getValue());
  SoDialogIntegerSlider* gpuMemory = (SoDialogIntegerSlider*)( (*myTopLevelDialog)->searchForAuditorId(SbString("gpuMemory")) );
  if ( gpuMemory )
    SoLDMGlobalResourceParameters::setMaxTexMemory(gpuMemory->value.getValue());


  // Combobox to select which object to show
  SoDialogComboBox* renderModeCombo = (SoDialogComboBox*)( (*myTopLevelDialog)->searchForAuditorId(SbString("RenderMode")) );
  if ( renderModeCombo != NULL )
  {
    SoSwitch* renderSwitch = searchName<SoSwitch>(g_rootSceneGraph,"VolumeRenderingSwitch");
    if ( renderSwitch )
    {
      for ( int i = 0; i < renderSwitch->getNumChildren(); ++i)
      {
        SoNode* child = renderSwitch->getChild(i);
        renderModeCombo->items.set1Value(i, child->getName().getString());
      }
      renderSwitch->whichChild.connectFrom( &renderModeCombo->selectedItem );
    }
    else
      renderModeCombo->enable = FALSE;
  }

  return customNode ? customNode->getWidget() : window;
}



// search a node by type
template<typename T>
T* 
searchNode(SoNode* scene)
{
  T* node = NULL;

  SoSearchAction sa;
  sa.setFind(SoSearchAction::TYPE);
  sa.setSearchingAll(true);
  sa.setType(T::getClassTypeId());
  sa.apply(scene);

  SoPath* path = sa.getPath();
  if ( path )
    node = dynamic_cast<T*>(path->getTail());
  if (!node)
    std::cerr << T::getClassTypeId().getName().getString() << " not found" << std::endl;
  return node;
}

// search a node by name
template<typename T>
T* 
searchName(SoNode* scene, SbName name)
{
  T* node = NULL;
  if (!scene)
  {
    node=(T*)SoNode::getByName(name);
  }
  else
  {
    SoSearchAction sa;
    sa.setFind(SoSearchAction::NAME);
    sa.setSearchingAll(true);
    sa.setType(T::getClassTypeId());
    sa.setName(name);
    sa.apply(scene);

    SoPath* path = sa.getPath();
    if ( path )
      node = dynamic_cast<T*>(path->getTail());
  }

  if ( node == NULL)
    std::cerr << T::getClassTypeId().getName().getString() << " not found" << std::endl;
  return node;
}


