/*----------------------------------------------------------------------------------------
LargeSliceSupport
Purpose : Show utility of LargeSliceSupport on simple test case
author : Ludovic Peine
January 2009
----------------------------------------------------------------------------------------*/

#if defined(_MSC_VER)
#pragma warning( disable : 4005 )
#endif

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/SbElapsedTime.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoRotation.h> 
#include <Inventor/sensors/SoAlarmSensor.h> 
#include <Inventor/nodes/SoCoordinate3.h> 
#include <Inventor/actions/SoSearchAction.h> 
#include <Inventor/nodes/SoVertexShape.h>
#include <Inventor/nodes/SoEventCallback.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <LDM/SoLDMMediator.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeSkin.h> 
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>

#include <LDM/manips/SoROIManip.h>

#include "utils.h"


//Define Default file name
SbString DefaultFileName = SbFileHelper::expandString("$OIVHOME/examples/data/VolumeViz/3DHEAD.ldm");

SbDataType dataType;

// Used method
SoSeparator *makeVolBBox( SbBox3f volSize );
SoOrthoSliceDragger* createOrthoDragger( SoSeparator* pRoot, SoVolumeData* pVolData);


SoVolumeData* pVolData;
SoOrthoSlice* pSlice;
SoROIManip*   pROIManip   = NULL;
SbVec3i32 volDim;

/**
* Simply reset the ROI to match the whole volume
*/
void resetRoi()
{
  pROIManip->box.setValue( SbVec3i32(0,0,0), pVolData->data.getSize() - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), pVolData->data.getSize()- SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;
}

void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Change the orientation of the orthoslice
  if (SO_KEY_PRESS_EVENT(event, X)) 
  {
    pSlice->axis.setValue(SoOrthoSlice::X);
  }
  else if (SO_KEY_PRESS_EVENT(event, Y)) 
  {
    pSlice->axis.setValue(SoOrthoSlice::Y);
  }
  else if (SO_KEY_PRESS_EVENT(event, Z)) 
  {
    pSlice->axis.setValue(SoOrthoSlice::Z);
  }
  // Invert the LargeSliceSupport value
  else if (SO_KEY_PRESS_EVENT(event, L)) 
  {
    bool largeSliceSupport = pSlice->largeSliceSupport.getValue();
    pSlice->largeSliceSupport.setValue(!largeSliceSupport);
  }
  // change the current orthoslice
  else if (SO_KEY_PRESS_EVENT(event, UP_ARROW)) 
  {
    int sliceNumber = pSlice->sliceNumber.getValue();
    if ( ++sliceNumber == volDim[pSlice->axis.getValue()] )
      sliceNumber=0;

    pSlice->sliceNumber.setValue(sliceNumber);
  }
  // change the current orthoslice
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) 
  {
    int sliceNumber = pSlice->sliceNumber.getValue();
    if ( --sliceNumber < 0 )
      sliceNumber = volDim[pSlice->axis.getValue()]-1;

    pSlice->sliceNumber.setValue(sliceNumber);
  }

}

///////////////////////////////////////////////////////////////////////
// Set VolumeViz parameters from config file
// (using VolRend keywords for convenience)
//
void initVolVizParams( SoVolumeData * volData )
{
  SoLDMGlobalResourceParameters::setScreenResolutionCulling( 0 );
  SoLDMGlobalResourceParameters::setViewCulling( 0 );
  SoLDMGlobalResourceParameters::setViewpointRefinement( 0);
  SoLDMGlobalResourceParameters::setSliceEqualResolution( 1 );

  volData->ldmResourceParameters.getValue()->maxMainMemory= 1;
}

void createOrthoSlices( SoGroup *pParent, const SbVec3i32 volumeDim )
{
  // For this example we just create 2 orthoslice to show the result
  // The first is a classic orthoslice
  pSlice = new SoOrthoSlice;
  pSlice->axis.setValue(SoOrthoSlice::Z);
  pSlice->sliceNumber.setValue(volumeDim[2]/2);
  pSlice->interpolation.setValue(SoOrthoSlice::LINEAR);
  pSlice->clipping.setValue(TRUE);
  pSlice->clippingSide.setValue(SoOrthoSlice::BACK);
  pParent->addChild(pSlice);
}

//////////////////////////////////////////////////////////////////////// 
int main(int argc, char **argv)
{
  char* filename = NULL;
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow)
    return 0;
  // Initialize DialogViz
  SoDialogViz::init();
  // Initialize VolumeViz
  SoVolumeRendering::init();

  SoTopLevelDialog *myTopLevelDialog;
  SbString InterfaceName = SbFileHelper::expandString("$OIVHOME/examples/source/VolumeViz/LargeSliceSupport/interface.iv");
  Widget parent = buildInterface(myWindow, InterfaceName.toLatin1(), "Viewer", &myTopLevelDialog);

  if ( argc < 2 )
    
    filename = (char*)DefaultFileName.toLatin1();
  else
    filename = (char*)SbFileHelper::expandString(argv[1]).toLatin1();

  pVolData = new SoVolumeData();
  initVolVizParams( pVolData );
  pVolData->fileName = filename;
  volDim = pVolData->data.getSize();

  SbBox3f volExtent = pVolData->extent.getValue();

  // Use predefined colorMap with SoTransferFunction
  SoTransferFunction* pTransFunc = new SoTransferFunction();
  pTransFunc->predefColorMap = SoTransferFunction::PHYSICS;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoSeparator *root = new SoSeparator;
  root->ref();
  SoVolumeRender* pVolRender = new SoVolumeRender;

    // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Add a ROI
  pROIManip = new SoROIManip();
  resetRoi();

  root->ref();
  root->addChild(eventCB);
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  createOrthoSlices(root, volDim);

  SoOrthoSliceDragger* pOrthoDragger = createOrthoDragger( root, pVolData);
  root->addChild(pOrthoDragger);

  root->addChild( pVolRender );

  // Set up viewer:
  SoXtExaminerViewer *m_viewer = new SoXtExaminerViewer(parent);
  m_viewer->setSceneGraph(root);

  m_viewer->show();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete m_viewer;
  root->unref();

  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}
  
///////////////////////////////////////////////////////////////////////
//
// Create a wireframe box showing bounding box of volume
//
// MHeck
//

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>

SoSeparator *makeVolBBox( SbBox3f volSize )
{
  // The box will be easier to see without lighting and with wide lines
  SoLightModel *pLModel = new SoLightModel;
  pLModel->model = SoLightModel::BASE_COLOR;

  SoDrawStyle *pStyle = new SoDrawStyle;
  pStyle->lineWidth = 2;

  // The box should be unpickable so manip can be used inside it
  SoPickStyle *pPickable = new SoPickStyle;
  pPickable->style = SoPickStyle::UNPICKABLE;

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  volSize.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty *pProp = new SoVertexProperty;
  pProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  pProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  pProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  pProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  pProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  pProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  pProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  pProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );
  pProp->orderedRGBA.set1Value( 0, 0xFF0000FF );

  // Draw it with a line set
  int coordIndices[] = {0,1,2,3,0,-1,4,5,6,7,4,-1,
                        0,4,-1, 1,5,-1, 2,6,-1, 3,7};
  int numCoordIndices = sizeof(coordIndices)/sizeof(int);
  SoIndexedLineSet *pLines = new SoIndexedLineSet;
  pLines->vertexProperty = pProp;
  pLines->coordIndex.setValues( 0, numCoordIndices, coordIndices );

  // Assemble scene graph
  SoSeparator *pBoxSep = new SoSeparator;
  pBoxSep->addChild( pLModel );
  pBoxSep->addChild( pPickable );
  pBoxSep->addChild( pStyle );
  pBoxSep->addChild( pLines );
  return pBoxSep;
}


SoOrthoSliceDragger*
createOrthoDragger( SoSeparator* pRoot, SoVolumeData* pVolData)
{
  SoOrthoSliceDragger* app = new SoOrthoSliceDragger();

  //init dragger
  pSlice->ref();
  SoSearchAction searchAction;
  searchAction.setSearchingAll(TRUE);
  searchAction.setNode( pSlice );
  searchAction.apply( pRoot );
  SoPath *path = searchAction.getPath();
  app->orthoSlicePath.setValue(path);
  pSlice->unrefNoDelete();

  return app;
}


