/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : G. SEGUIN (Oct 2001)
**=======================================================================*/

/////////////////////////////////////////////////////////////////////////////////
//
//Demo for realtime volume machining using VolumeRendering in Open Inventor
//
// Requires VolumeRendering library v3.0 or better
//
//
// Permission is granted for licensed customers of the 
// VolumeRendering library to use this source for any reasonable
// purpose that involves the VolumeRendering library.
//
// Files details:
//    - Machining.cxx/Machining.h: files containing VolumeRendering and Inventor machining code.
//    - SoVRBrush.cxx/SoVRBrush.h: utility class to help machining volumes.
//    - Controls.cxx/Controls.h: 3DMasterSuite dialog box.
//
// If you want to enable/disable Dialog and MPEG support you can define/undefine
// NO_DIALOG and USE_MPEG in the project settings/Makefile.
//
/////////////////////////////////////////////////////////////////////////////////
#include "Machining.h"
#include <DialogViz/SoDialogVizAll.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

Widget buildInterface(Widget);

#include <Inventor/SoWinApp.h>

#include <Inventor/helpers/SbFileHelper.h>

// Sets brush shape and size as well as updating the scaling of the shape (depends on brush size)
void setBrush(SbVec3s brushSize, SoVRBrush::PredefinedMatrix predefMatrix = SoVRBrush::CYLINDER, unsigned char * matrix = NULL)
{
  m_brush.setMatrix(brushSize, predefMatrix, matrix);

  if (m_shapeSize[0]==0)
    m_scale[0] = 1.0f;
  else
    m_scale[0] = (m_xDim-2*m_xOffset-(m_brush.getSize())[0])/m_shapeSize[0];
  if (m_shapeSize[1]==0)
    m_scale[1] = 1.0f;
  else
    m_scale[1] = (m_yDim-2*m_yOffset-(m_brush.getSize())[1])/m_shapeSize[1];
  if (m_shapeSize[2]==0)
    m_scale[2] = 1.0f;
  else
    m_scale[2] = (m_zDim-2*m_zOffset-(m_brush.getSize())[2])/m_shapeSize[2];
}

// Performs the actual drawing of the shape. Called by m_timer TimerSensor.
void shapeCB(void *, SoSensor * callingSensor)
{
  if (!m_workPath) {
    callingSensor->unschedule();
    return;
  }

  static int x;
  static int y;
  static int z;
  static SbVec3f step;
  static SoMFVec3f * points;

  for (unsigned int loop = 0; loop < m_drawingSpeed; loop++) {
    // Loop m_updateStep times in order to increase drawing speed.

    m_counter++;
    if (m_counter>=m_maxCounter) {
      // End of line
      m_counter = 0;
      m_curPoint++;
      if (m_curPoint>=(m_numPoints-1)) {
        // Shaping of the current path finished
        m_curPoint = 0;
        m_curChild++;
        if (m_curChild >= (m_lineSets->getLength())) {
          // No more lineSet to draw
          callingSensor->unschedule();
          return;
        }

        // Retrieves current set of points
        SoLineSet * lineSet = (SoLineSet *)((*m_lineSets)[m_curChild]->getTail());
        SoVertexProperty * vertexProperty = (SoVertexProperty *)(lineSet->vertexProperty).getValue();
        points = &vertexProperty->vertex;
        m_numPoints = (int)(*points).getNum();
      }

      // Calculates steps
      step[0] = (*points)[m_curPoint+1][0];
      step[1] = (*points)[m_curPoint+1][1];
      step[2] = (*points)[m_curPoint+1][2];
      step[0] -= (*points)[m_curPoint][0];
      step[1] -= (*points)[m_curPoint][1];
      step[2] -= (*points)[m_curPoint][2];
      step[0] *= m_scale[0];
      step[1] *= m_scale[1];
      step[2] *= m_scale[2];
      float maxStep = (float)abs((int)(step[0]));
      if (maxStep < (float)abs((int)(step[1])))
        maxStep = (float)abs((int)(step[1]));
      if (maxStep < (float)abs((int)(step[2])))
        maxStep = (float)abs((int)(step[2]));
      if (maxStep>0) {
        m_maxCounter = (int)maxStep;
        step[0] /= maxStep;
        step[1] /= maxStep;
        step[2] /= maxStep;
      }
      else m_maxCounter = 1;
    }

    // Calculates drawing coordinates
    x = (int)(((*points)[m_curPoint][0] - m_start[0])*m_scale[0] + step[0] * m_counter);
    y = (int)(((*points)[m_curPoint][1] - m_start[1])*m_scale[1] + step[1] * m_counter);
    z = (int)(((*points)[m_curPoint][2] - m_start[2])*m_scale[2] + step[2] * m_counter);


    // Draw brush: modify user data.
    m_brush.paint(SbVec3s(m_xDim-(x+m_xOffset+(m_brush.getSize())[0]-1), y+m_yOffset, z+m_zOffset), m_data, SbVec3s(m_xDim, m_yDim, m_zDim), 1);

    // Update affected region: update OpenGL texture to reflect data change.
    SbVec3i32 upMin = SbVec3s(m_xDim-(x+m_xOffset+(m_brush.getSize())[0]-1), y+m_yOffset, z+m_zOffset);
    SbVec3i32 upMax = SbVec3s(m_xDim-(x+m_xOffset), y+m_yOffset+(m_brush.getSize())[1]-1, z+m_zOffset+(m_brush.getSize())[2]-1);
    SbBox3i32 updateRegion = SbBox3i32(upMin, upMax);
    m_dataNode->updateRegions(&updateRegion, 1);
  }

#ifdef USE_MPEG
  if (m_recording) {
    // Render frame to MPEG
    m_MPEGRenderer->recordFrame();
  }
#endif /* USE_MPEG */
}


// Loads texture file from disk
// Replace by your code if you want to load other textures.
// The loaded texture must be an eight bits paletted 3D texture.
void loadTexture(void)
{
  unsigned char r, g, b;
  if ( m_colorMap == 0 )
    m_colorMap = new SoMFFloat;

  SbString textureFileName = m_textureFileName;
  SbString filepath = "$OIVHOME/examples/source/VolumeViz/Machining/Data/" + textureFileName;
  FILE * textureFile = SbFileHelper::open(filepath, "rb");
  if (!textureFile) {
    filepath = "../Machining/Data/" + textureFileName;
    textureFile = SbFileHelper::open(filepath, "rb");
    if (!textureFile) {
      SoDebugError::post("loadTexture", "The file was not loaded");
      exit(1);
    }
  }

  // Reads texture dimensions
  fread(&m_xDim, 4, 1, textureFile);
  fread(&m_yDim, 4, 1, textureFile);
  fread(&m_zDim, 4, 1, textureFile);

  // Little/Big endian adjustment
  unsigned char * bytePtr = (unsigned char *)&m_xDim;
  m_xDim = (bytePtr[3]<<24) | (bytePtr[2]<<16) | (bytePtr[1]<<8) | (bytePtr[0]);
  bytePtr = (unsigned char *)&m_yDim;
  m_yDim = (bytePtr[3]<<24) | (bytePtr[2]<<16) | (bytePtr[1]<<8) | (bytePtr[0]);
  bytePtr = (unsigned char *)&m_zDim;
  m_zDim = (bytePtr[3]<<24) | (bytePtr[2]<<16) | (bytePtr[1]<<8) | (bytePtr[0]);

  //notify volume data we are about to delete the pointer in memory
  //(NO_COPY mode we are using the pointer passed, so we have to make sure it is
  //not in use before the application deletes it)
  m_dataNode->data.setValue( SbVec3i32(m_xDim,m_yDim,m_zDim), m_dataNode->getDataType(), 0, (void *)NULL, SoSFArray::NO_COPY );
  if (m_data)
    delete []  m_data;
  m_data = new unsigned char[m_xDim*m_yDim*m_zDim];

  // Reads palette
  // IMPORTANT: you are free to use the first 128 colors of the palette as you wish
  // but the 128 others must be set as follows:
  // color[x+128] = color[x] with alpha component set to 0
  for (int i = 0; i < 256; i++) {
    fread(&r, 1, 1, textureFile);
    fread(&g, 1, 1, textureFile);
    fread(&b, 1, 1, textureFile);
    if (i<128) {
      m_colorMap->set1Value(i<<2, (float)r/255);
      m_colorMap->set1Value((i<<2)+1, (float)g/255);
      m_colorMap->set1Value((i<<2)+2, (float)b/255);
      m_colorMap->set1Value((i<<2)+3, 1.0f);
      m_colorMap->set1Value(((i+128)<<2), (float)r/255);
      m_colorMap->set1Value(((i+128)<<2)+1, (float)g/255);
      m_colorMap->set1Value(((i+128)<<2)+2, (float)b/255);
      m_colorMap->set1Value(((i+128)<<2)+3, 0.0f);
    }
  }

  // Reads actual texture
  fread(m_data, 1, m_xDim*m_yDim*m_zDim, textureFile);

  fclose(textureFile);
}


// Loads the shape to be machined on the volume.
// This is any inventor file containing lineSets here.
// m_lineSets must contain a pathList of lineSet nodes to be drawn.
void loadShape(void)
{
  if (m_workPath) {
    m_workPath->unref();
    m_workPath = NULL;
  }

  SoInput * myInput = new SoInput();
  SbBool ioError = FALSE;

  SbString shapeFileName = m_shapeFileName;
  SbString filepath = "$OIVHOME/examples/data/VolumeViz/" + shapeFileName;
  FILE * shapeFile = SbFileHelper::open(filepath, "rb");
  if (!shapeFile) {
    filepath = "$OIVHOME/examples/source/VolumeViz/Machining/Data/" + shapeFileName;
    shapeFile = SbFileHelper::open(filepath, "rb");
    if (!shapeFile) {
      SoDebugError::post("loadShape", "The file was not loaded");
      exit(1);
    }
  }
  fclose(shapeFile);

  myInput->openFile(filepath, ioError);
  if (ioError)
    return;

  m_workPath = SoDB::readAll(myInput);
  myInput->closeFile();
  delete myInput;
  if (!m_workPath)
    return;
  m_workPath->ref();

  // Calculates min & max values of the shape for scaling purpose
  SoSearchAction lineSetsearch;
  SbBox3f boundingBox;

  lineSetsearch.setType(SoLineSet::getClassTypeId());
  lineSetsearch.setInterest(SoSearchAction::ALL);
  lineSetsearch.apply(m_workPath);
  *m_lineSets = lineSetsearch.getPaths();
  if (m_lineSets->getLength()) {
    SoLineSet * curSet = (SoLineSet *)(*m_lineSets)[0]->getTail();
    SoVertexProperty * vertexProperty = (SoVertexProperty *)(curSet->vertexProperty).getValue();
    SoMFVec3f * points = &vertexProperty->vertex;
    if (points->getNum())
      boundingBox.setBounds((*points)[0], (*points)[0]);
    else
      boundingBox.setBounds(0,0,0,0,0,0);
  }
  for (int i = 0; i < m_lineSets->getLength(); i++) {
    SoLineSet * curSet = (SoLineSet *)(*m_lineSets)[i]->getTail();
    SoVertexProperty * vertexProperty = (SoVertexProperty *)(curSet->vertexProperty).getValue();
    // Do not directly affect vertexProperty->vertex since it will delete points (due to unreferencing)
    SoMFVec3f * points = &vertexProperty->vertex;

    for (int j = 0; j < points->getNum(); j++)
      for (int k = 0; k < 3; k++) {
        if ((*points)[j][k] > (boundingBox.getMax())[k])
          (boundingBox.getMax())[k] = (*points)[j][k];
        if ((*points)[j][k] < (boundingBox.getMin())[k])
          (boundingBox.getMin())[k] = (*points)[j][k];
      }
  }

  m_shapeSize = boundingBox.getMax() - boundingBox.getMin();

  m_start[0] = (short)(boundingBox.getMin())[0];
  m_start[1] = (short)(boundingBox.getMin())[1];
  m_start[2] = (short)(boundingBox.getMin())[2];
}


// Start/Restart drawing the shape
void restartDrawing(void)
{
  m_timer->unschedule();

  // Reset static vars to their initial values
  m_curChild = -1;
  m_curPoint = -1;
  m_counter = -1;
  m_maxCounter = 0;
  m_numPoints = 0;

  // Load shape
  loadShape();
  // Load unmachined texture
  loadTexture();
  m_dataNode->data.setValue( SbVec3i32(m_xDim,m_yDim,m_zDim), m_dataNode->getDataType(), 0, m_data, SoSFArray::NO_COPY );
  m_transferFunction->colorMap = (*m_colorMap);
  m_transferFunction->minValue = 0; m_transferFunction->maxValue = 255;

  float maxDim = (float)m_xDim;
  if (maxDim < m_yDim) maxDim = (float)m_yDim;
  if (maxDim < m_zDim) maxDim = (float)m_zDim;
  SbVec3f aspectRatio(m_xDim/maxDim, m_yDim/maxDim, 2*m_zDim/maxDim);
  m_dataNode->extent.setValue( SbBox3f(-aspectRatio, aspectRatio) );

  m_dataNode->ldmResourceParameters.getValue()->forceFixedResolution(true);
  // Forces creation of paging instance.
  m_dataNode->extent.getValue();

  setBrush(SbVec3s(10,10,14), m_brushType);
}


int
main(int, char** /*argv*/)
{
  SbString WorkPath3File = "$OIVHOME/examples/source/VolumeViz/Machining/Data/workPath3.iv";
  FILE *f = SbFileHelper::open(WorkPath3File,"r");
  if (f == NULL) {
    char message[256];
    sprintf(message, "Data file not found or not installed. \nYou may need to download the optional VolumeViz data files. \n");
#ifdef _WIN32
    MessageBox(NULL, message, "Machining",MB_ICONSTOP | MB_OK | MB_TASKMODAL );
#else
    printf("%s\n", message) ;
#endif
    exit(1);
  }
  fclose(f);
  // This file is *not* part of the customer demo distribution
#ifdef DEMO_UNLOCK
#  include "../../../../DoNotDistribute/VolRenDemoUnlock.h"
#endif

  Widget myWindow = SoXt::init("Machining");
  if (myWindow == NULL)
    return 1;

  SoVolumeRendering::init();
  SoDialogViz::init();

  m_lineSets = new SoPathList;

  SoSeparator * myRoot = new SoSeparator();
  myRoot->ref();

  // Volume rendering code begins here
  m_transferFunction = new SoTransferFunction();

  // Data node
  m_dataNode = new SoVolumeData();
  m_dataNode->useCompressedTexture = TRUE;
  m_dataNode->usePalettedTexture   = TRUE;
  myRoot->addChild(m_dataNode);

  // Timer to call the machining method
  m_timer = new SoTimerSensor(&shapeCB, NULL);
  m_timer->setBaseTime(SbTime(0.0));
  m_timer->setInterval(SbTime(0.02));

  // Material characteristics
	SoMaterial * myMaterial = new SoMaterial();
	myMaterial->ambientColor.setValue(1,1,1);
	myMaterial->diffuseColor.setValue(1,1,1);
	myMaterial->transparency.setValue(0);
	myMaterial->shininess.setValue(0);
	myRoot->addChild(myMaterial);

  // Transfer function
  m_transferFunction->colorMapType = SoTransferFunction::RGBA;
  myRoot->addChild(m_transferFunction);

  m_renderQuality = new SoVolumeRenderingQuality();
  m_renderQuality->deferredLighting = TRUE;
  myRoot->addChild(m_renderQuality);

  // Render
  m_render = new SoVolumeRender();
  m_render->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
  myRoot->addChild(m_render);

  m_textureFileName = "Wood.t3d";
  m_shapeFileName = "workPath3.iv";

  // Loads data and initializes the machining
  restartDrawing();

  // End Volume Rendering Code

  // Set up viewer
  Widget parent = buildInterface(myWindow);
  SoXtExaminerViewer *myExaminerViewer = new SoXtExaminerViewer( parent );

  // Usual Inventor stuff
  myExaminerViewer->setTitle( "Volume Rendering - Machining Demo" );
  myExaminerViewer->setSceneGraph( myRoot );
  myExaminerViewer->viewAll();
  SoCamera * myCam = myExaminerViewer->getCamera();
  // Positions the camera to see the machining
  SbVec3f camPos = myCam->position.getValue();
  camPos[2] = -camPos[2];
  myCam->position.setValue(camPos);
  myCam->pointAt(SbVec3f(0,0,0));
  myExaminerViewer->saveHomePosition();
  myExaminerViewer->show();

#ifdef USE_MPEG
  // Creates the MPEG renderer
  m_MPEGRenderer = new SoMPEGFrameRenderer ();
  m_MPEGRenderer->setSceneGraph(myExaminerViewer->getSceneManager()->getSceneGraph());
  m_MPEGRenderer->setBackgroundColor(SbColor (0, 0, 0));
  m_MPEGRenderer->setSize (SbVec2s (320,200));
  m_MPEGRenderer->setCompressionRate(0);
  m_MPEGRenderer->setGLRenderAction(myExaminerViewer->getGLRenderAction());
  m_MPEGFileName = "machining.mpg";
#endif /* USE_MPEG */

  SoXt::show(myWindow);

  SoXt::mainLoop();

  if (m_workPath)
    m_workPath->unref();

#ifdef USE_MPEG
  delete m_MPEGRenderer;
#endif /* USE_MPEG */ 

  delete m_lineSets;

  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return(0);
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

#include "MachiningAuditor.h"

SoTopLevelDialog *m_topLevelDialog;

Widget
buildInterface(Widget window)
{
  SoInput myInput;
  if (! myInput.openFile( "$OIVHOME/examples/source/VolumeViz/Machining/MachiningDialog.iv" ))
    return NULL;
  SoGroup *myGroup = SoDB::readAll( &myInput );
  if (! myGroup)
    return NULL;
  m_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  MachiningAuditor *myAuditor = new MachiningAuditor;
  m_topLevelDialog->addAuditor(myAuditor);

  SoDialogCustom *customNode = (SoDialogCustom *)m_topLevelDialog->searchForAuditorId(SbString("Viewer"));

  m_topLevelDialog->buildDialog( window, customNode != NULL );
  m_topLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}


