/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : G. SEGUIN (Mar 2001)
**=======================================================================*/
/////////////////////////////////////////////////////////////////////////////////
//
//Demo for realtime volume machining using VolumeRendering in Open Inventor
//
// Requires VolumeRendering library v3.0 or better
//
// Permission is granted for licensed customers of the 
// VolumeRendering library to use this source for any reasonable
// purpose that involves the VolumeRendering library.
//
// Files details:
//    - Machining.cxx/Machining.h: files containing VolumeRendering and Inventor machining code.
//    - SoVRBrush.cxx/SoVRBrush.h: utility class to help machining volumes.
//    - Controls.cxx/Controls.h: 3DMasterSuite dialog box.
//
/////////////////////////////////////////////////////////////////////////////////
#include "SoVRBrush.h"
#include <math.h>

// Paints the brush into data at position. dataDim contains the dimensions of the data. bytesPerVoxel should be self-explanatory.
void
SoVRBrush::paint(SbVec3s position, unsigned char * data, SbVec3s dataDim, unsigned int bytesPerVoxel)
{
  if (!m_matrix) return;

  SbVec3s drawSize = m_size;
  if ((position[0]+m_size[0]) >= dataDim[0])
    drawSize[0] = dataDim[0] - position[0];
  if ((position[1]+m_size[1]) >= dataDim[1])
    drawSize[1] = dataDim[1] - position[1];
  if ((position[2]+m_size[2]) >= dataDim[2])
    drawSize[2] = dataDim[2] - position[2];

  unsigned int lineSize =  dataDim[0]*bytesPerVoxel;
  unsigned int sliceSize = dataDim[1]*lineSize;
  int zMax = ((position[2]+m_size[2]) >= dataDim[2]);

  int offset = ((position[2]*dataDim[1] + position[1])*dataDim[0] + position[0])*bytesPerVoxel;
  unsigned char * brushCursor = m_matrix;
  if (bytesPerVoxel==1) {
    unsigned char * dataCursor = data + offset;
    for (int z = 0; z < drawSize[2]; z++, dataCursor += ((dataDim[1]-drawSize[1])*dataDim[0]), brushCursor += ((m_size[1]-drawSize[1])*m_size[0]))
      for (int y = 0; y < drawSize[1]; y++, dataCursor += (dataDim[0]-drawSize[0]), brushCursor += (m_size[0]-drawSize[0]))
        for (int x = 0; x < drawSize[0]; x++, dataCursor++, brushCursor++)
          if ((x>=-position[0]) && (y>=-position[1]) && (z>=-position[2]) && (*dataCursor < 128)) {
            switch (*brushCursor) {
            case Simple:
              *dataCursor += 128;
              break;
            case Empty:
              break;
            default:
              if ((!zMax) && (*brushCursor & ZM)) {
                *dataCursor = *(dataCursor+sliceSize);
                if ((*dataCursor) < 128)
                   *dataCursor += 128;
                break;
              }

              *dataCursor+=128;
              break;
            }
          }
  }
  else {
    unsigned short * dataCursor = ((unsigned short *)data) + offset;
    for (int z = 0; z < drawSize[2]; z++, dataCursor += ((dataDim[1]-drawSize[1])*dataDim[0]), brushCursor += ((m_size[1]-drawSize[1])*m_size[0]))
      for (int y = 0; y < drawSize[1]; y++, dataCursor += (dataDim[0]-drawSize[0]), brushCursor += (m_size[0]-drawSize[0]))
        for (int x = 0; x < drawSize[0]; x++, dataCursor++, brushCursor++)
          if ((x>=-position[0]) && (y>=-position[1]) && (z>=-position[2]) && (*dataCursor < 128)) {
            switch (*brushCursor) {
            case Simple:
              *dataCursor += 128;
              break;
            case Empty:
              break;
            default:
              if ((!zMax) && (*brushCursor & ZM)) {
                *dataCursor = *(dataCursor+sliceSize);
                if ((*dataCursor) < 128)
                   *dataCursor += 128;
                break;
              }

              *dataCursor+=128;
              break;
            }
          }
  }
}


// Sets the brush matrix. Can either be a predefined matrix or a user-supplied one.
// Be carefull: either way the matrix will be deleted upon brush change/instance deletion.
void SoVRBrush::setMatrix(SbVec3s size, PredefinedMatrix predefMatrix, unsigned char * matrix)
{
  if (m_matrix) {
    delete [] m_matrix;
    m_matrix = NULL;
  }

  m_size = size;
  if (!matrix) {
    switch(predefMatrix) {
    case CONE:
      createConeMatrix();
      break;
    case SQUARE:
      createSquareMatrix();
      break;
    case SPHERE:
      createSphereMatrix();
      break;
    case CYLINDER:
      createCylinderMatrix();
      break;
    }
  }
  else
    m_matrix = matrix;
}


// Predefined matrices creation methods
void
SoVRBrush::createSquareMatrix(void)
{
  m_matrix = new unsigned char[m_size[0]*m_size[1]*m_size[2]];
  if (!m_matrix) return;

  for (int k = 0; k < m_size[2]; k++)
    for (int j = 0; j < m_size[1]; j++)
      for (int i = 0; i < m_size[0]; i++) {
        unsigned int index = (k*m_size[1]+j)*m_size[0]+i;
        m_matrix[index] = Simple;
        if (!i) m_matrix[index] |= Xm;
        if (!j) m_matrix[index] |= Ym;
        if (!k) m_matrix[index] |= Zm;
        if (i==(m_size[0]-1)) m_matrix[index] |= XM;
        if (j==(m_size[1]-1)) m_matrix[index] |= YM;
        if (k==(m_size[2]-1)) m_matrix[index] |= ZM;
      }
}

void
SoVRBrush::createConeMatrix(void)
{
  m_matrix = new unsigned char[m_size[0]*m_size[1]*m_size[2]];
  if (!m_matrix) return;
  for (int l = 0; l < (m_size[0]*m_size[1]*m_size[2]); l++)
    m_matrix[l] = Empty;

  int maxXYRadius = m_size[0]*m_size[1]/4;
  int maxZRadius = m_size[2]*m_size[2];

  for (int k = 0; k < m_size[2]; k++) {
    int radius = ((m_size[2]-k)*(m_size[2]-k)*maxXYRadius)/maxZRadius;
    for (int j = 0; j < m_size[1]; j++) {
      int xBorder = radius-(j-m_size[1]/2)*(j-m_size[1]/2);
      int outside = xBorder<0;
      if (!outside)
        xBorder = (int)sqrt((double)xBorder);
      for (int i = 0; i < m_size[0]; i++) {
        unsigned int index = (k*m_size[1]+j)*m_size[0]+i;
        if (outside)
          m_matrix[index] = Empty;
        else if ((i<(m_size[0]/2-xBorder)) || (i>m_size[0]/2+xBorder))
            m_matrix[index] = Empty;
        else if ((i==(m_size[0]/2-xBorder)) || (i==m_size[0]/2+xBorder))
          m_matrix[index] = Simple;
        else
          m_matrix[index] = Simple;
      }
    }
  }
}

void
SoVRBrush::createSphereMatrix(void)
{
  m_matrix = new unsigned char[m_size[0]*m_size[1]*m_size[2]];
  if (!m_matrix) return;
  for (int l = 0; l < (m_size[0]*m_size[1]*m_size[2]); l++)
    m_matrix[l] = Empty;

  int maxXYRadius = m_size[0]*m_size[1]/4;
  int maxYZRadius = m_size[1]*m_size[2]/4;

  for (int k = 0; k < m_size[2]; k++) {
    for (int j = 0; j < m_size[1]; j++) {
      int rj = (j>=(m_size[1]/2))?(m_size[1]-j):j;
      int rk = (k>=(m_size[2]/2))?(m_size[2]-k):k;
      int radius = ((rj*rk)*maxXYRadius)/maxYZRadius;

      int xBorder = radius-(j-m_size[1]/2)*(j-m_size[1]/2);
      int outside = xBorder<0;
      if (!outside)
        xBorder = (int)sqrt((double)xBorder);
      for (int i = 0; i < m_size[0]; i++) {
        unsigned int index = (k*m_size[1]+j)*m_size[0]+i;
        if (outside)
          m_matrix[index] = Empty;
        else if ((i<(m_size[0]/2-xBorder)) || (i>m_size[0]/2+xBorder))
            m_matrix[index] = Empty;
        else if ((i==(m_size[0]/2-xBorder)) || (i==m_size[0]/2+xBorder))
          m_matrix[index] = Simple;
        else
          m_matrix[index] = Simple;
      }
    }
  }
}

void
SoVRBrush::createCylinderMatrix(void)
{
  m_matrix = new unsigned char[m_size[0]*m_size[1]*m_size[2]];
  if (!m_matrix) return;
  for (int l = 0; l < (m_size[0]*m_size[1]*m_size[2]); l++)
    m_matrix[l] = Empty;

  int maxXYRadius = m_size[0]*m_size[1]/4;

  for (int k = 0; k < m_size[2]; k++) {
    for (int j = 0; j < m_size[1]; j++) {
      int xBorder = maxXYRadius-(j-m_size[1]/2)*(j-m_size[1]/2);
      int outside = xBorder<0;
      if (!outside)
        xBorder = (int)sqrt((double)xBorder);
      for (int i = 0; i < m_size[0]; i++) {
        unsigned int index = (k*m_size[1]+j)*m_size[0]+i;
        if (outside)
          m_matrix[index] = Empty;
        else if ((i<(m_size[0]/2-xBorder)) || (i>m_size[0]/2+xBorder))
            m_matrix[index] = Empty;
        else if ((i==(m_size[0]/2-xBorder)) || (i==m_size[0]/2+xBorder))
          m_matrix[index] = Empty;
        else
          m_matrix[index] = Simple;
      }
    }
  }
}


